/*
*   Class   ProbabilityPlot
*
*   USAGE:  Probability Plots
*
*   WRITTEN BY: Dr Michael Thomas Flanagan
*
*   DATE:   29-30 September 2008, 1-5 October 2008, 13-24 October 2009, 2 November 2010, 8 December 2010
*           7 December 2011, 4 January 2012, 30 January - 24 February 2015, 24 March 2014, 9 April 2015, 20 November 2015
*
*   DOCUMENTATION:
*   See Michael Thomas Flanagan's Java library on-line web page:
*   http://www.ee.ucl.ac.uk/~mflanaga/java/ProbabilityPlot.html
*   http://www.ee.ucl.ac.uk/~mflanaga/java/
*
*   Copyright (c) 2009 - 2015 Michael Thomas Flanagan 
*
*   PERMISSION TO COPY:
*
* Permission to use, copy and modify this software and its documentation for NON-COMMERCIAL purposes is granted, without fee,
* provided that an acknowledgement to the author, Dr Michael Thomas Flanagan at www.ee.ucl.ac.uk/~mflanaga, appears in all copies333
* and associated documentation or publications.
*
* Public listing of the source codes on the internet is not permitted.
*
* Redistribution of the source codes or of the flanagan.jar file is not permitted.
*
* Redistribution in binary form of all or parts of these classes is not permitted.
*
* Dr Michael Thomas Flanagan makes no representations about the suitability or fitness of the software for any or for a particular purpose.
*
* Dr Michael Thomas Flanagan shall not be liable for any damages suffered as a result of using, modifying or distributing this software
* or its derivatives.
*
***************************************************************************************/

package flanagan.analysis;

import java.util.*;
import java.math.*;

import flanagan.math.*;
import flanagan.plot.PlotGraph;
import flanagan.interpolation.CubicSpline;
import flanagan.io.PrintToScreen;
import flanagan.io.FileOutput;
import flanagan.math.TimeAndDate;

public class ProbabilityPlot{

        // INSTANCE VARIABLES
        private String title = "";                              // title used in output file and graph names
        
        private double[] array = null;                          // array of data
        private Stat arrayAsStat = null;                        // array of data as Stat
        private double[] sortedData = null;                     // data sorted into ascending order
        private int[] indices = null;                           // original indices of data after sorting

        private double[] weights = null;                        // weights 
        private Stat weightsAsStat = null;                      // array of weights as Stat
        private double[] sortedWeights = null;                  // weights sorted into ascending order of data points
        private boolean weighted = false;                       // = true if weighted regression to be performed
        private int weightingOption = 0;                        // weighting option 
                                                                // 0 no weighting
                                                                // 1 weights supplied by user
                                                                // 2 weight = data value
                                                                // 3 weight = square root of the data value
                                                                // 4 weight = reciprocal of the data value
                                                                // 5 weight = reciprocal of the square root of the data value
        
        private double mean = Double.NaN;                       // array mean
        private double standardDeviation = Double.NaN;          // array standard deviation
        private double minimum = Double.NaN;                    // array minimum
        private double maximum = Double.NaN;                    // array maximum
        private double range = Double.NaN;                      // array range
        private double halfWidth = Double.NaN;                  // rough estimate of peak width at half peak height
        private double peakPoint = Double.NaN;                  // rough estimate of peak position

        private int numberOfDataPoints = 0;                     // number of data points

        private double dataOffset = 0.0;                        // data offset if data shifted
        private boolean dataShifted = false;                    // = true if data offset

        private double[] initialEstimates = null;               // initial estimates used in last call to a probability plot method

        private int lastMethod = -1;                            // Last probability plot method called
                                                                // -1   No priobability plot method called
                                                                //  0   Gaussian
                                                                //  1   Weibull (three parameter)
                                                                //  2   Exponential
                                                                //  3   Rayleigh
                                                                //  4   Pareto
                                                                //  5   Gumbel (minimum order statistic)
                                                                //  6   Gumbel (maximum order statistic)
                                                                //  7   Frechet
                                                                //  8   Logistic
                                                                //  9   Lorentzian  // TO BE ADDED
                                                                //  10  Log-Normal (three parameter) // TO BE ADDED
                                                                //  11  Log-Normal (two parameter // TO BE ADDED
                                                                //  12  Weibull (two parameter)
                                                                //  13  Weibull (standard)
                                                                //  14  Standard Gaussian
                                                                //  15  F-distribution
                                                                //  16  Two Parameter Frechet
                                                                //  17  Standard Frechet
        private int nMethods = 16;                              // Number of methods
        private String[] methodNames = {"Gaussian", "Three Parameter Weibull", "Exponential", "Rayleigh", "Pareto", "Gumbel (minimum order statistic)", "Gumbel (maximum order statistic)", "Three Parameter Frechet", "Logistic", "Lorentzian", "Three Parameter Log-Normal", "Two Parameter Log-Normal", "Two Parameter Weibull", "One Parameter (Standard) Weibull", "Standard Gaussian", "F-distribution", "Two Parameter Frechet", "One Parameter (Standard) Frechet"};
        private String[][] methodParameters = {{"mu", "sigma"}, {"mu", "sigma", "gamma"}, {"mu", "sigma"}, {"beta"}, {"alpha", "beta"}, {"mu", "sigma"}, {"mu", "sigma"}, {"mu", "sigma", "gamma"}, {"mu", "beta"}, {"mu", "gamma"}, {"alpha", "beta", "gamma"}, {"mu", "sigma"}, {"sigma", "gamma"}, {"gamma"}, {" "}, {" "}, {"sigma", "gamma"}, {"gamma"}}; 
        private int fDistributionNu1 = 0;                       // f-Distribution degrees of freedom, nu1
        private int fDistributionNu2 = 0;                       // f-Distribution degrees of freedom, nu2
        
        
        private boolean suppressPlot = false;                   // set to true to suppress the display of the plot 
        private boolean suppressFile = false;                   // set to true to suppress the output to text file 
        private boolean suppressErrorMessages = false;          // set to true to suppress the output of Regression convergence and chiSquare error messages    
        
        private double significance = 0.05;                     // significance used in obtaining a critical value for the correlation coefficient
        
        private int gaussianNumberOfParameters = 2;             // number of Gaussian parameters
        private double[] gaussianOrderMedians = null;           // Gaussian order statistic medians
        private double[] gaussianParam = null;                  // Gaussian parameters obtained by the minimization procedure
        private double[] gaussianParamErrors = null;            // estimates of the errors of the Gaussian parameters obtained by the minimization procedure
        private double gaussianSumOfSquares = Double.NaN;       // sum of squares at Gaussian minimum
        private double gaussianWeightedSumOfSquares = Double.NaN;  // weighted sum of squares at Gaussian minimum
        private double[] gaussianLine = null;                   // Gaussian probability plot gradient and intercept
        private double[] gaussianLineErrors = null;             // estimated errors of the Gaussian probability plot gradient and intercept
        private double gaussianCorrCoeff = Double.NaN;          // Gaussian correlation coefficient of the probability plot
        private boolean gaussianDone = false;                   // = true after Gaussian probability plot drawn
        

        private int gaussianStandardNumberOfParameters = 0;     // number of Standard Gauss parameters
        private double[] gaussianStandardOrderMedians = null;   // Standard Gauss order statistic medians
        private double gaussianStandardSumOfSquares = Double.NaN;   // sum of squares at Standard Gauss minimum
        private double gaussianStandardWeightedSumOfSquares = Double.NaN;   // sum of weighted squares at Standard Gauss minimum

        private double[] gaussianStandardLine = null;           // Standard Gauss probability plot gradient and intercept
        private double[] gaussianStandardLineErrors = null;     // estimated errors of the Standard Gauss probability plot gradient and intercept
        private double gaussianStandardCorrCoeff = Double.NaN;  // Standard Gauss correlation coefficient of the probability plot
        private boolean gaussianStandardDone = false;           // = true after Standard Gauss probability plot drawn

        private int exponentialNumberOfParameters = 2;          // number of Exponential parameters
        private double[] exponentialOrderMedians = null;        // Exponential order statistic medians
        private double[] exponentialParam = null;               // Exponential parameters obtained by the minimization procedure
        private double[] exponentialParamErrors = null;         // estimates of the errors of the Exponential parameters obtained by the minimization procedure
        private double exponentialSumOfSquares = Double.NaN;    // sum of squares at Exponential minimum
        private double exponentialWeightedSumOfSquares = Double.NaN;    // weighted sum of squares at Exponential minimum
        private double[] exponentialLine = null;                // Exponential probability plot gradient and intercept
        private double[] exponentialLineErrors = null;          // estimated errors of the Exponential probability plot gradient and intercept
        private double exponentialCorrCoeff = Double.NaN;       // Exponential correlation coefficient of the probability plot
        private boolean exponentialDone = false;                // = true after Exponential probability plot drawn

        private int fDistributionNumberOfParameters = 0;        // number of F-distribution parameters
        private double[] fDistributionOrderMedians = null;      // F-distribution order statistic medians
        private double fDistributionSumOfSquares = Double.NaN;  // sum of squares at F-distribution minimum
        private double fDistributionWeightedSumOfSquares = Double.NaN;  // weighted sum of squares at F-distribution minimum
        private double[] fDistributionLine = null;              // F-distribution probability plot gradient and intercept
        private double[] fDistributionLineErrors = null;        // estimated errors of the F-distribution probability plot gradient and intercept
        private double fDistributionCorrCoeff = Double.NaN;     // F-distribution correlation coefficient of the probability plot
        private boolean fDistributionDone = false;              // = true after F-distribution probability plot drawn

        private int frechetNumberOfParameters = 3;              // number of Frechet parameters
        private double[] frechetOrderMedians = null;            // Frechet order statistic medians
        private double[] frechetParam = null;                   // Frechet parameters obtained by the minimization procedure
        private double[] frechetParamErrors = null;             // estimates of the errors of the Frechet parameters obtained by the minimization procedure
        private double frechetSumOfSquares = Double.NaN;        // sum of squares at Frechet minimum
        private double frechetWeightedSumOfSquares = Double.NaN;// weighted sum of squares at Frechet minimum
        private double[] frechetLine = null;                    // Frechet probability plot gradient and intercept
        private double[] frechetLineErrors = null;              // estimated errors of the Frechet probability plot gradient and intercept
        private double frechetCorrCoeff = Double.NaN;           // Frechet correlation coefficient of the probability plot
        private boolean frechetDone = false;                    // = true after Frechet probability plot drawn
        
        private int frechetTwoParNumberOfParameters = 2;        // number of Two Parameter Frechet parameters
        private double[] frechetTwoParOrderMedians = null;      // Two Parameter Frechet order statistic medians
        private double[] frechetTwoParParam = null;             // Two ParameterFrechet parameters obtained by the minimization procedure
        private double[] frechetTwoParParamErrors = null;       // estimates of the errors of the Two Parameter Frechet parameters obtained by the minimization procedure
        private double frechetTwoParSumOfSquares = Double.NaN;  // sum of squares at Two Parameter Frechet minimum
        private double frechetTwoParWeightedSumOfSquares = Double.NaN;  // weighted sum of squares at Two Parameter Frechet minimum
        private double[] frechetTwoParLine = null;              // Two Parameter Frechet probability plot gradient and intercept
        private double[] frechetTwoParLineErrors = null;        // estimated errors of the Two Parameter Frechet probability plot gradient and intercept
        private double frechetTwoParCorrCoeff = Double.NaN;     // Two ParameterFrechet correlation coefficient of the probability plot
        private boolean frechetTwoParDone = false;              // = true after Two Parameter Frechet probability plot drawn
        
        private int frechetStandardNumberOfParameters = 1;      // number of Standard Frechet parameters
        private double[] frechetStandardOrderMedians = null;    // Standard Frechet order statistic medians
        private double[] frechetStandardParam = null;           // StandardFrechet parameters obtained by the minimization procedure
        private double[] frechetStandardParamErrors = null;     // estimates of the errors of the Standard Frechet parameters obtained by the minimization procedure
        private double frechetStandardSumOfSquares = Double.NaN;// sum of squares at Standard Frechet minimum
        private double frechetStandardWeightedSumOfSquares = Double.NaN;    // weighted sum of squares at Standard Frechet minimum
        private double[] frechetStandardLine = null;            // Standard Frechet probability plot gradient and intercept
        private double[] frechetStandardLineErrors = null;      // estimated errors of the Standard Frechet probability plot gradient and intercept
        private double frechetStandardCorrCoeff = Double.NaN;   // StandardFrechet correlation coefficient of the probability plot
        private boolean frechetStandardDone = false;            // = true after Standard Frechet probability plot drawn

        private int gumbelMinNumberOfParameters = 3;            // number of Gumbel (minimum order statistic) parameters
        private double[] gumbelMinOrderMedians = null;          // Gumbel (minimum order statistic) order statistic medians
        private double[] gumbelMinParam = null;                 // Gumbel (minimum order statistic) parameters obtained by the minimization procedure
        private double[] gumbelMinParamErrors = null;           // estimates of the errors of the Gumbel (minimum order statistic) parameters obtained by the minimization procedure
        private double gumbelMinSumOfSquares = Double.NaN;      // sum of squares at Gumbel (minimum order statistic) minimum
        private double gumbelMinWeightedSumOfSquares = Double.NaN;      // weighted sum of squares at Gumbel (minimum order statistic) minimum
        private double[] gumbelMinLine = null;                  // Gumbel (minimum order statistic) probability plot gradient and intercept
        private double[] gumbelMinLineErrors = null;            // estimated errors of the Gumbel (minimum order statistic) probability plot gradient and intercept
        private double gumbelMinCorrCoeff = Double.NaN;         // Gumbel (minimum order statistic) correlation coefficient of the probability plot
        private boolean gumbelMinDone = false;                  // = true after Gumbel (minimum order statistic) probability plot drawn

        private int gumbelMaxNumberOfParameters = 3;            // number of Gumbel (maximum order statistic) parameters
        private double[] gumbelMaxOrderMedians = null;          // Gumbel (maximum order statistic) order statistic medians
        private double[] gumbelMaxParam = null;                 // Gumbel (maximum order statistic) parameters obtained by the maximization procedure
        private double[] gumbelMaxParamErrors = null;           // estimates of the errors of the Gumbel (maximum order statistic) parameters obtained by the maximization procedure
        private double gumbelMaxSumOfSquares = Double.NaN;      // sum of squares at Gumbel (maximum order statistic) maximum
        private double gumbelMaxWeightedSumOfSquares = Double.NaN;      // weighted sum of squares at Gumbel (maximum order statistic) maximum
        private double[] gumbelMaxLine = null;                  // Gumbel (maximum order statistic) probability plot gradient and intercept
        private double[] gumbelMaxLineErrors = null;            // estimated errors of the Gumbel (maximum order statistic) probability plot gradient and intercept
        private double gumbelMaxCorrCoeff = Double.NaN;         // Gumbel (maximum order statistic) correlation coefficient of the probability plot
        private boolean gumbelMaxDone = false;                  // = true after Gumbel (maximum order statistic) probability plot drawn

        private int logisticNumberOfParameters = 3;             // number of Logistic parameters
        private double[] logisticOrderMedians = null;           // Logistic order statistic medians
        private double[] logisticParam = null;                  // Logistic parameters obtained by the minimization procedure
        private double[] logisticParamErrors = null;            // estimates of the errors of the Logistic parameters obtained by the minimization procedure
        private double logisticSumOfSquares = Double.NaN;       // sum of squares at Logistic minimum
        private double logisticWeightedSumOfSquares = Double.NaN;      // weighted sum of squares at Logistic minimum
        private double[] logisticLine = null;                   // Logistic probability plot gradient and intercept
        private double[] logisticLineErrors = null;             // estimated errors of the Logistic probability plot gradient and intercept
        private double logisticCorrCoeff = Double.NaN;          // Logistic correlation coefficient of the probability plot
        private boolean logisticDone = false;                   // = true after Logistic probability plot drawn

        private int paretoNumberOfParameters = 2;               // number of Pareto parameters
        private double[] paretoOrderMedians = null;             // Pareto order statistic medians
        private double[] paretoParam = null;                    // Pareto parameters obtained by the minimization procedure
        private double[] paretoParamErrors = null;              // estimates of the errors of the Pareto parameters obtained by the minimization procedure
        private double paretoSumOfSquares = Double.NaN;         // sum of squares at Pareto minimum
        private double paretoWeightedSumOfSquares = Double.NaN; // weighted sum of squares at Pareto minimum
        private double[] paretoLine = null;                     // Pareto probability plot gradient and intercept
        private double[] paretoLineErrors = null;               // estimated errors of the Pareto probability plot gradient and intercept
        private double paretoCorrCoeff = Double.NaN;            // Pareto correlation coefficient of the probability plot
        private boolean paretoDone = false;                     // = true after Pareto probability plot drawn

        private int rayleighNumberOfParameters = 2;             // number of Rayleigh parameters
        private double[] rayleighOrderMedians = null;           // Rayleigh order statistic medians
        private double[] rayleighParam = null;                  // Rayleigh parameters obtained by the minimization procedure
        private double[] rayleighParamErrors = null;            // estimates of the errors of the Rayleigh parameters obtained by the minimization procedure
        private double rayleighSumOfSquares = Double.NaN;       // sum of squares at Rayleigh minimum
        private double rayleighWeightedSumOfSquares = Double.NaN;      // weighted sum of squares at Rayleigh minimum
        private double[] rayleighLine = null;                   // Rayleigh probability plot gradient and intercept
        private double[] rayleighLineErrors = null;             // estimated errors of the Rayleigh probability plot gradient and intercept
        private double rayleighCorrCoeff = Double.NaN;          // Rayleigh correlation coefficient of the probability plot
        private boolean rayleighDone = false;                   // = true after Rayleigh probability plot drawn

        private int weibullNumberOfParameters = 3;              // number of Three Parameter Weibull parameters
        private double[] weibullOrderMedians = null;            // Three Parameter Weibull order statistic medians
        private double[] weibullParam = null;                   // Three Parameter Weibull parameters obtained by the minimization procedure
        private double[] weibullParamErrors = null;             // estimates of the errors of the Three Parameter Weibull parameters obtained by the minimization procedure
        private double weibullSumOfSquares = Double.NaN;        // sum of squares at Three Parameter Weibull minimum
        private double weibullWeightedSumOfSquares = Double.NaN;// weighted sum of squares at Three Parameter Weibull minimum
        private double[] weibullLine = null;                    // Three Parameter Weibull probability plot gradient and intercept
        private double[] weibullLineErrors = null;              // estimated errors of the Three Parameter Weibull probability plot gradient and intercept
        private double weibullCorrCoeff = Double.NaN;           // Three Parameter Weibull correlation coefficient of the probability plot
        private boolean weibullDone = false;                    // = true after Three Parameter Weibull probability plot drawn

        private int weibullTwoParNumberOfParameters = 2;        // number of Two Parameter Weibull parameters
        private double[] weibullTwoParOrderMedians = null;      // Two Parameter Weibull order statistic medians
        private double[] weibullTwoParParam = null;             // Two Parameter Weibull parameters obtained by the minimization procedure
        private double[] weibullTwoParParamErrors = null;       // estimates of the errors of the Two Parameter Weibull parameters obtained by the minimization procedure
        private double weibullTwoParSumOfSquares = Double.NaN;  // sum of squares at Two Parameter Weibull minimum
        private double weibullTwoParWeightedSumOfSquares = Double.NaN;  // weighted sum of squares at Two Parameter Weibull minimum
        private double[] weibullTwoParLine = null;              // Two Parameter Weibull probability plot gradient and intercept
        private double[] weibullTwoParLineErrors = null;        // estimated errors of the Two Parameter Weibull probability plot gradient and intercept
        private double weibullTwoParCorrCoeff = Double.NaN;     // Two Parameter Weibull correlation coefficient of the probability plot
        private boolean weibullTwoParDone = false;              // = true after Two Parameter Weibull probability plot drawn

        private int weibullStandardNumberOfParameters = 1;      // number of Standard Weibull parameters
        private double[] weibullStandardOrderMedians = null;    // Standard Weibull order statistic medians
        private double[] weibullStandardParam = null;           // Standard Weibull parameters obtained by the minimization procedure
        private double[] weibullStandardParamErrors = null;     // estimates of the errors of the Standard Weibull parameters obtained by the minimization procedure
        private double weibullStandardSumOfSquares = Double.NaN;// sum of squares at Standard Weibull minimum
        private double weibullStandardWeightedSumOfSquares = Double.NaN;// weighted sum of squares at Standard Weibull minimum
        private double[] weibullStandardLine = null;            // Standard Weibull probability plot gradient and intercept
        private double[] weibullStandardLineErrors = null;      // estimated errors of the Standard Weibull probability plot gradient and intercept
        private double weibullStandardCorrCoeff = Double.NaN;   // Standard Weibull correlation coefficient of the probability plot
        private boolean weibullStandardDone = false;            // = true after Standard Weibull probability plot drawn
        private boolean probPlotDone = false;                   // = true after any probability plot drawn

        private double delta = 1e-2;                            // step fraction in numerical differentiation
        private double stepFactor = 0.2;                        // initial estimate multiplier in setting step size
        private double tolerance = 1E-15;                       // tolerance in exiting minimisation procedure

        private boolean nFactorOptionI = false;                 // = true  variance, covariance and standard deviation denominator = n                                                                                // = false varaiance, covariance and standard deviation denominator = n-1
        private boolean nFactorReset = false;                   // = true when instance method resetting the denominator is called
        
        private boolean[] userSuppliedEstimates = null;               // = true if user supplies initial estimates
                                                                // index = above method number
        private double[][] userInitialEstimates = null;         // user supplied initial estimates
                                                                // first index = above method number

        private TimeAndDate tad = new TimeAndDate();            // instance of TimeAndDate
        private int trunc = 4;                                  // truncation number for output file
        
        private String titleStart = "";                         // start of graph title - for use with Outlier class

        // CONSTRUCTORS
        public ProbabilityPlot(double[] xx){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weighted = false;
             this.weightingOption = 0;
             this.initialise();
        }
        
        public ProbabilityPlot(double[] xx, double[] ww){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weightsAsStat = new Stat(ww);
             this.weights = this.weightsAsStat.array();
             this.weighted = true;
             this.weightingOption = 1;
             this.initialise();
        }

        public ProbabilityPlot(Double[] xx){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weighted = false;
             this.weightingOption = 0;
             this.initialise();
        }
        
        public ProbabilityPlot(Double[] xx, Double[] ww){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weightsAsStat = new Stat(ww);
             this.weights = this.weightsAsStat.array();
             this.weighted = true;
             this.weightingOption = 1;
             this.initialise();
        }

        public ProbabilityPlot(float[] xx){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weighted = false;
             this.weightingOption = 0;
             this.initialise();
        }
        
        public ProbabilityPlot(float[] xx, float[] ww){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weightsAsStat = new Stat(ww);
             this.weights = this.weightsAsStat.array();
             this.weighted = true;
             this.weightingOption = 1;
             this.initialise();
        }

        public ProbabilityPlot(Float[] xx){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weighted = false;
             this.weightingOption = 0;
             this.initialise();
        }
        
        public ProbabilityPlot(Float[] xx, Float[] ww){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weightsAsStat = new Stat(ww);
             this.weights = this.weightsAsStat.array();
             this.weighted = true;
             this.weightingOption = 1;
             this.initialise();
        }

        public ProbabilityPlot(long[] xx){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weighted = false;
             this.weightingOption = 0;
             this.initialise();
        }
        
        public ProbabilityPlot(long[] xx, long[] ww){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weightsAsStat = new Stat(ww);
             this.weights = this.weightsAsStat.array();
             this.weighted = true;
             this.weightingOption = 1;
             this.initialise();
        }

        public ProbabilityPlot(Long[] xx){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weighted = false;
             this.weightingOption = 0;
             this.initialise();
        }

        public ProbabilityPlot(Long[] xx, Long[] ww){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weightsAsStat = new Stat(ww);
             this.weights = this.weightsAsStat.array();
             this.weighted = true;
             this.weightingOption = 1;
             this.initialise();
        }
        
        public ProbabilityPlot(int[] xx){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weighted = false;
             this.weightingOption = 0;
             this.initialise();
        }
        
        public ProbabilityPlot(int[] xx, int[] ww){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weightsAsStat = new Stat(ww);
             this.weights = this.weightsAsStat.array();
             this.weighted = true;
             this.weightingOption = 1;
             this.initialise();
        }

        public ProbabilityPlot(Integer[] xx){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weighted = false;
             this.weightingOption = 0;
             this.initialise();
        }
        
        public ProbabilityPlot(Integer[] xx, Integer[] ww){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weightsAsStat = new Stat(ww);
             this.weights = this.weightsAsStat.array();
             this.weighted = true;
             this.weightingOption = 1;
             this.initialise();
        }

        public ProbabilityPlot(short[] xx){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weighted = false;
             this.weightingOption = 0;
             this.initialise();
        }
        
        public ProbabilityPlot(short[] xx, short[] ww){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weightsAsStat = new Stat(ww);
             this.weights = this.weightsAsStat.array();
             this.weighted = true;
             this.weightingOption = 1;
             this.initialise();
        }

        public ProbabilityPlot(Short[] xx){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weightingOption = 0;
             this.initialise();
        }
        
        public ProbabilityPlot(Short[] xx, Short[] ww){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weightsAsStat = new Stat(ww);
             this.weights = this.weightsAsStat.array();
             this.weighted = true;
             this.weightingOption = 1;
             this.initialise();
        }

        public ProbabilityPlot(byte[] xx){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weightingOption = 0;
             this.weighted = false;
             this.initialise();
        }
        
        public ProbabilityPlot(byte[] xx, byte[] ww){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weightsAsStat = new Stat(ww);
             this.weights = this.weightsAsStat.array();
             this.weighted = true;
             this.weightingOption = 1;
             this.initialise();
        }

        public ProbabilityPlot(Byte[] xx){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weighted = false;
             this.weightingOption = 0;
             this.initialise();
        }

        public ProbabilityPlot(Byte[] xx, Byte[] ww){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weightsAsStat = new Stat(ww);
             this.weights = this.weightsAsStat.array();
             this.weighted = true;
             this.weightingOption = 1;
             this.initialise();
        }
        
        public ProbabilityPlot(BigDecimal[] xx){
             this.arrayAsStat = new Stat(xx);
             this.weighted = false;
             this.weightingOption = 0;
             this.initialise();
        }
        
        public ProbabilityPlot(BigDecimal[] xx, BigDecimal[] ww){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weightsAsStat = new Stat(ww);
             this.weights = this.weightsAsStat.array();
             this.weighted = true;
             this.weightingOption = 1;
             this.initialise();
        }

        public ProbabilityPlot(BigInteger[] xx){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weighted = false;
             this.weightingOption = 0;
             this.initialise();
        }
        
        public ProbabilityPlot(BigInteger[] xx, BigInteger[] ww){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weightsAsStat = new Stat(ww);
             this.weights = this.weightsAsStat.array();
             this.weighted = true;
             this.weightingOption = 1;
             this.initialise();
        }

        public ProbabilityPlot(Object[] xx){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weighted = false;
             this.weightingOption = 0;
             this.initialise();
        }
        
        public ProbabilityPlot(Object[] xx, Object[] ww){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weightsAsStat = new Stat(ww);
             this.weights = this.weightsAsStat.array();
             this.weighted = true;
             this.weightingOption = 1;
             this.initialise();
        }

        public ProbabilityPlot(Vector<Object> xx){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weighted = false;
             this.weightingOption = 0;
             this.initialise();
        }

        public ProbabilityPlot(Vector<Object> xx, Vector<Object> ww){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weightsAsStat = new Stat(ww);
             this.weights = this.weightsAsStat.array();
             this.weighted = true;
             this.weightingOption = 1;
             this.initialise();
        }
        
        public ProbabilityPlot(ArrayList<Object> xx){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weighted = false;
             this.weightingOption = 0;
             this.initialise();
        }
        
        public ProbabilityPlot(ArrayList<Object> xx, ArrayList<Object> ww){
             this.arrayAsStat = new Stat(xx);
             this.array = this.arrayAsStat.array();
             this.weightsAsStat = new Stat(ww);
             this.weights = this.weightsAsStat.array();
             this.weighted = true;
             this.weightingOption = 1;
             this.initialise();
        }

        public ProbabilityPlot(ArrayMaths xx){
             double[] aa = xx.array();
             this.arrayAsStat = new Stat(aa);
             this.array = this.arrayAsStat.array();
             this.weighted = false;
             this.weightingOption = 0;
             this.initialise();
        }
        
        public ProbabilityPlot(ArrayMaths xx, ArrayMaths ww){
             double[] aa = xx.array();
             this.arrayAsStat = new Stat(aa);
             this.array = this.arrayAsStat.array();
             double[] zz = ww.array();
             this.weightsAsStat = new Stat(zz);
             this.weights = this.weightsAsStat.array();
             this.weighted = true;
             this.weightingOption = 1;
             this.initialise();
             
        }

        public ProbabilityPlot(Stat xx){
             this.arrayAsStat = xx;
             this.array = this.arrayAsStat.array();
             this.weighted = false;
             this.weightingOption = 0;
             this.initialise();
        }
        
         public ProbabilityPlot(Stat xx, Stat ww){
             this.arrayAsStat = xx;
             this.array = this.arrayAsStat.array();
             this.weightsAsStat = ww;
             this.weights = this.weightsAsStat.array();
             this.weighted = true;
             this.weightingOption = 1;
             this.initialise();
        }


        // INITIALISATIONS
        private void initialise(){
            this.numberOfDataPoints = this.array.length;
            ArrayMaths sorted = arrayAsStat.sort();
            this.indices = sorted.originalIndices();
            this.sortedData = sorted.array();
            this.mean = arrayAsStat.mean();
            this.standardDeviation = arrayAsStat.standardDeviation();
            this.minimum = arrayAsStat.minimum();
            this.maximum = arrayAsStat.maximum();
            this.range = this.maximum - this.minimum;
            
            this.sortedWeights = new double[this.numberOfDataPoints];
            if(this.weighted){
                for(int i=0; i<this.numberOfDataPoints; i++){
                    this.sortedWeights[i] = this.weights[this.indices[i]];
                }
            }
            else{
                this.weights = new double[this.numberOfDataPoints];
                for(int i=0; i<this.numberOfDataPoints; i++){
                    this.weights[i] = 1.0;
                    this.sortedWeights[i] = 1.0;
                }
            }
            
            this.userSuppliedEstimates = new boolean[this.nMethods];
            for(int i=0; i<this.nMethods; i++)this.userSuppliedEstimates[i] = false;
            this.userSuppliedEstimates = new boolean[this.nMethods];
            this.userInitialEstimates = new double[this.nMethods][];
        }
        
        // RESET THE SIGNIFICANCE LEVEL
        public void resetSignificance(double signif){
            if(signif<0.0 || signif>1.0)throw new IllegalArgumentException("The argument signif, " + signif + ", must lie between 0 and 1 inclusive");
            this.significance = signif;
        }
        
        public double getSignificance(){
            return this.significance;
        }

        // SUPPRESS DISPLAY OF PLOT
        public void suppressDisplay(){
            this.suppressPlot = true;
        }
        
        public void restoreDisplay(){
            this.suppressPlot = false;
        }
        
        // SUPPRESS OUTPUT TO TEXT FILE
        public void suppressFileOutput(){
            this.suppressFile = true;
        }
        
        public void restoreFileOutput(){
            this.suppressFile = false;
        }
        
        // SUPPRESS REGRESSION CONVERGENCE AND CHISQUARE ERROR MESSAGES    
        public void suppressErrorMessages(){
            this.suppressErrorMessages = true;          
        }
        
        public void restoreErrorMessages(){
            this.suppressErrorMessages = false;          
        }
        
        
        
        // WEIGHTING OPTION
        
        // Set weighting option 
        public void setWeightingOption(int option){
            if(option<1 || option>4)throw new IllegalArgumentException("The option argument, " + option + ", should be 1, 2, 3 or 4");
            if(this.weighted && this.weightingOption==1)throw new IllegalArgumentException("Weights already entered via a constructor");
            
            switch(option){
                case 1: for(int i=0; i<this.numberOfDataPoints; i++)this.sortedWeights[i] = Math.abs(this.sortedData[i]);
                        break;
                case 2: for(int i=0; i<this.numberOfDataPoints; i++)this.sortedWeights[i] = Math.sqrt(Math.abs(this.sortedData[i]));
                        break;
                case 3: for(int i=0; i<this.numberOfDataPoints; i++)this.sortedWeights[i] = 1.0/(Math.abs(this.sortedData[i]));
                        break;
                case 4: for(int i=0; i<this.numberOfDataPoints; i++)this.sortedWeights[i] = 1.0/(Math.sqrt(Math.abs(this.sortedData[i])));
                        break;
            }
            this.weightingOption = option + 1;
        }

        // Get weighting option - String output
        public String getWeightingOption(){
            String ret = null;
            switch(this.weightingOption){
                case 0: ret = "Unweighted regression";
                        break;
                case 1: ret = "Weighted regression: user supplied weights";
                        break;
                case 2: ret = "Weighted regression: weight = response value";
                        break;
                case 3: ret = "Weighted regression: weight = square root of the response value";
                        break;
                case 4: ret = "Weighted regression: weight = reciprocal of the response value";
                        break;
                case 5: ret = "Weighted regression: weight = reciprocal of the square root of the response value";
                        break;
            }

            return ret;
        }

        // Set weighting option to unweighted regression
        public void removeWeights(){
            this.weighted = false;
            for(int i=0; i<this.numberOfDataPoints; i++)this.sortedWeights[i] = 1.0;
        }
        
        // Restore weighting option to previously entered weighted regression
        public void restoreWeights(){
            this.weighted =true;
            for(int i=0; i<this.numberOfDataPoints; i++)this.sortedWeights[i] = this.weights[this.indices[i]];
        }
        
        

        // DATA SHIFT
        // Tests if any negative or zero data points present and shifts data to positive domain
        // Called by probability plot methods requiring all positive, non-zero data
        private void negativeAndNonZeroDataShift(){
            this.dataShifted = false;
            if(this.minimum<=0){
                this.dataOffset = this.range*0.01 - this.minimum;
                this.dataShift();
            }
        }


        // Tests if any negative data points present and shifts data to positive domain
        // Called by probability plot methods requiring all positive, non-zero data
        private void negativeDataShift(){
            this.dataShifted = false;
            if(this.minimum<0.0){
                this.dataOffset = -this.minimum;
                this.dataShift();
            }
        }

        // Shifts data
        private void dataShift(){
            for(int i=0; i<this.numberOfDataPoints; i++){
                this.sortedData[i] += this.dataOffset;
            }
            this.minimum += this.dataOffset;
            this.maximum += this.dataOffset;
            this.mean += this.dataOffset;
            this.dataShifted = true;
        }

        // Returns data offset
        public double getdataOffset(){
            return this.dataOffset;
        }



        // rough estimate of the half-height peak width
        private double peakWidth(){

            this.halfWidth = 0.0;
            double[] interpData = null;
            int nInterp = 10000;

            // Interpolate to increase number of points to allow binning
            if(this.numberOfDataPoints>=1000){
                interpData = this.sortedData;
                nInterp = this.numberOfDataPoints;
            }
            else{
                double[] dataX = new double[this.numberOfDataPoints];
                for(int i=0; i<this.numberOfDataPoints; i++)dataX[i]=i;
                double incrI = ((double)(this.numberOfDataPoints-1))/(nInterp-1);

                interpData = new double[nInterp];
                CubicSpline cs = new CubicSpline(dataX, this.sortedData);
                double interp = 0.0;
                for(int i=0; i<nInterp-1; i++){
                    interpData[i]=cs.interpolate(interp);
                    interp += incrI;

                }
                interpData[nInterp-1] = (double)(this.numberOfDataPoints-1);
            }

            // Bin the data
            int nBins = 100;
            double[] binnedData = new double[nBins];
            double[] bins = new double[nBins];
            double binWidth = this.range/nBins;
            double binLower = this.minimum;
            double binUpper = binLower + binWidth;
            int counter = 0;
            for(int i=0; i<nBins; i++){
                bins[i] = (binUpper + binLower)/2.0;
                binnedData[i] = 0.0;
                boolean test = true;
                if(counter>=nInterp)test = false;
                while(test){
                    if(interpData[counter]<binUpper){
                        binnedData[i] += 1.0;
                    }
                    else{
                        test = false;
                    }
                    counter++;
                    if(counter>=nInterp)test = false;
                }
                binLower = binUpper;
                binUpper = binLower + binWidth;
            }
            if(counter<nInterp)binnedData[nBins-1] += (double)(nInterp-counter);

            // Identify peak
            ArrayMaths am = new ArrayMaths(binnedData);
            double maxI = am.maximum();
            int maxIindex = am.maximumIndex();
            this.peakPoint = bins[maxIindex];
            double halfHeight = maxI/2.0;
            double widthLower = 0.0;
            boolean lowerCheck = false;
            double widthUpper = 0.0;
            boolean upperCheck = false;

            // lower limit
            if(binnedData[0]==halfHeight){
                widthLower = bins[0];
                lowerCheck = true;
            }
            else{
                if(binnedData[0]<halfHeight){
                    if(maxIindex>=2){
                        double[] interpLy = new double[maxIindex+1];
                        double[] interpLx = new double[maxIindex+1];
                        for(int i=0; i<=maxIindex; i++){
                            interpLy[i] = binnedData[i];
                            interpLx[i] = bins[i];
                        }
                        CubicSpline csl = new CubicSpline(interpLx, interpLy);
                        double[] tempx = new double[100];
                        double[] tempy = new double[100];
                        double incr = (interpLx[maxIindex]-interpLx[0])/99;
                        double intr = interpLx[0];
                        for(int i=0; i<99; i++){
                            tempx[i] = intr;
                            tempy[i] = csl.interpolate(intr);
                            intr += incr;
                        }
                        tempy[99] = interpLy[maxIindex];
                        tempx[99] = interpLx[maxIindex];
                        boolean testt = true;
                        int ii = 0;
                        while(testt){
                            if(halfHeight<=tempy[ii]){
                                if(ii==0){
                                    widthLower = tempx[0];
                                    testt = false;
                                    lowerCheck = true;
                                }else{
                                    if(ii==99){
                                        widthLower = tempx[99];
                                        testt = false;
                                        lowerCheck = true;
                                    }
                                    else{
                                        widthLower = (tempx[ii] + tempx[ii-1])/2.0;
                                        testt = false;
                                        lowerCheck = true;
                                    }
                                }
                            }
                            ii++;
                            if(ii>=100)testt = false;
                        }
                    }
                    else{
                        if(maxIindex==2){
                            if(binnedData[1]>=halfHeight){
                                widthLower = bins[0] + (bins[1] - bins[0])*(halfHeight - binnedData[0])/(binnedData[1] - binnedData[0]);
                                lowerCheck = true;
                            }
                            else{
                                widthLower = bins[1] + (bins[2] - bins[1])*(halfHeight - binnedData[1])/(binnedData[2] - binnedData[1]);
                                lowerCheck = true;
                            }
                        }
                        else{
                            widthLower = bins[0] + (bins[1] - bins[0])*(halfHeight - binnedData[0])/(binnedData[1] - binnedData[0]);
                            lowerCheck = true;
                         }
                    }
                }
                else{
                    if(maxIindex>2){
                        if((binnedData[maxIindex]-binnedData[0])>halfHeight*0.5){
                            widthLower = bins[0] + (bins[1] - bins[0])*(halfHeight - binnedData[0])/(binnedData[1] - binnedData[0]);
                            lowerCheck = true;
                        }
                    }
                }
            }

            // upper limit
            int nTop = nBins - 1;
            int nDif = nBins - maxIindex;
            if(binnedData[nTop]==halfHeight){
                widthUpper = bins[nTop];
                upperCheck = true;
            }
            else{
                if(binnedData[nTop]<halfHeight){
                    if(nDif>=3){
                        double[] interpLy = new double[nDif];
                        double[] interpLx = new double[nDif];
                        int ii = 0;
                        for(int i=maxIindex; i<nBins; i++){
                            interpLy[ii] = binnedData[i];
                            interpLx[ii] = bins[i];
                            ii++;
                        }
                        CubicSpline csl = new CubicSpline(interpLx, interpLy);
                        double[] tempx = new double[100];
                        double[] tempy = new double[100];
                        double incr = (interpLx[nDif-1]-interpLx[0])/99;
                        double intr = interpLx[0];
                        for(int i=0; i<99; i++){
                            tempx[i] = intr;
                            tempy[i] = csl.interpolate(intr);
                            intr += incr;
                        }
                        tempy[99] = interpLy[nDif-1];
                        tempx[99] = interpLx[nDif-1];
                        boolean testt = true;
                        ii = 0;
                        while(testt){
                            if(halfHeight<=tempy[ii]){
                                if(ii==0){
                                    widthUpper = tempx[0];
                                    testt = false;
                                    upperCheck = true;
                                }else{
                                    if(ii==99){
                                        widthUpper = tempx[99];
                                        testt = false;
                                        upperCheck = true;
                                    }
                                    else{
                                        widthUpper = (tempx[ii] + tempx[ii-1])/2.0;
                                        testt = false;
                                        upperCheck = true;
                                    }
                                }
                            }
                            ii++;
                            if(ii>=100)testt = false;
                        }
                    }
                    else{
                        if(nDif==2){
                            if(binnedData[nTop-1]>=halfHeight){
                                widthUpper = bins[nTop-1] + (bins[nTop] - bins[nTop-1])*(halfHeight - binnedData[nTop-1])/(binnedData[nTop] - binnedData[nTop-1]);
                                upperCheck = true;
                            }
                            else{
                                widthUpper = bins[nTop-2] + (bins[nTop-1] - bins[nTop-2])*(halfHeight - binnedData[nTop-2])/(binnedData[nTop-1] - binnedData[nTop-2]);
                                upperCheck = true;
                            }
                        }
                        else{
                            widthUpper = bins[nTop-1] + (bins[nTop] - bins[nTop-1])*(halfHeight - binnedData[nTop-1])/(binnedData[nTop] - binnedData[nTop-1]);
                            upperCheck = true;
                        }
                    }
                }
                else{
                    if(nDif>2){
                        if((binnedData[maxIindex]-binnedData[nTop])>halfHeight*0.5){
                            widthUpper = bins[nTop-1] + (bins[nTop] - bins[nTop-1])*(halfHeight - binnedData[nTop-1])/(binnedData[nTop] - binnedData[nTop-1]);
                            upperCheck = true;
                        }
                    }
                }
            }

            // combine lower and upper half widths
            if(lowerCheck){
                if(upperCheck){
                    this.halfWidth = widthUpper - widthLower;
                }
                else{
                    this.halfWidth = (this.peakPoint - widthLower)*1.3;
                }
            }
            else{
                if(upperCheck){
                    this.halfWidth = (widthUpper - this.peakPoint)*1.3;
                }
                else{
                    System.out.println("Half height width could not be calculated - half range returned");
                    this.halfWidth = this.range/2.0;
                }

            }
            return this.halfWidth;
        }
        
        // CRITICAL VALUES
        // Critical value for the correlation coefficient
         public double correlationCoefficientCriticalValue(){
             return this.correlationCoefficientCriticalValue(this.significance);
         }
        
        public double correlationCoefficientCriticalValue(double signif){
            double critValue = Double.NaN;
            critValue = Stat.corrCoeffInverseCDFoneTailed(signif, this.numberOfDataPoints-2);      
            return critValue;
        }

        // GAUSSIAN PROBABILITY PLOT
        // user supplied initial estimates
        public void gaussianUserSuppliedInitialEstimates(double mue, double sigmae){
            this.userSuppliedEstimates[0] = true;
            this.userInitialEstimates[0] = new double[2];
            this.userInitialEstimates[0][0] = mue;
            this.userInitialEstimates[0][1] = sigmae;
        }
        
        public void normalUserSuppliedInitialEstimates(double mue, double sigmae){
            this.userSuppliedEstimates[0] = true;
            this.userInitialEstimates[0] = new double[2];
            this.userInitialEstimates[0][0] = mue;
            this.userInitialEstimates[0][1] = sigmae;
        }
        
        // Remove user supplied initial estimates
        public void removeGaussianUserSuppliedInitialEstimates(){
            this.userSuppliedEstimates[0] = false;
        }
        
        public void removeNormalUserSuppliedInitialEstimates(){
            this.userSuppliedEstimates[0] = false;
        }
        
        public void gaussianProbabilityPlot(){  
            this.gaussianProbabilityPlot("");
        }
        
        public void gaussianProbabilityPlot(String title){
            this.title = title;
            this.lastMethod = 0;

            // Check for suffient data points
            this.gaussianNumberOfParameters = 2;
            if(this.numberOfDataPoints<3)throw new IllegalArgumentException("There must be at least three data points - preferably considerably more");

            // Create instance of Regression
            Regression min = new Regression(this.sortedData, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)min.suppressErrorMessages();
            
            // Initial estimates
            double[] start = new double[2];
            if(this.userSuppliedEstimates[0]){
                start[0] = this.userInitialEstimates[0][0];
                start[1] = this.userInitialEstimates[0][1];
                if(start[1]<=0)start[1]=this.range*0.01;
            }
            else{
                start[0] = this.mean;
                start[1] = this.standardDeviation;
            }
            this.initialEstimates = start;
            double[] step = {this.stepFactor*Math.abs(start[0]), this.stepFactor*Math.abs(start[1])};
            if(step[0]==0)step[0] = this.stepFactor*this.standardDeviation;
            if(step[1]==0)step[1] = this.stepFactor*this.standardDeviation;

            // Add constraint; sigma>0
            min.addConstraint(1, -1, 0);

            // Create an instance of GaussProbPlotFunc
            GaussProbPlotFunc gppf = new GaussProbPlotFunc();
            gppf.setDataArray(this.numberOfDataPoints);

            // Obtain best probability plot varying mu and sigma
            // by minimizing the sum of squares of the differences between the ordered data and the ordered statistic medians
            min.simplex(gppf, DeepCopy.copy(start), step, this.tolerance);
            
             // Obtain best estimates for first minimisation
            double[] firstBests = min.getBestEstimates();

            // Get mu and sigma value errors
            double[] firstErrors = min.getBestEstimatesErrors();

            // Get sum of squares
            double ssu = min.getSumOfSquares();
            double ssw = ssu;
            double ss = ssu;
            if(this.weighted){
                ssw = min.getSumOfWeightedResidualSquares();
                ss = ssw;
            }
                    
            //Calculate new initial estimates
            double[] start2 = new double[this.gaussianNumberOfParameters];
            start2[0] = 2.0*firstBests[0] - start[0];
            step[0] = Math.abs(start2[0]*this.stepFactor);
            if(step[0]==0)step[0] = this.range*0.01;
            start2[1] = 2.0*firstBests[1] - start[1];
            if(start2[1]<=0.0)start2[1] = this.range*0.01;
            step[1] = Math.abs(start2[1]*this.stepFactor);
            
            min.simplex(gppf, DeepCopy.copy(start2), step, this.tolerance);

            // Get mu and sigma for best correlation coefficient
            this.gaussianParam = min.getBestEstimates();

            // Get mu and sigma errors for best correlation coefficient
            this.gaussianParamErrors = min.getBestEstimatesErrors();

            // Get sum of squares
            this.gaussianSumOfSquares = min.getSumOfSquares();
            this.gaussianWeightedSumOfSquares = this.gaussianSumOfSquares;
            double ss2 = this.gaussianSumOfSquares;
            if(this.weighted){
                this.gaussianWeightedSumOfSquares = min.getChiSquare();
                ss2 = this.gaussianWeightedSumOfSquares;
            }

            if(ss<ss2){
                this.gaussianParam = firstBests;
                this.gaussianParamErrors = firstErrors;
                this.gaussianSumOfSquares = ssu;
                if(this.weighted)this.gaussianWeightedSumOfSquares = ssw;
            }
            
            // Calculate Gaussian order statistic medians
            this.gaussianOrderMedians = Stat.gaussianOrderStatisticMedians(this.gaussianParam[0], this.gaussianParam[1], this.numberOfDataPoints);

            // Regression of the ordered data on the Gaussian order statistic medians
            Regression reg = new Regression(this.gaussianOrderMedians, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)reg.suppressErrorMessages();
            reg.linear();

            // Intercept and gradient of best fit straight line
            this.gaussianLine = reg.getBestEstimates();

            // Estimated erors of the intercept and gradient of best fit straight line
            this.gaussianLineErrors = reg.getBestEstimatesErrors();

            // Correlation coefficient
            this.gaussianCorrCoeff = reg.getSampleR();
            
            // Get sum of squares
            this.gaussianSumOfSquares = min.getSumOfSquares();
            
            // Output results to text file
            if(!this.suppressFile){
                    this.outputFile(this.lastMethod, min, reg, this.gaussianNumberOfParameters);
            }

            if(!this.suppressPlot){
             // Initialize data arrays for plotting
                double[][] data = PlotGraph.data(2,this.numberOfDataPoints);

                // Assign data to plotting arrays
                data[0] = this.gaussianOrderMedians;
                data[1] = this.sortedData;

                data[2] = this.gaussianOrderMedians;
                for(int i=0; i<this.numberOfDataPoints; i++){
                    data[3][i] = this.gaussianLine[0] + this.gaussianLine[1]*this.gaussianOrderMedians[i];
                }
                
                // Create instance of PlotGraph
                PlotGraph pg = new PlotGraph(data);
                int[] points = {4, 0};
                pg.setPoint(points);
                int[] lines = {0, 3};
                pg.setLine(lines);
                pg.setXaxisLegend("Gaussian Order Statistic Medians");
                pg.setYaxisLegend("Ordered Data Values");
                
                pg.setGraphTitle(this.titleStart + "Gaussian probability plot, " + this.title + ":   gradient = " + Fmath.truncate(this.gaussianLine[1], 4) + ", intercept = "  +  Fmath.truncate(this.gaussianLine[0], 4) + ",  R = " + Fmath.truncate(this.gaussianCorrCoeff, 4));
                pg.setGraphTitle2("  mu = " + Fmath.truncate(this.gaussianParam[0], 4) + ", sigma = "  +  Fmath.truncate(this.gaussianParam[1], 4));

               // Plot
                pg.plot();
                this.probPlotDone = true;
            }
                
            this.gaussianDone = true;
        }

        public void normalProbabilityPlot(String title){
            this.gaussianProbabilityPlot(title);
        }
        public void normalProbabilityPlot(){
            this.gaussianProbabilityPlot();
        }

        // Return Gaussian mu
        public double gaussianMu(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianParam[0];
        }

        // Return Gaussian mu error
        public double gaussianMuError(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianParamErrors[0];
        }

        // Return Gaussian sigma
        public double gaussianSigma(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianParam[1];
        }

        // Return Gaussian sigma error
        public double gaussianSigmaError(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianParamErrors[1];
        }

        // Return the Gaussian gradient
        public double gaussianGradient(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianLine[1];
        }

        // Return the error of the Gaussian gradient
        public double gaussianGradientError(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianLineErrors[1];
        }

        // Return the Gaussian intercept
        public double gaussianIntercept(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianLine[0];
        }

        // Return the error of the Gaussian intercept
        public double gaussianInterceptError(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianLineErrors[0];
        }

        // Return the Gaussian correlation coefficient
        public double gaussianCorrelationCoefficient(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianCorrCoeff;
        }

        // Return the sum of squares at the Gaussian minimum
        public double gaussianSumOfSquares(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianSumOfSquares;
        }

        // Return the Weighted sum of squares at the Gaussian minimum
        public double gaussianWeightedSumOfSquares(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianWeightedSumOfSquares;
        }


        // Return Gaussian order statistic medians
        public double[] gaussianOrderStatisticMedians(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianOrderMedians;
        }


        public double normalMu(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianParam[0];
        }

        // Return Gaussian mu error
        public double normalMuError(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianParamErrors[0];
        }

        // Return Gaussian sigma
        public double normalSigma(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianParam[1];
        }

        // Return Gaussian sigma error
        public double normalSigmaError(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianParamErrors[1];
        }

        // Return the Gaussian gradient
        public double normalGradient(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianLine[1];
        }

        // Return the error of the Gaussian gradient
        public double normalGradientError(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianLineErrors[1];
        }

        // Return the Gaussian intercept
        public double normalIntercept(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianLine[0];
        }

        // Return the error of the Gaussian intercept
        public double normalInterceptError(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianLineErrors[0];
        }

        // Return the Gaussian correlation coefficient
        public double normalCorrelationCoefficient(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianCorrCoeff;
        }

        // Return the sum of squares at the Gaussian minimum
        public double normalSumOfSquares(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianSumOfSquares;
        }

        // Return the Weighted sum of squares at the Gaussian minimum
        public double normalWeightedSumOfSquares(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianWeightedSumOfSquares;
        }
        
        // Return Gaussian order statistic medians
        public double[] normalOrderStatisticMedians(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianOrderMedians;
        }

        // STANDARD GAUSSIAN PROBABILITY PLOT
        public void gaussianStandardProbabilityPlot(){
            this.gaussianStandardProbabilityPlot("");
        }

        public void gaussianStandardProbabilityPlot(String title){
            this.title = title;
            this.lastMethod = 14;

            // Check for suffient data points
            this.gaussianStandardNumberOfParameters = 2;
            if(this.numberOfDataPoints<3)throw new IllegalArgumentException("There must be at least three data points - preferably considerably more");

            // Calculate Standard Gaussian order statistic medians
            this.gaussianStandardOrderMedians = Stat.gaussianOrderStatisticMedians(this.numberOfDataPoints);

            // Regression of the ordered data on the Standard Gaussian order statistic medians
            Regression reg = new Regression(this.gaussianStandardOrderMedians, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)reg.suppressErrorMessages();
            
            reg.linear();

            // Intercept and gradient of best fit straight line
            this.gaussianStandardLine = reg.getBestEstimates();

            // Estimated erors of the intercept and gradient of best fit straight line
            this.gaussianStandardLineErrors = reg.getBestEstimatesErrors();

            // Correlation coefficient
            this.gaussianStandardCorrCoeff = reg.getSampleR();

            // Get sum of squares
            this.gaussianStandardSumOfSquares = reg.getSumOfSquares();
            this.gaussianStandardWeightedSumOfSquares = reg.getSumOfSquares();
            if(this.weighted)this.gaussianStandardWeightedSumOfSquares = reg.getChiSquare();
            
            // Output results to text file
            if(!this.suppressFile){
                    this.outputFile(this.lastMethod, null, reg, 0);
            }

            if(!this.suppressPlot){
                // Initialize data arrays for plotting
                double[][] data = PlotGraph.data(2,this.numberOfDataPoints);

                // Assign data to plotting arrays
                data[0] = this.gaussianStandardOrderMedians;
                data[1] = this.sortedData;

                data[2] = this.gaussianStandardOrderMedians;
                for(int i=0; i<this.numberOfDataPoints; i++){
                    data[3][i] = this.gaussianStandardLine[0] + this.gaussianStandardLine[1]*this.gaussianStandardOrderMedians[i];
                }

                // Create instance of PlotGraph
                PlotGraph pg = new PlotGraph(data);
                int[] points = {4, 0};
                pg.setPoint(points);
                int[] lines = {0, 3};
                pg.setLine(lines);
                pg.setXaxisLegend("Standard Gaussian Order Statistic Medians");
                pg.setYaxisLegend("Ordered Data Values");
                pg.setGraphTitle("Standard Gaussian probability plot, " + this.title + ":   gradient = " + Fmath.truncate(this.gaussianStandardLine[1], 4) + ", intercept = "  +  Fmath.truncate(this.gaussianStandardLine[0], 4) + ",  R = " + Fmath.truncate(this.gaussianStandardCorrCoeff, 4));

                // Plot
                pg.plot();
                this.probPlotDone = true;
            }
            this.gaussianStandardDone = true;
            
        }

        public void normalStandardProbabilityPlot(){
            this.gaussianStandardProbabilityPlot();
        }
        
        public void normalStandardProbabilityPlot(String title){
            this.gaussianStandardProbabilityPlot(title);
        }

        // Return the Standard Gaussian gradient
        public double gaussianStandardGradient(){
            if(!this.gaussianStandardDone)this.gaussianStandardProbabilityPlot(this.title);
            return this.gaussianStandardLine[1];
        }

        // Return the error of the Standard Gaussian gradient
        public double gaussianStandardGradientError(){
            if(!this.gaussianStandardDone)this.gaussianStandardProbabilityPlot(this.title);
            return this.gaussianStandardLineErrors[1];
        }

        // Return the Standard Gaussian intercept
        public double gaussianStandardIntercept(){
            if(!this.gaussianStandardDone)this.gaussianStandardProbabilityPlot(this.title);
            return this.gaussianStandardLine[0];
        }

        // Return the error of the Standard Gaussian intercept
        public double gaussianStandardInterceptError(){
            if(!this.gaussianStandardDone)this.gaussianStandardProbabilityPlot(this.title);
            return this.gaussianStandardLineErrors[0];
        }

        // Return the Standard Gaussian correlation coefficient
        public double gaussianStandardCorrelationCoefficient(){
            if(!this.gaussianStandardDone)this.gaussianStandardProbabilityPlot(this.title);
            return this.gaussianStandardCorrCoeff;
        }

        // Return the sum of squares at the Standard Gaussian minimum
        public double gaussianStandardSumOfSquares(){
            if(!this.gaussianStandardDone)this.gaussianStandardProbabilityPlot(this.title);
            return this.gaussianStandardSumOfSquares;
        }

        // Return Standard Gaussian order statistic medians
        public double[] gaussianStandardOrderStatisticMedians(){
            if(!this.gaussianStandardDone)this.gaussianStandardProbabilityPlot(this.title);
            return this.gaussianStandardOrderMedians;
        }


        // Return the Standard Gaussian gradient
        public double normalStandardGradient(){
            if(!this.gaussianStandardDone)this.gaussianStandardProbabilityPlot(this.title);
            return this.gaussianStandardLine[1];
        }

        // Return the error of the Standard Gaussian gradient
        public double normalstandardGradientError(){
            if(!this.gaussianStandardDone)this.gaussianStandardProbabilityPlot(this.title);
            return this.gaussianStandardLineErrors[1];
        }

        // Return the Standard Gaussian intercept
        public double normalStandardInterceptError(){
            if(!this.gaussianStandardDone)this.gaussianStandardProbabilityPlot(this.title);
            return this.gaussianStandardLineErrors[0];
        }

        // Return the Standard Gaussian correlation coefficient
        public double normalStandardCorrelationCoefficient(){
            if(!this.gaussianStandardDone)this.gaussianStandardProbabilityPlot(this.title);
            return this.gaussianStandardCorrCoeff;
        }

        // Return the sum of squares at the Standard Gaussian minimum
        public double normalStandardSumOfSquares(){
            if(!this.gaussianStandardDone)this.gaussianStandardProbabilityPlot(this.title);
            return this.gaussianStandardSumOfSquares;
        }
        
        // Return the weighted sum of squares at the Standard Gaussian minimum
        public double normalStandardWeightedSumOfSquares(){
            if(!this.gaussianDone)this.gaussianProbabilityPlot(this.title);
            return this.gaussianStandardWeightedSumOfSquares;
        }

        // Return Standard Gaussian order statistic medians
        public double[] normalStandardOrderStatisticMedians(){
            if(!this.gaussianDone)this.gaussianStandardProbabilityPlot(this.title);
            return this.gaussianStandardOrderMedians;
        }

        
        // LOGISTIC PROBABILITY PLOT
        // user supplied initial estimates
        public void logisticUserSuppliedInitialEstimates(double mue, double betae){
            this.userSuppliedEstimates[8] = true;
            this.userInitialEstimates[8] = new double[2];
            this.userInitialEstimates[8][0] = mue;
            this.userInitialEstimates[8][1] = betae;
        }

        // Remove user supplied initial estimates
        public void removeLogisticUserSuppliedInitialEstimates(){
            this.userSuppliedEstimates[8] = false;
        }
        
        public void logisticProbabilityPlot(){
            this.lastMethod = 8;

            // Check for suffient data points
            this.logisticNumberOfParameters = 2;
            if(this.numberOfDataPoints<3)throw new IllegalArgumentException("There must be at least three data points - preferably considerably more");

            // Create instance of Regression
            Regression min = new Regression(this.sortedData, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)min.suppressErrorMessages();
            
            // Create initial estimates
            double[] start = new double[2];
            if(this.userSuppliedEstimates[8]){
                start[0] = this.userInitialEstimates[8][0];
                start[1] = this.userInitialEstimates[8][1];
                if(start[1]<=0.0)start[0] = this.standardDeviation/3.0;

            }
            else{
                start[0] = mean;
                start[1] = this.standardDeviation;            
            }
            this.initialEstimates = start;
            double[] step = {this.stepFactor*Math.abs(start[0]), this.stepFactor*Math.abs(start[1])};
            if(step[0]<=0.0)step[0] = this.standardDeviation/10.0;
            if(step[1]<=0.0)step[1] = this.standardDeviation/10.0;

             // Add constraint; beta>0
            min.addConstraint(1, -1, 0);

            // Create an instance of LogisticProbPlotFunc
            LogisticProbPlotFunc lppf = new LogisticProbPlotFunc();
            lppf.setDataArray(this.numberOfDataPoints);

            // Obtain best probability plot varying mu and sigma
            // by minimizing the sum of squares of the differences between the ordered data and the ordered statistic medians
            min.simplex(lppf, start, step, this.tolerance);

            // Get mu and beta for best correlation coefficient
            this.logisticParam = min.getBestEstimates();

            // Get mu and beta errors for best correlation coefficient
            this.logisticParamErrors = min.getBestEstimatesErrors();

            // Get sum of squares
            this.logisticSumOfSquares = min.getSumOfSquares();
            this.logisticWeightedSumOfSquares = this.logisticSumOfSquares;
            if(this.weighted)this.logisticWeightedSumOfSquares = min.getChiSquare();

            // Calculate Logistic order statistic medians
            this.logisticOrderMedians = Stat.logisticOrderStatisticMedians(this.logisticParam[0], this.logisticParam[1], this.numberOfDataPoints);

            // Regression of the ordered data on the Logistic order statistic medians3
            Regression reg = new Regression(this.logisticOrderMedians, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)reg.suppressErrorMessages();
            
            reg.linear();

            // Intercept and gradient of best fit straight line
            this.logisticLine = reg.getBestEstimates();

            // Estimated erors of the intercept and gradient of best fit straight line
            this.logisticLineErrors = reg.getBestEstimatesErrors();

            // Correlation coefficient
            this.logisticCorrCoeff = reg.getSampleR();
            
            // Get sum of squares
            this.logisticSumOfSquares = min.getSumOfSquares();

            if(!this.suppressFile){
                this.outputFile(this.lastMethod, min, reg, this.logisticNumberOfParameters);
            }
             
            if(!this.suppressPlot){
                // Initialize data arrays for plotting
                double[][] data = PlotGraph.data(2,this.numberOfDataPoints);

                // Assign data to plotting arrays
                data[0] = this.logisticOrderMedians;
                data[1] = this.sortedData;

                data[2] = logisticOrderMedians;
                for(int i=0; i<this.numberOfDataPoints; i++){
                    data[3][i] = this.logisticLine[0] + this.logisticLine[1]*logisticOrderMedians[i];
                }

                // Create instance of PlotGraph
                PlotGraph pg = new PlotGraph(data);
                int[] points = {4, 0};
                pg.setPoint(points);
                int[] lines = {0, 3};
                pg.setLine(lines);
                pg.setXaxisLegend("Logistic Order Statistic Medians");
                pg.setYaxisLegend("Ordered Data Values");
                pg.setGraphTitle("Logistic probability plot, " + this.title + ":   gradient = " + Fmath.truncate(this.logisticLine[1], 4) + ", intercept = "  +  Fmath.truncate(this.logisticLine[0], 4) + ",  R = " + Fmath.truncate(this.logisticCorrCoeff, 4));
                pg.setGraphTitle2("  mu = " + Fmath.truncate(this.logisticParam[0], 4) + ", beta = "  +  Fmath.truncate(this.logisticParam[1], 4));

                // Plot
                pg.plot();
                this.probPlotDone = true;
            }
            this.logisticDone = true;
        }

        // Return Logistic mu
        public double logisticMu(){
            if(!this.logisticDone)this.logisticProbabilityPlot();
            return this.logisticParam[0];
        }

        // Return Logistic mu error
        public double logisticMuError(){
            if(!this.logisticDone)this.logisticProbabilityPlot();
            return this.logisticParamErrors[0];
        }

        // Return Logistic beta
        public double logisticBeta(){
            if(!this.logisticDone)this.logisticProbabilityPlot();
            return this.logisticParam[1];
        }

        // Return Logistic beta error
        public double logisticBetaError(){
            if(!this.logisticDone)this.logisticProbabilityPlot();
            return this.logisticParamErrors[1];
        }

        // Return Logistic order statistic medians
        public double[] logisticOrderStatisticMedians(){
            if(!this.logisticDone)this.logisticProbabilityPlot();
            return this.logisticOrderMedians;
        }

        // Return the Logistic gradient
        public double logisticGradient(){
            if(!this.logisticDone)this.logisticProbabilityPlot();
            return this.logisticLine[1];
        }

        // Return the error of the Logistic gradient
        public double logisticGradientError(){
            if(!this.logisticDone)this.logisticProbabilityPlot();
            return this.logisticLineErrors[1];
        }

        // Return the Logistic intercept
        public double logisticIntercept(){
            if(!this.logisticDone)this.logisticProbabilityPlot();
            return this.logisticLine[0];
        }

        // Return the error of the Logistic intercept
        public double logisticInterceptError(){
            if(!this.logisticDone)this.logisticProbabilityPlot();
            return this.logisticLineErrors[0];
        }

        // Return the Logistic correlation coefficient
        public double logisticCorrelationCoefficient(){
            if(!this.logisticDone)this.logisticProbabilityPlot();
            return this.logisticCorrCoeff;
        }

        // Return the sum of squares at the Logistic minimum
        public double logisticSumOfSquares(){
            if(!this.logisticDone)this.logisticProbabilityPlot();
            return this.logisticSumOfSquares;
        }
        
        // Return the weighted sum of squares at the Logistic minimum
        public double logisticWeightedSumOfSquares(){
            if(!this.logisticDone)this.logisticProbabilityPlot();
            return this.logisticWeightedSumOfSquares;
        }

        

        // WEIBULL PROBABILITY PLOT
        //  Three parameter
        // User supplied initial estimates
        public void weibullUserSuppliedInitialEstimates(double mue, double sigmae, double gammae){
            this.userSuppliedEstimates[1] = true;
            this.userInitialEstimates[1] = new double[3];
            this.userInitialEstimates[1][0] = mue;
            this.userInitialEstimates[1][1] = sigmae;
            this.userInitialEstimates[1][2] = gammae;
        }
        
        // Remove user supplied initial estimates
        public void removeWeibullUserSuppliedInitialEstimates(){
            this.userSuppliedEstimates[1] = false;
        }
        
        //  Three parameter
        public void weibullProbabilityPlot(){
            this.lastMethod = 1;

            // Check for suffient data points
            this.weibullNumberOfParameters = 3;
            if(this.numberOfDataPoints<4)throw new IllegalArgumentException("There must be at least four data points - preferably considerably more");

            // Create instance of Regression
            Regression min = new Regression(this.sortedData, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)min.suppressErrorMessages();

            // Calculate or set initial estimates
            double[] start = new double[3];
            if(this.userSuppliedEstimates[1]){
                start[0] = this.userInitialEstimates[1][0];
                if(start[0]>this.minimum)start[0] = this.minimum - 0.1*Math.abs(this.minimum);
                start[1] = this.userInitialEstimates[1][1];
                if(start[1]<=0)start[1] = this.range*0.1;
                start[2] = this.userInitialEstimates[1][2];
                if(start[2]<=0)start[2] = 0.1;
            }
            else{
                start[0] = this.minimum - 0.1*Math.abs(this.minimum);
                start[1] = this.peakWidth();
                if(start[1]<=0)start[1] = this.range*0.1;
                start[2] = 4.0;
            }
            this.initialEstimates = start;
            double[] step = {Math.abs(this.stepFactor*start[0]), Math.abs(this.stepFactor*start[1]), Math.abs(this.stepFactor*start[2])};
            if(step[0]<=0)step[0] = this.range*0.01;
            if(step[1]<=0)step[1] = this.range*0.01;
            if(step[2]<=0)step[2] = 0.01;

            // Add constraint; mu<minimum, sigma>0, gamma>0
            min.addConstraint(0, +1, minimum);
            min.addConstraint(1, -1, 0);
            min.addConstraint(2, -1, 0);

            // Create an instance of WeibullProbPlotFunc
            WeibullProbPlotFunc wppf = new WeibullProbPlotFunc();
            wppf.setDataArray(this.numberOfDataPoints);

            // Obtain best probability plot varying mu, sigma and gamma
            // by minimizing the sum of squares of the differences between the ordered data and the ordered statistic medians
            min.simplex(wppf, Conv.copy(start), step, this.tolerance);

            // Obtain best estimates for first minimisation
            double[] firstBests = min.getBestEstimates();

            // Get mu and sigma value errors
            double[] firstErrors = min.getBestEstimatesErrors();

            // Get sum of squares
            double ssu = min.getSumOfSquares();
            double ssw = ssu;
            double ss = ssu;
            if(this.weighted){
                ssw = min.getSumOfWeightedResidualSquares();
                ss = ssw;
            }


            //Calculate new initial estimates
            double[] start2 = new double[this.weibullNumberOfParameters];
            start2[0] = 2.0*firstBests[0] - start[0];
            if(start2[0]>minimum)start2[0] = minimum*(1.0 - Math.abs(minimum)*0.05);
            step[0] = Math.abs(start2[0]*0.1);
            if(step[0]==0)step[0] = this.range*0.01;
            start2[1] = 2.0*firstBests[1] - start[1];
            if(start2[1]<=0.0)start2[1] = Math.abs(2.0*firstBests[1] - 0.98*start[1]);
            step[1] = Math.abs(start2[1]*0.1);
            start2[2] = 2.0*firstBests[2] - start[2];
            if(start2[1]<=0.0)start2[2] = Math.abs(2.0*firstBests[2] - 0.98*start[2]);
            step[2] = Math.abs(start2[2]*0.1);

            min.simplex(wppf, Conv.copy(start2), step, this.tolerance);

            // Get mu, sigma and gamma for best correlation coefficient
            this.weibullParam = min.getBestEstimates();

            // Get mu and sigma value errors
            this.weibullParamErrors = min.getBestEstimatesErrors();
          
            // Get sum of squares
            this.weibullSumOfSquares = min.getSumOfSquares();
            this.weibullWeightedSumOfSquares = this.weibullSumOfSquares;
            double ss2 = this.weibullSumOfSquares;
            if(this.weighted){
                this.weibullWeightedSumOfSquares = min.getChiSquare();
                ss2 = this.weibullWeightedSumOfSquares;
            }

            if(ss<ss2){
                this.weibullParam = firstBests;
                this.weibullParamErrors = firstErrors;
                this.weibullSumOfSquares = ssu;
                if(this.weighted)this.weibullWeightedSumOfSquares = ssw;
            }

            // Calculate Weibull order statistic medians
            this.weibullOrderMedians = Stat.weibullOrderStatisticMedians(this.weibullParam[0], this.weibullParam[1], this.weibullParam[2], this.numberOfDataPoints);

            // Regression of the ordered data on the Weibull order statistic medians
            Regression reg = new Regression(this.weibullOrderMedians, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)reg.suppressErrorMessages();
            reg.linear();

            // Intercept and gradient of best fit straight line
            this.weibullLine = reg.getBestEstimates();

            // Estimated erors of the intercept and gradient of best fit straight line
            this.weibullLineErrors = reg.getBestEstimatesErrors();

            // Correlation coefficient
            this.weibullCorrCoeff = reg.getSampleR();
            
            // Output results to text file
            if(!this.suppressFile){
                    this.outputFile(this.lastMethod, min, reg, this.weibullNumberOfParameters);
            }

            if(!this.suppressPlot){
                // Initialize data arrays for plotting
                double[][] data = PlotGraph.data(2,this.numberOfDataPoints);

                // Assign data to plotting arrays
                data[0] = this.weibullOrderMedians;
                data[1] = this.sortedData;

                data[2] = weibullOrderMedians;
                for(int i=0; i<this.numberOfDataPoints; i++){
                    data[3][i] = this.weibullLine[0] + this.weibullLine[1]*weibullOrderMedians[i];
                }


                // Create instance of PlotGraph
                PlotGraph pg = new PlotGraph(data);
                int[] points = {4, 0};
                pg.setPoint(points);
                int[] lines = {0, 3};
                pg.setLine(lines);
                pg.setXaxisLegend("Weibull Order Statistic Medians");
                pg.setYaxisLegend("Ordered Data Values");
                pg.setGraphTitle("Weibull probability plot, " + this.title + ":   gradient = " + Fmath.truncate(this.weibullLine[1], 4) + ", intercept = "  +  Fmath.truncate(this.weibullLine[0], 4) + ",  R = " + Fmath.truncate(this.weibullCorrCoeff, 4));
                pg.setGraphTitle2("  mu = " + Fmath.truncate(this.weibullParam[0], 4) + ", sigma = "  +  Fmath.truncate(this.weibullParam[1], 4) + ", gamma = "  +  Fmath.truncate(this.weibullParam[2], 4));

                // Plot
                pg.plot();
                this.probPlotDone = true;
            }
            this.weibullDone = true;
        }

        // WEIBULL PROBABILITY PLOT
        //  Three parameter
        public void weibullThreeParProbabilityPlot(){
            this.weibullProbabilityPlot();
        }

        // Return Weibull mu
        public double weibullMu(){
            if(!this.weibullDone)this.weibullProbabilityPlot();
            return this.weibullParam[0];
        }

        // Return Weibull mu error
        public double weibullMuError(){
            if(!this.weibullDone)this.weibullProbabilityPlot();
            return this.weibullParamErrors[0];
        }

        // Return Weibull sigma
        public double weibullSigma(){
            if(!this.weibullDone)this.weibullProbabilityPlot();
            return this.weibullParam[1];
        }

        // Return Weibull sigma error
        public double weibullSigmaError(){
            if(!this.weibullDone)this.weibullProbabilityPlot();
            return this.weibullParamErrors[1];
        }

        // Return Weibull gamma
        public double weibullGamma(){
            if(!this.weibullDone)this.weibullProbabilityPlot();
            return this.weibullParam[2];
        }

        // Return Weibull gamma error
        public double weibullGammaError(){
            if(!this.weibullDone)this.weibullProbabilityPlot();
            return this.weibullParamErrors[2];
        }

        // Return Weibull order statistic medians
        public double[] weibullOrderStatisticMedians(){
            if(!this.weibullDone)this.weibullProbabilityPlot();
            return this.weibullOrderMedians;
        }

        // Return the Weibull gradient
        public double weibullGradient(){
            if(!this.weibullDone)this.weibullProbabilityPlot();
            return this.weibullLine[1];
        }

        // Return the error of the Weibull gradient
        public double weibullGradientError(){
            if(!this.weibullDone)this.weibullProbabilityPlot();
            return this.weibullLineErrors[1];
        }

        // Return the Weibull intercept
        public double weibullIntercept(){
            if(!this.weibullDone)this.weibullProbabilityPlot();
            return this.weibullLine[0];
        }

        // Return the error of the Weibull intercept
        public double weibullInterceptError(){
            if(!this.weibullDone)this.weibullProbabilityPlot();
            return this.weibullLineErrors[0];
        }

        // Return the Weibull correlation coefficient
        public double weibullCorrelationCoefficient(){
            if(!this.weibullDone)this.weibullProbabilityPlot();
            return this.weibullCorrCoeff;
        }

        // Return the sum of squares at the Weibull minimum
        public double weibullSumOfSquares(){
            if(!this.weibullDone)this.weibullProbabilityPlot();
            return this.weibullSumOfSquares;
        }
        
         // Return the weighted sum of squares at the Weibull minimum
        public double weibullWeightedSumOfSquares(){
            if(!this.weibullDone)this.weibullProbabilityPlot();
            return this.weibullWeightedSumOfSquares;
        }

        // WEIBULL PROBABILITY PLOT
        //  Two parameter  (mu = 0)
        // User supplied initial estimates
        public void weibullTwoParUserSuppliedInitialEstimates(double sigmae, double gammae){
            this.userSuppliedEstimates[12] = true;
            this.userInitialEstimates[12] = new double[2];
            this.userInitialEstimates[12][0] = sigmae;
            this.userInitialEstimates[1][1] = gammae;
        }
        
        // Remove user supplied initial estimates
        public void removeWeibullTwoParUserSuppliedInitialEstimates(){
            this.userSuppliedEstimates[12] = false;
        }
        
        public void weibullTwoParProbabilityPlot(){
             this.lastMethod = 12;

            // Check for negative x values
            if(this.sortedData[0]<0){
                System.out.println("Method weibullTwoParProbabilityPlot: negative x value found - weibullThreeParProbabilityPlot called");
                this.weibullThreeParProbabilityPlot();
            }

            // Check data for suffient points
            this.weibullTwoParNumberOfParameters = 2;
            if(this.numberOfDataPoints<3)throw new IllegalArgumentException("There must be at least three data points - preferably considerably more");

            // Create instance of Regression
            Regression min = new Regression(this.sortedData, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)min.suppressErrorMessages();
            
            // Calculate initial estimates
            double[] start = new double[2];
            if(this.userSuppliedEstimates[12]){
                start[0] = this.userInitialEstimates[12][0];
                if(start[0]<=0.0)start[0] = this.range*0.01;
                start[1] = this.userInitialEstimates[12][1];
                if(start[1]<=0.0)start[1] = 1.0;
            }
            else{
                start[0] = this.peakWidth();
                start[1] = 4.0;
            }
            this.initialEstimates = start;
            double[] step = {Math.abs(this.stepFactor*start[0]), Math.abs(this.stepFactor*start[1])};
            if(step[0]<=0)step[0] = this.range*0.01;

            // Add constraint; sigma>0, gamma>0
            min.addConstraint(0, -1, 0);
            min.addConstraint(1, -1, 0);

            // Create an instance of WeibullTwoParProbPlotFunc
            WeibullTwoParProbPlotFunc wppf = new WeibullTwoParProbPlotFunc();
            wppf.setDataArray(this.numberOfDataPoints);

            // Obtain best probability plot varying sigma and gamma
            // by minimizing the sum of squares of the differences between the ordered data and the ordered statistic medians
            min.simplex(wppf, Conv.copy(start), step, this.tolerance);

            // Obtain best estimates for first minimisation
            double[] firstBests = min.getBestEstimates();

            // Get mu and sigma value errors
            double[] firstErrors = min.getBestEstimatesErrors();

            // Get sum of squares
            double ssu = min.getSumOfSquares();
            double ssw = ssu;
            double ss = ssu;
            if(this.weighted){
                ssw = min.getSumOfWeightedResidualSquares();
                ss = ssw;
            }
            
            //Calculate new initial estimates
            double[] start2 = new double[this.weibullTwoParNumberOfParameters];
            start2[0] = 2.0*firstBests[0] - start[0];
            if(start2[0]<=0)start2[0] = Math.abs(2.0*firstBests[0] - 0.98*start[0]);
            step[0] = Math.abs(start2[0]*0.1);
            if(step[0]<=0)step[0] = this.range*0.01;
            start2[1] = 2.0*firstBests[1] - start[1];
            if(start2[1]<=0.0)start2[1] = Math.abs(2.0*firstBests[1] - 0.98*start[1]);
            step[1] = Math.abs(start2[1]*0.1);

            min.simplex(wppf, Conv.copy(start2), step, this.tolerance);

            // Get sigma and gamma for best correlation coefficient
            this.weibullTwoParParam = min.getBestEstimates();

            // Get sigma and gamma value errors
            this.weibullTwoParParamErrors = min.getBestEstimatesErrors();

            // Get sum of squares
            this.weibullTwoParSumOfSquares = min.getSumOfSquares();
            this.weibullTwoParWeightedSumOfSquares = this.weibullTwoParSumOfSquares;
            double ss2 = this.weibullTwoParSumOfSquares;
            if(this.weighted){
                this.weibullTwoParWeightedSumOfSquares = min.getChiSquare();
                ss2 = this.weibullWeightedSumOfSquares;
            }

            if(ss<ss2){
                this.weibullTwoParParam = firstBests;
                this.weibullTwoParParamErrors = firstErrors;
                this.weibullTwoParSumOfSquares = ssu;
                if(this.weighted)this.weibullTwoParWeightedSumOfSquares = ssw;
            }

            // Calculate WeibullTwoPar order statistic medians
            this.weibullTwoParOrderMedians = Stat.weibullOrderStatisticMedians(this.weibullTwoParParam[0], this.weibullTwoParParam[1], this.numberOfDataPoints);

            // Regression of the ordered data on the Weibull order statistic medians
            Regression reg = new Regression(this.weibullTwoParOrderMedians, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)reg.suppressErrorMessages();
            reg.linear();

            // Intercept and gradient of best fit straight line
            this.weibullTwoParLine = reg.getBestEstimates();

            // Estimated erors of the intercept and gradient of best fit straight line
            this.weibullTwoParLineErrors = reg.getBestEstimatesErrors();

            // Correlation coefficient
            this.weibullTwoParCorrCoeff = reg.getSampleR();

             // Output results to text file
            if(!this.suppressFile){
                    this.outputFile(this.lastMethod, min, reg, this.weibullTwoParNumberOfParameters);
            }

            if(!this.suppressPlot){
                // Initialize data arrays for plotting
                double[][] data = PlotGraph.data(2,this.numberOfDataPoints);

                // Assign data to plotting arrays
                data[0] = this.weibullTwoParOrderMedians;
                data[1] = this.sortedData;

                data[2] = weibullTwoParOrderMedians;
                for(int i=0; i<this.numberOfDataPoints; i++){
                    data[3][i] = this.weibullTwoParLine[0] + this.weibullTwoParLine[1]*weibullTwoParOrderMedians[i];
                }

                // Create instance of PlotGraph
                PlotGraph pg = new PlotGraph(data);
                int[] points = {4, 0};
                pg.setPoint(points);
                int[] lines = {0, 3};
                pg.setLine(lines);
                pg.setXaxisLegend("Weibull Order Statistic Medians");
                pg.setYaxisLegend("Ordered Data Values");
                pg.setGraphTitle("Two Parameter Weibull probability plot, " + this.title + ":   gradient = " + Fmath.truncate(this.weibullTwoParLine[1], 4) + ", intercept = "  +  Fmath.truncate(this.weibullTwoParLine[0], 4) + ",  R = " + Fmath.truncate(this.weibullTwoParCorrCoeff, 4));
                pg.setGraphTitle2("  mu = 0, sigma = "  +  Fmath.truncate(this.weibullTwoParParam[0], 4) + ", gamma = "  +  Fmath.truncate(this.weibullTwoParParam[1], 4));

                // Plot
                pg.plot();
                this.probPlotDone = true;
            }

            this.weibullTwoParDone = true;
        }

        // Return Two Parameter Weibull sigma
        public double weibullTwoParSigma(){
            if(!this.weibullTwoParDone)this.weibullTwoParProbabilityPlot();
            return this.weibullTwoParParam[0];
        }

        // Return Two Parameter Weibull sigma error
        public double weibullTwoParSigmaError(){
            if(!this.weibullTwoParDone)this.weibullTwoParProbabilityPlot();
            return this.weibullTwoParParamErrors[0];
        }

        // Return Two Parameter Weibull gamma
        public double weibullTwoParGamma(){
            if(!this.weibullTwoParDone)this.weibullTwoParProbabilityPlot();
            return this.weibullTwoParParam[1];
        }

        // Return Two Parameter Weibull gamma error
        public double weibullTwoParGammaError(){
            if(!this.weibullTwoParDone)this.weibullTwoParProbabilityPlot();
            return this.weibullTwoParParamErrors[1];
        }

        // Return Two Parameter Weibull order statistic medians
        public double[] weibullTwoParOrderStatisticMedians(){
            if(!this.weibullTwoParDone)this.weibullTwoParProbabilityPlot();
            return this.weibullTwoParOrderMedians;
        }

        // Return the Two Parameter Weibull gradient
        public double weibullTwoParGradient(){
            if(!this.weibullTwoParDone)this.weibullTwoParProbabilityPlot();
            return this.weibullTwoParLine[1];
        }

        // Return the error of the Two Parameter Weibull gradient
        public double weibullTwoParGradientError(){
            if(!this.weibullTwoParDone)this.weibullTwoParProbabilityPlot();
            return this.weibullTwoParLineErrors[1];
        }

        // Return the Two Parameter Weibull intercept
        public double weibullTwoParIntercept(){
            if(!this.weibullTwoParDone)this.weibullTwoParProbabilityPlot();
            return this.weibullTwoParLine[0];
        }

        // Return the error of the Two Parameter Weibull intercept
        public double weibullTwoParInterceptError(){
            if(!this.weibullTwoParDone)this.weibullTwoParProbabilityPlot();
            return this.weibullTwoParLineErrors[0];
        }

        // Return the Two Parameter Weibull correlation coefficient
        public double weibullTwoParCorrelationCoefficient(){
            if(!this.weibullTwoParDone)this.weibullTwoParProbabilityPlot();
            return this.weibullTwoParCorrCoeff;
        }

        // Return the sum of squares at the Two Parameter Weibull minimum
        public double weibullTwoParSumOfSquares(){
            if(!this.weibullTwoParDone)this.weibullTwoParProbabilityPlot();
            return this.weibullTwoParSumOfSquares;
        }
        
        // Return the weighted sum of squares at the Two parameter Weibull minimum
        public double weibullTwoParWeightedSumOfSquares(){
            if(!this.weibullTwoParDone)this.weibullTwoParProbabilityPlot();
            return this.weibullTwoParWeightedSumOfSquares;
        }
        

        // STANDARD WEIBULL PROBABILITY PLOT
        // Standard (one parameter)  (mu = 0, sigma =1)
        // User supplied initial estimates
        public void weibullStandardUserSuppliedInitialEstimates(double gammae){
            this.userSuppliedEstimates[13] = true;
            this.userInitialEstimates[13] = new double[1];
            this.userInitialEstimates[13][0] = gammae;
        }
        
        // Remove user supplied initial estimates
        public void removeWeibullStandardUserSuppliedInitialEstimates(){
            this.userSuppliedEstimates[13] = false;
        }
        
        public void weibullStandardProbabilityPlot(){
            this.lastMethod = 13;

            // Check for negative x values
            if(this.sortedData[0]<0){
                System.out.println("Method weibullStandardProbabilityPlot: negative x value found - weibullThreeParProbabilityPlot called");
                this.weibullThreeParProbabilityPlot();
            }

            // Check data for suffient points
            this.weibullStandardNumberOfParameters = 1;
            if(this.numberOfDataPoints<3)throw new IllegalArgumentException("There must be at least three data points - preferably considerably more");

            // Create instance of Regression
            Regression min = new Regression(this.sortedData, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)min.suppressErrorMessages();

            // Calculate initial estimates
            double[] start = new double[1];
            if(this.userSuppliedEstimates[13]){
                start[0] = this.userInitialEstimates[13][0];
                if(start[0]<=0)start[0] = 1.0;
            }
            else{
                start[0] = 4.0;
            }
            double[] step = {Math.abs(this.stepFactor*start[0])};
            this.initialEstimates = start;

            // Add constraint; gamma>0
            min.addConstraint(0, -1, 0);

            // Create an instance of WeibullStandardProbPlotFunc
            WeibullStandardProbPlotFunc wppf = new WeibullStandardProbPlotFunc();
            wppf.setDataArray(this.numberOfDataPoints);

            // Obtain best probability plot varying gamma
            // by minimizing the sum of squares of the differences between the ordered data and the ordered statistic medians
            min.simplex(wppf, Conv.copy(start), step, this.tolerance);

            // Obtain best estimates for first minimisation
            double[] firstBests = min.getBestEstimates();

            // Get mu and sigma value errors
            double[] firstErrors = min.getBestEstimatesErrors();

            // Get sum of squares
            double ssu = min.getSumOfSquares();
            double ssw = ssu;
            double ss = ssu;
            if(this.weighted){
                ssw = min.getSumOfWeightedResidualSquares();
                ss = ssw;
            }

            //Calculate new initial estimates
            double[] start2 = new double[this.weibullStandardNumberOfParameters];
            start2[0] = 2.0*firstBests[0] - start[0];
            if(start2[0]<=0)start2[0] = Math.abs(2.0*firstBests[0] - 0.98*start[0]);
            step[0] = Math.abs(start2[0]*0.1);
            if(step[0]==0)step[0] = this.range*0.01;

            min.simplex(wppf, Conv.copy(start2), step, this.tolerance);

            // Get gamma for best correlation coefficient
            this.weibullStandardParam = min.getBestEstimates();

            // Get gamma value errors
            this.weibullStandardParamErrors = min.getBestEstimatesErrors();

            // Get sum of squares
            this.weibullStandardSumOfSquares = min.getSumOfSquares();
            this.weibullStandardWeightedSumOfSquares = this.weibullStandardSumOfSquares;
            double ss2 = this.weibullStandardSumOfSquares;
            if(this.weighted){
                this.weibullStandardWeightedSumOfSquares = min.getChiSquare();
                ss2 = this.weibullStandardWeightedSumOfSquares;
            }

            if(ss<ss2){
                this.weibullStandardParam = firstBests;
                this.weibullStandardParamErrors = firstErrors;
                this.weibullStandardSumOfSquares = ssu;
                if(this.weighted)this.weibullStandardWeightedSumOfSquares = ssw;
            }

            // Calculate Weibull Standard order statistic medians
            this.weibullStandardOrderMedians = Stat.weibullOrderStatisticMedians(this.weibullStandardParam[0], this.numberOfDataPoints);

            // Regression of the ordered data on the Weibull order statistic medians
            Regression reg = new Regression(this.weibullStandardOrderMedians, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)reg.suppressErrorMessages();
            reg.linear();

            // Intercept and gradient of best fit straight line
            this.weibullStandardLine = reg.getBestEstimates();

            // Estimated erors of the intercept and gradient of best fit straight line
            this.weibullStandardLineErrors = reg.getBestEstimatesErrors();

            // Correlation coefficient
            this.weibullStandardCorrCoeff = reg.getSampleR();

            // Output results to text file
            if(!this.suppressFile){
                    this.outputFile(this.lastMethod, min, reg, this.weibullStandardNumberOfParameters);
            }

            if(!this.suppressPlot){
                // Initialize data arrays for plotting
                double[][] data = PlotGraph.data(2,this.numberOfDataPoints);

                // Assign data to plotting arrays
                data[0] = this.weibullStandardOrderMedians;
                data[1] = this.sortedData;

                data[2] = weibullStandardOrderMedians;
                for(int i=0; i<this.numberOfDataPoints; i++){
                    data[3][i] = this.weibullStandardLine[0] + this.weibullStandardLine[1]*weibullStandardOrderMedians[i];
                }

                // Create instance of PlotGraph
                PlotGraph pg = new PlotGraph(data);
                int[] points = {4, 0};
                pg.setPoint(points);
                int[] lines = {0, 3};
                pg.setLine(lines);
                pg.setXaxisLegend("Weibull Order Statistic Medians");
                pg.setYaxisLegend("Ordered Data Values");
                pg.setGraphTitle("Standard Weibull probability plot, " + this.title + ":   gradient = " + Fmath.truncate(this.weibullStandardLine[1], 4) + ", intercept = "  +  Fmath.truncate(this.weibullStandardLine[0], 4) + ",  R = " + Fmath.truncate(this.weibullStandardCorrCoeff, 4));
                pg.setGraphTitle2("  mu = 0, sigma = 1, gamma = "  +  Fmath.truncate(this.weibullStandardParam[0], 4));

                // Plot
                pg.plot();
                this.probPlotDone = true;
            }

            this.weibullStandardDone = true;
        }

        // WEIBULL PROBABILITY PLOT
        // Standard (one parameter)  (mu = 0, sigma = 1)
        public void weibullOneParProbabilityPlot(){
             this.weibullStandardProbabilityPlot();
        }

        // Return Standard Weibull gamma
        public double weibullStandardGamma(){
            if(!this.weibullStandardDone)this.weibullStandardProbabilityPlot();
            return this.weibullStandardParam[0];
        }

        // Return Standard Weibull gamma error
        public double weibullStandardGammaError(){
            if(!this.weibullStandardDone)this.weibullStandardProbabilityPlot();
            return this.weibullStandardParamErrors[0];
        }

        // Return Standard Weibull order statistic medians
        public double[] weibullStandardOrderStatisticMedians(){
            if(!this.weibullStandardDone)this.weibullStandardProbabilityPlot();
            return this.weibullStandardOrderMedians;
        }

        // Return the Standard Weibull gradient
        public double weibullStandardGradient(){
            if(!this.weibullStandardDone)this.weibullStandardProbabilityPlot();
            return this.weibullStandardLine[1];
        }

        // Return the error of the Standard Weibull gradient
        public double weibullStandardGradientError(){
            if(!this.weibullStandardDone)this.weibullStandardProbabilityPlot();
            return this.weibullStandardLineErrors[1];
        }

        // Return the Standard Weibull intercept
        public double weibullStandardIntercept(){
            if(!this.weibullStandardDone)this.weibullStandardProbabilityPlot();
            return this.weibullStandardLine[0];
        }

        // Return the error of the Standard Weibull intercept
        public double weibullStandardInterceptError(){
            if(!this.weibullStandardDone)this.weibullStandardProbabilityPlot();
            return this.weibullStandardLineErrors[0];
        }

        // Return the Standard Weibull correlation coefficient
        public double weibullStandardCorrelationCoefficient(){
            if(!this.weibullStandardDone)this.weibullStandardProbabilityPlot();
            return this.weibullStandardCorrCoeff;
        }

        // Return the sum of squares at the Standard Weibull minimum
        public double weibullStandardSumOfSquares(){
            if(!this.weibullStandardDone)this.weibullStandardProbabilityPlot();
            return this.weibullStandardSumOfSquares;
        }
        
        // Return the weighted sum of squares at the Standard Weibull minimum
        public double weibullStandardWeightedSumOfSquares(){
            if(!this.weibullStandardDone)this.weibullStandardProbabilityPlot();
            return this.weibullStandardWeightedSumOfSquares;
        }


        // EXPONENTIAL PROBABILITY PLOT
        // User supplied initial estimates
        public void exponentialUserSuppliedInitialEstimates(double mue, double sigmae){
            this.userSuppliedEstimates[2] = true;
            this.userInitialEstimates[2] = new double[2];
            this.userInitialEstimates[2][0] = mue;
            this.userInitialEstimates[2][1] = sigmae;
        }
        
        // Remove user supplied initial estimates
        public void exponentialUserSuppliedInitialEstimates(){
            this.userSuppliedEstimates[2] = false;
        }    
        
        public void exponentialProbabilityPlot(){
            this.lastMethod = 2;

            // Check for suffient data points
            this.exponentialNumberOfParameters = 2;
            if(this.numberOfDataPoints<3)throw new IllegalArgumentException("There must be at least three data points - preferably considerably more");

            // Create instance of Regression
            Regression min = new Regression(this.sortedData, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)min.suppressErrorMessages();
            
            double[] start = new double[2];
            if(this.userSuppliedEstimates[2]){
                start[0] = this.userInitialEstimates[2][0];
                start[1] = this.userInitialEstimates[2][1];
            }
            else{
                start[0] = minimum;
                if(start[0]==0.0)start[0] = this.standardDeviation/3.0;
                start[1] = this.standardDeviation;
            }
            this.initialEstimates = start;
            double[] step = {this.stepFactor*start[0], this.stepFactor*start[1]};
            if(step[0]<=0.0)step[0] = this.standardDeviation/30.0;

             // Add constraint; sigma>0
            min.addConstraint(1, -1, 0);

            // Create an instance of ExponentialProbPlotFunc
            ExponentialProbPlotFunc eppf = new ExponentialProbPlotFunc();
            eppf.setDataArray(this.numberOfDataPoints);

            // Obtain best probability plot varying mu and sigma
            // by minimizing the sum of squares of the differences between the ordered data and the ordered statistic medians
            min.simplex(eppf, start, step, this.tolerance);

            // Get mu and sigma values
            this.exponentialParam = min.getBestEstimates();

            // Get mu and sigma value errors
            this.exponentialParamErrors = min.getBestEstimatesErrors();

            // Get sum of squares
            this.exponentialSumOfSquares = min.getSumOfSquares();
            this.exponentialWeightedSumOfSquares = this.exponentialSumOfSquares;
            if(this.weighted)this.exponentialWeightedSumOfSquares = min.getChiSquare();

            // Calculate Exponential order statistic medians (Weibull with gamma = 1)
            this.exponentialOrderMedians = Stat.weibullOrderStatisticMedians(this.exponentialParam[0], this.exponentialParam[1], 1.0, this.numberOfDataPoints);

            // Regression of the ordered data on the Exponential order statistic medians
            Regression reg = new Regression(this.exponentialOrderMedians, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)reg.suppressErrorMessages();
            reg.linear();

            // Intercept and gradient of best fit straight line
            this.exponentialLine = reg.getBestEstimates();

            // Estimated erors of the intercept and gradient of best fit straight line
            this.exponentialLineErrors = reg.getBestEstimatesErrors();

            // Correlation coefficient
            this.exponentialCorrCoeff = reg.getSampleR();
            
            // Output results to text file
            if(!this.suppressFile){
                    this.outputFile(this.lastMethod, min, reg, this.exponentialNumberOfParameters);
            }

            if(!this.suppressPlot){
                // Initialize data arrays for plotting
                double[][] data = PlotGraph.data(2,this.numberOfDataPoints);

                // Assign data to plotting arrays
                data[0] = this.exponentialOrderMedians;
                data[1] = this.sortedData;

                data[2] = exponentialOrderMedians;
                for(int i=0; i<this.numberOfDataPoints; i++){
                    data[3][i] = this.exponentialLine[0] + this.exponentialLine[1]*exponentialOrderMedians[i];
                }

                // Create instance of PlotGraph
                PlotGraph pg = new PlotGraph(data);
                int[] points = {4, 0};
                pg.setPoint(points);
                int[] lines = {0, 3};
                pg.setLine(lines);
                pg.setXaxisLegend("Exponential Order Statistic Medians");
                pg.setYaxisLegend("Ordered Data Values");
                pg.setGraphTitle("Exponential probability plot, " + this.title + ":   gradient = " + Fmath.truncate(this.exponentialLine[1], 4) + ", intercept = "  +  Fmath.truncate(this.exponentialLine[0], 4) + ",  R = " + Fmath.truncate(this.exponentialCorrCoeff, 4));
                pg.setGraphTitle2("  mu = " + Fmath.truncate(this.exponentialParam[0], 4) + ", sigma = "  +  Fmath.truncate(this.exponentialParam[1], 4));

                // Plot
                pg.plot();
                this.probPlotDone = true;
            }

            this.exponentialDone = true;   
        }

        // Return Exponential mu
        public double exponentialMu(){
            if(!this.exponentialDone)this.exponentialProbabilityPlot();
            return this.exponentialParam[0];
        }

        // Return Exponential mu error
        public double exponentialMuError(){
            if(!this.exponentialDone)this.exponentialProbabilityPlot();
            return this.exponentialParamErrors[0];
        }

        // Return Exponential sigma
        public double exponentialSigma(){
            if(!this.exponentialDone)this.exponentialProbabilityPlot();
            return this.exponentialParam[1];
        }

        // Return Exponential sigma error
        public double exponentialSigmaError(){
            if(!this.exponentialDone)this.exponentialProbabilityPlot();
            return this.exponentialParamErrors[1];
        }


        // Return Exponential order statistic medians
        public double[] exponentialOrderStatisticMedians(){
            if(!this.exponentialDone)this.exponentialProbabilityPlot();
            return this.exponentialOrderMedians;
        }

        // Return the Exponential gradient
        public double exponentialGradient(){
            if(!this.exponentialDone)this.exponentialProbabilityPlot();
            return this.exponentialLine[1];
        }

        // Return the error of the Exponential gradient
        public double exponentialGradientError(){
            if(!this.exponentialDone)this.exponentialProbabilityPlot();
            return this.exponentialLineErrors[1];
        }

        // Return the Exponential intercept
        public double exponentialIntercept(){
            if(!this.exponentialDone)this.exponentialProbabilityPlot();
            return this.exponentialLine[0];
        }

        // Return the error of the Exponential intercept
        public double exponentialInterceptError(){
            if(!this.exponentialDone)this.exponentialProbabilityPlot();
            return this.exponentialLineErrors[0];
        }

        // Return the Exponential correlation coefficient
        public double exponentialCorrelationCoefficient(){
            if(!this.exponentialDone)this.exponentialProbabilityPlot();
            return this.exponentialCorrCoeff;
        }

        // Return the sum of squares at the  minimum
        public double exponentialSumOfSquaresExponential(){
            if(!this.exponentialDone)this.exponentialProbabilityPlot();
            return this.exponentialSumOfSquares;
        }
        
        // Return the weighted sum of squares at the Exponential minimum
        public double exponentialWeightedSumOfSquares(){
            if(!this.exponentialDone)this.exponentialProbabilityPlot();
            return this.exponentialWeightedSumOfSquares;
        }


        // FRECHET PROBABILITY PLOT
        // Three parameter
        // User supplied initial estimates
        public void frechetUserSuppliedInitialEstimates(double mue, double sigmae, double gammae){
            this.userSuppliedEstimates[7] = true;
            this.userInitialEstimates[7] = new double[3];
            this.userInitialEstimates[7][0] = mue;
            this.userInitialEstimates[7][1] = sigmae;
            this.userInitialEstimates[7][2] = gammae;
        }
        
        // Remove user supplied initial estimates
        public void removeFrechetUserSuppliedInitialEstimates(){
            this.userSuppliedEstimates[7] = false;
        }
        
        public void frechetProbabilityPlot(){
            this.lastMethod = 7;

            // Check for suffient data points
            this.frechetNumberOfParameters = 3;
            if(this.numberOfDataPoints<4)throw new IllegalArgumentException("There must be at least four data points - preferably considerably more");

            // Create instance of Regression
            Regression min = new Regression(this.sortedData, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)min.suppressErrorMessages();

            // Calculate initial estimates
            double[] start = new double[3];
            if(this.userSuppliedEstimates[7]){
                start[0] = this.userInitialEstimates[7][0];
                if(start[0]>this.minimum)start[0] = this.minimum - 0.1*Math.abs(this.minimum);
                start[1] = this.userInitialEstimates[7][1];
                if(start[1]<=0.0)start[1] = 0.001*this.range;
                start[2] = this.userInitialEstimates[7][2];
                if(start[2]<0.0)start[2] = 0.0001;
            }
            else{
                start[0] = this.minimum - 0.1*Math.abs(this.minimum);
                start[1] = this.peakWidth();
                if(start[1]<=0.0)start[1] = 0.001*this.range;
                start[2] = 4.0;
            }
            this.initialEstimates = start;
            
            double[] step = {Math.abs(this.stepFactor*start[0]), Math.abs(this.stepFactor*start[1]), Math.abs(this.stepFactor*start[2])};
            if(step[0]==0)step[0] = this.range*0.01;

            // Add constraint; mu<minimum, sigma>0, gamma>0
            min.addConstraint(0, +1, minimum);
            min.addConstraint(1, -1, 0);
            min.addConstraint(2, -1, 0);

            // Create an instance of FrechetProbPlotFunc
            FrechetProbPlotFunc fppf = new FrechetProbPlotFunc();
            fppf.setDataArray(this.numberOfDataPoints);

            // Obtain best probability plot varying mu, sigma and gamma
            // by minimizing the sum of squares of the differences between the ordered data and the ordered statistic medians
            min.simplex(fppf, Conv.copy(start), step, this.tolerance);

            // Obtain best estimates for first minimisation
            double[] firstBests = min.getBestEstimates();
            double[] firstErrors = min.getBestEstimatesErrors();

            // Get sum of squares
            double ssu = min.getSumOfSquares();
            double ssw = ssu;
            double ss = ssu;
            if(this.weighted){
                ssw = min.getSumOfWeightedResidualSquares();
                ss = ssw;
            }

            //Calculate new initial estimates
            double[] start2 = new double[this.frechetNumberOfParameters];
            start2[0] = 2.0*firstBests[0] - start[0];
            if(start2[0]>minimum)start2[0] = minimum*(1.0 - Math.abs(minimum)*0.05);
            step[0] = Math.abs(start2[0]*0.1);
            if(step[0]==0)step[0] = this.range*0.01;
            start2[1] = 2.0*firstBests[1] - start[1];
            if(start2[1]<=0.0)start2[1] = Math.abs(2.0*firstBests[1] - 0.98*start[1]);
            step[1] = Math.abs(start2[1]*0.1);
            start2[2] = 2.0*firstBests[2] - start[2];
            if(start2[2]<=0.0)start2[2] = Math.abs(2.0*firstBests[2] - 0.98*start[2]);
            step[2] = Math.abs(start2[2]*0.1);

            min.simplex(fppf, Conv.copy(start2), step, this.tolerance);

            // Get mu, sigma and gamma for best correlation coefficient
            this.frechetParam = min.getBestEstimates();
            this.frechetParamErrors = min.getBestEstimatesErrors();
            
            // Get sum of squares
            this.frechetSumOfSquares = min.getSumOfSquares();
            this.frechetWeightedSumOfSquares = this.frechetSumOfSquares;
            double ss2 = this.frechetSumOfSquares;
            if(this.weighted){
                this.frechetWeightedSumOfSquares = min.getChiSquare();
                ss2 = this.frechetWeightedSumOfSquares;
            }

            if(ss<ss2){
                this.frechetParam = firstBests;
                this.frechetParamErrors = firstErrors;
                this.frechetSumOfSquares = ssu;
                if(this.weighted)this.frechetWeightedSumOfSquares = ssw;
            }

            // Calculate Frechet order statistic medians
            this.frechetOrderMedians = Stat.frechetOrderStatisticMedians(this.frechetParam[0], this.frechetParam[1], this.frechetParam[2], this.numberOfDataPoints);

            // Regression of the ordered data on the Frechet order statistic medians
            Regression reg = new Regression(this.frechetOrderMedians, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)reg.suppressErrorMessages();
            reg.linear();

            // Intercept and gradient of best fit straight line
            this.frechetLine = reg.getBestEstimates();

            // Estimated erors of the intercept and gradient of best fit straight line
            this.frechetLineErrors = reg.getBestEstimatesErrors();

            // Correlation coefficient
            this.frechetCorrCoeff = reg.getSampleR();
            
            // Get sum of squares
            this.gaussianSumOfSquares = min.getSumOfSquares();
            
            if(!this.suppressFile){
                    this.outputFile(this.lastMethod, min, reg, this.frechetNumberOfParameters);
            }

            if(!this.suppressPlot){
                // Initialize data arrays for plotting
                double[][] data = PlotGraph.data(2,this.numberOfDataPoints);

                // Assign data to plotting arrays
                data[0] = this.frechetOrderMedians;
                data[1] = this.sortedData;

                data[2] = frechetOrderMedians;
                for(int i=0; i<this.numberOfDataPoints; i++){
                    data[3][i] = this.frechetLine[0] + this.frechetLine[1]*frechetOrderMedians[i];
                }

                // Create instance of PlotGraph
                PlotGraph pg = new PlotGraph(data);
                int[] points = {4, 0};
                pg.setPoint(points);
                int[] lines = {0, 3};
                pg.setLine(lines);
                pg.setXaxisLegend("Frechet Order Statistic Medians");
                pg.setYaxisLegend("Ordered Data Values");
                pg.setGraphTitle("Frechet probability plot, " + this.title + ":   gradient = " + Fmath.truncate(this.frechetLine[1], 4) + ", intercept = "  +  Fmath.truncate(this.frechetLine[0], 4) + ",  R = " + Fmath.truncate(this.frechetCorrCoeff, 4));
                pg.setGraphTitle2("  mu = " + Fmath.truncate(this.frechetParam[0], 4) + ", sigma = "  +  Fmath.truncate(this.frechetParam[1], 4) + ", gamma = "  +  Fmath.truncate(this.frechetParam[2], 4));

                // Plot
                pg.plot();
                this.probPlotDone = true;
            }
            this.frechetDone = true;
            
        }

        // Return Frechet mu
        public double frechetMu(){
            if(!this.frechetDone)this.frechetProbabilityPlot();
            return this.frechetParam[0];
        }

        // Return Frechet mu error
        public double frechetMuError(){
            if(!this.frechetDone)this.frechetProbabilityPlot();
            return this.frechetParamErrors[0];
        }

        // Return Frechet sigma
        public double frechetSigma(){
            if(!this.frechetDone)this.frechetProbabilityPlot();
            return this.frechetParam[1];
        }

        // Return Frechet sigma error
        public double frechetSigmaError(){
            if(!this.frechetDone)this.frechetProbabilityPlot();
            return this.frechetParamErrors[1];
        }

        // Return Frechet gamma
        public double frechetGamma(){
            if(!this.frechetDone)this.frechetProbabilityPlot();
            return this.frechetParam[2];
        }

        // Return Frechet gamma error
        public double frechetGammaError(){
            if(!this.frechetDone)this.frechetProbabilityPlot();
            return this.frechetParamErrors[2];
        }

        // Return Frechet order statistic medians
        public double[] frechetOrderStatisticMedians(){
            if(!this.frechetDone)this.frechetProbabilityPlot();
            return this.frechetOrderMedians;
        }

        // Return the Frechet gradient
        public double frechetGradient(){
            if(!this.frechetDone)this.frechetProbabilityPlot();
            return this.frechetLine[1];
        }

        // Return the error of the Frechet gradient
        public double frechetGradientError(){
            if(!this.frechetDone)this.frechetProbabilityPlot();
            return this.frechetLineErrors[1];
        }

        // Return the Frechet intercept
        public double frechetIntercept(){
            if(!this.frechetDone)this.frechetProbabilityPlot();
            return this.frechetLine[0];
        }

        // Return the error of the Frechet intercept
        public double frechetInterceptError(){
            if(!this.frechetDone)this.frechetProbabilityPlot();
            return this.frechetLineErrors[0];
        }

        // Return the Frechet correlation coefficient
        public double frechetCorrelationCoefficient(){
            if(!this.frechetDone)this.frechetProbabilityPlot();
            return this.frechetCorrCoeff;
        }

        // Return the sum of squares at the Frechet minimum
        public double frechetSumOfSquares(){
            if(!this.frechetDone)this.frechetProbabilityPlot();
            return this.frechetSumOfSquares;
        }
        
        // Return the weighted sum of squares at the Frechet minimum
        public double frechetWeightedSumOfSquares(){
            if(!this.frechetDone)this.frechetProbabilityPlot();
            return this.frechetWeightedSumOfSquares;
        }

        public void frechetThreeParProbabilityPlot(){
            this.frechetProbabilityPlot();
        }
           
        
        // FRECHET PROBABILITY PLOT
        // Two parameter
        // User supplied initial estimates
        public void frechetTwoParUserSuppliedInitialEstimates(double sigmae, double gammae){
            this.userSuppliedEstimates[16] = true;
            this.userInitialEstimates[16] = new double[2];
            this.userInitialEstimates[16][0] = sigmae;
            this.userInitialEstimates[16][1] = gammae;
        }
        
        // Remove user supplied initial estimates
        public void removeFrechetTwoParUserSuppliedInitialEstimates(){
            this.userSuppliedEstimates[16] = false;
        }
        
        public void frechetTwoParProbabilityPlot(){
            this.lastMethod = 16;

            // Check for suffient data points
            this.frechetTwoParNumberOfParameters = 2;
            if(this.numberOfDataPoints<3)throw new IllegalArgumentException("There must be at least three data points - preferably considerably more");

            // Create instance of Regression
            Regression min = new Regression(this.sortedData, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)min.suppressErrorMessages();

            // Calculate initial estimates
            double[] start = new double[2];
            if(this.userSuppliedEstimates[16]){
                start[0] = this.userInitialEstimates[16][0];
                start[1] = this.userInitialEstimates[16][1];
            }
            else{
                start[0] = this.peakWidth();
                start[1] = 4.0;
            }
            if(start[0]<=0.0)start[0] = this.range/100.0;
            if(start[1]<=0.0)start[1] = 1.0;
            this.initialEstimates = start;
            double[] step = {Math.abs(this.stepFactor*start[0]), Math.abs(this.stepFactor*start[1])};

            // Add constraint; sigma>0, gamma>0
            min.addConstraint(0, -1, 0);
            min.addConstraint(1, -1, 0);

            // Create an instance of FrechetProbPlotFunc
            FrechetTwoParProbPlotFunc fppf = new FrechetTwoParProbPlotFunc();
            fppf.setDataArray(this.numberOfDataPoints);

            // Obtain best probability plot varying mu, sigma and gamma
            // by minimizing the sum of squares of the differences between the ordered data and the ordered statistic medians
            min.simplex(fppf, Conv.copy(start), step, this.tolerance);

            // Obtain best estimates for first minimisation
            double[] firstBests = min.getBestEstimates();
            double[] firstErrors = min.getBestEstimatesErrors();

            // Get sum of squares
            double ssu = min.getSumOfSquares();
            double ssw = ssu;
            double ss = ssu;
            if(this.weighted){
                ssw = min.getSumOfWeightedResidualSquares();
                ss = ssw;
            }

            //Calculate new initial estimates
            double[] start2 = new double[this.frechetTwoParNumberOfParameters];
            start2[0] = 2.0*firstBests[0] - start[0];
            if(start2[0]<=0.0)start2[0] = Math.abs(2.0*firstBests[0] - 0.98*start[0]);
            step[0] = Math.abs(start2[0]*0.1);
            start2[1] = 2.0*firstBests[1] - start[1];
            if(start2[1]<=0.0)start2[1] = Math.abs(2.0*firstBests[1] - 0.98*start[1]);
            step[1] = Math.abs(start2[1]*0.1);

            min.simplex(fppf, Conv.copy(start2), step, this.tolerance);

            // Get sigma and gamma for best correlation coefficient
            this.frechetTwoParParam = min.getBestEstimates();
            this.frechetTwoParParamErrors = min.getBestEstimatesErrors();
            
            // Get sum of squares
            this.frechetTwoParSumOfSquares = min.getSumOfSquares();
            this.frechetTwoParWeightedSumOfSquares = this.frechetTwoParSumOfSquares;
            double ss2 = this.frechetTwoParSumOfSquares;
            if(this.weighted){
                this.frechetTwoParWeightedSumOfSquares = min.getChiSquare();
                ss2 = this.frechetTwoParWeightedSumOfSquares;
            }

            if(ss<ss2){
                this.frechetTwoParParam = firstBests;
                this.frechetTwoParParamErrors = firstErrors;
                this.frechetTwoParSumOfSquares = ssu;
                if(this.weighted)this.frechetTwoParWeightedSumOfSquares = ssw;
            }

            // Calculate Frechet order statistic medians
            this.frechetTwoParOrderMedians = Stat.frechetOrderStatisticMedians(this.frechetTwoParParam[0], this.frechetTwoParParam[1], this.numberOfDataPoints);

            // Regression of the ordered data on the Frechet order statistic medians
            Regression reg = new Regression(this.frechetTwoParOrderMedians, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)reg.suppressErrorMessages();
            reg.linear();

            // Intercept and gradient of best fit straight line
            this.frechetTwoParLine = reg.getBestEstimates();

            // Estimated erors of the intercept and gradient of best fit straight line
            this.frechetTwoParLineErrors = reg.getBestEstimatesErrors();

            // Correlation coefficient
            this.frechetTwoParCorrCoeff = reg.getSampleR();

            // Output results to text file
            if(!this.suppressFile){
                    this.outputFile(this.lastMethod, min, reg, this.frechetTwoParNumberOfParameters);
            }

            if(!this.suppressPlot){
                // Initialize data arrays for plotting
                double[][] data = PlotGraph.data(2,this.numberOfDataPoints);

                // Assign data to plotting arrays
                data[0] = this.frechetTwoParOrderMedians;
                data[1] = this.sortedData;

                data[2] = frechetTwoParOrderMedians;
                for(int i=0; i<this.numberOfDataPoints; i++){
                    data[3][i] = this.frechetTwoParLine[0] + this.frechetTwoParLine[1]*this.frechetTwoParOrderMedians[i];
                }

                // Create instance of PlotGraph
                PlotGraph pg = new PlotGraph(data);
                int[] points = {4, 0};
                pg.setPoint(points);
                int[] lines = {0, 3};
                pg.setLine(lines);
                pg.setXaxisLegend("Two Parameter Frechet Order Statistic Medians");
                pg.setYaxisLegend("Ordered Data Values");
                pg.setGraphTitle("Two parameter Frechet probability plot, " + this.title + ":   gradient = " + Fmath.truncate(this.frechetTwoParLine[1], 4) + ", intercept = "  +  Fmath.truncate(this.frechetTwoParLine[0], 4) + ",  R = " + Fmath.truncate(this.frechetTwoParCorrCoeff, 4));
                pg.setGraphTitle2("  sigma = "  +  Fmath.truncate(this.frechetTwoParParam[1], 4) + ", gamma = "  +  Fmath.truncate(this.frechetTwoParParam[2], 4));

                // Plot
                pg.plot();
                this.probPlotDone = true;
            }

            this.frechetTwoParDone = true;
        }

        // Return Frechet mu
        public double frechetTwoParMu(){
            if(!this.frechetTwoParDone)this.frechetTwoParProbabilityPlot();
            return this.frechetTwoParParam[0];
        }

        // Return Frechet mu error
        public double frechetTwoParMuError(){
            if(!this.frechetTwoParDone)this.frechetTwoParProbabilityPlot();
            return this.frechetTwoParParamErrors[0];
        }

        // Return Frechet sigma
        public double frechetTwoParSigma(){
            if(!this.frechetTwoParDone)this.frechetTwoParProbabilityPlot();
            return this.frechetTwoParParam[1];
        }

        // Return Frechet sigma error
        public double frechetTwoParSigmaError(){
            if(!this.frechetTwoParDone)this.frechetTwoParProbabilityPlot();
            return this.frechetTwoParParamErrors[1];
        }

        // Return Frechet gamma
        public double frechetTwoParGamma(){
            if(!this.frechetTwoParDone)this.frechetTwoParProbabilityPlot();
            return this.frechetTwoParParam[2];
        }

        // Return Frechet gamma error
        public double frechetTwoParGammaError(){
            if(!this.frechetTwoParDone)this.frechetTwoParProbabilityPlot();
            return this.frechetTwoParParamErrors[2];
        }

        // Return Frechet order statistic medians
        public double[] frechetTwoParOrderStatisticMedians(){
            if(!this.frechetTwoParDone)this.frechetTwoParProbabilityPlot();
            return this.frechetTwoParOrderMedians;
        }

        // Return the Frechet gradient
        public double frechetTwoParGradient(){
            if(!this.frechetTwoParDone)this.frechetTwoParProbabilityPlot();
            return this.frechetTwoParLine[1];
        }

        // Return the error of the Frechet gradient
        public double frechetTwoParGradientError(){
            if(!this.frechetTwoParDone)this.frechetTwoParProbabilityPlot();
            return this.frechetTwoParLineErrors[1];
        }

        // Return the Frechet intercept
        public double frechetTwoParIntercept(){
            if(!this.frechetTwoParDone)this.frechetTwoParProbabilityPlot();
            return this.frechetTwoParLine[0];
        }

        // Return the error of the Frechet intercept
        public double frechetTwoParInterceptError(){
            if(!this.frechetTwoParDone)this.frechetTwoParProbabilityPlot();
            return this.frechetTwoParLineErrors[0];
        }

        // Return the Frechet correlation coefficient
        public double frechetTwoParCorrelationCoefficient(){
            if(!this.frechetTwoParDone)this.frechetTwoParProbabilityPlot();
            return this.frechetTwoParCorrCoeff;
        }

        // Return the sum of squares at the Frechet minimum
        public double frechetTwoParSumOfSquares(){
            if(!this.frechetTwoParDone)this.frechetTwoParProbabilityPlot();
            return this.frechetTwoParSumOfSquares;
        }
        
        // Return the weighted sum of squares at the Two Parameter Frechet minimum
        public double frechetTwoParWeightedSumOfSquares(){
            if(!this.frechetTwoParDone)this.frechetTwoParProbabilityPlot();
            return this.frechetTwoParWeightedSumOfSquares;
        }

        
        
        
        // FRECHET PROBABILITY PLOT
        // Standard
        // User supplied initial estimates
        public void frechetStandardUserSuppliedInitialEstimates(double gammae){
            this.userSuppliedEstimates[17] = true;
            this.userInitialEstimates[17] = new double[1];
            this.userInitialEstimates[17][0] = gammae;
        }
        
        // Remove user supplied initial estimates
        public void removeFrechetStandardUserSuppliedInitialEstimates(){
            this.userSuppliedEstimates[17] = false;
        }
        
        public void frechetStandardProbabilityPlot(){
            this.lastMethod = 17;

            // Check for suffient data points
            this.frechetStandardNumberOfParameters = 1;
            if(this.numberOfDataPoints<2)throw new IllegalArgumentException("There must be at least two data points - preferably considerably more");

            // Create instance of Regression
            Regression min = new Regression(this.sortedData, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)min.suppressErrorMessages();

            // Calculate initial estimates
            double[] start = new double[1];
            if(this.userSuppliedEstimates[17]){
                start[0] = this.userInitialEstimates[17][0];
                if(start[0]<=0)start[0] = 0.1;
            }
            else{
                start[0] = 4.0;
            }
            this.initialEstimates = start;
            double[] step = {Math.abs(this.stepFactor*start[0])};


            // Add constraint;  gamma>0
            min.addConstraint(0, -1, 0);

            // Create an instance of FrechetProbPlotFunc
            FrechetStandardProbPlotFunc fppf = new FrechetStandardProbPlotFunc();
            fppf.setDataArray(this.numberOfDataPoints);

            // Obtain best probability plot varying mu, sigma and gamma
            // by minimizing the sum of squares of the differences between the ordered data and the ordered statistic medians
            min.simplex(fppf, Conv.copy(start), step, this.tolerance);

            // Obtain best estimates for first minimisation
            double[] firstBests = min.getBestEstimates();
            double[] firstErrors = min.getBestEstimatesErrors();
            
            // Get sum of squares
            double ssu = min.getSumOfSquares();
            double ssw = ssu;
            double ss = ssu;
            if(this.weighted){
                ssw = min.getSumOfWeightedResidualSquares();
                ss = ssw;
            }

            //Calculate new initial estimates
            double[] start2 = new double[this.frechetStandardNumberOfParameters];
            start2[0] = 2.0*firstBests[0] - start[0];
            if(start2[0]<=0)start2[0] = 2.0*firstBests[0] - 0.98*start[0];

            min.simplex(fppf, Conv.copy(start2), step, this.tolerance);

            // Get gamma for best correlation coefficient
            this.frechetStandardParam = min.getBestEstimates();
            this.frechetStandardParamErrors = min.getBestEstimatesErrors();
            
            // Get sum of squares
            this.frechetStandardSumOfSquares = min.getSumOfSquares();
            this.frechetStandardWeightedSumOfSquares = this.frechetStandardSumOfSquares;
            double ss2 = this.frechetStandardSumOfSquares;
            if(this.weighted){
                this.frechetStandardWeightedSumOfSquares = min.getChiSquare();
                ss2 = this.frechetStandardWeightedSumOfSquares;
            }

            if(ss<ss2){
                this.frechetStandardParam = firstBests;
                this.frechetStandardParamErrors = firstErrors;
                this.frechetStandardSumOfSquares = ssu;
                if(this.weighted)this.frechetStandardWeightedSumOfSquares = ssw;
            }

            // Calculate Frechet order statistic medians
            this.frechetStandardOrderMedians = Stat.frechetOrderStatisticMedians(this.frechetStandardParam[0], this.numberOfDataPoints);

            // Regression of the ordered data on the Frechet order statistic medians
            Regression reg = new Regression(this.frechetStandardOrderMedians, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)reg.suppressErrorMessages();
            reg.linear();

            // Intercept and gradient of best fit straight line
            this.frechetStandardLine = reg.getBestEstimates();

            // Estimated erors of the intercept and gradient of best fit straight line
            this.frechetStandardLineErrors = reg.getBestEstimatesErrors();

            // Correlation coefficient
            this.frechetStandardCorrCoeff = reg.getSampleR();
            
             // Output results to text file
            if(!this.suppressFile){
                    this.outputFile(this.lastMethod, min, reg, this.frechetStandardNumberOfParameters);
            }

            if(!this.suppressPlot){
                // Initialize data arrays for plotting
                double[][] data = PlotGraph.data(2,this.numberOfDataPoints);

                // Assign data to plotting arrays
                data[0] = this.frechetStandardOrderMedians;
                data[1] = this.sortedData;

                data[2] = frechetStandardOrderMedians;
                for(int i=0; i<this.numberOfDataPoints; i++){
                    data[3][i] = this.frechetStandardLine[0] + this.frechetStandardLine[1]*frechetStandardOrderMedians[i];
                }

                // Create instance of PlotGraph
                PlotGraph pg = new PlotGraph(data);
                int[] points = {4, 0};
                pg.setPoint(points);
                int[] lines = {0, 3};
                pg.setLine(lines);
                pg.setXaxisLegend("Frechet StandardOrder Statistic Medians");
                pg.setYaxisLegend("Ordered Data Values");
                pg.setGraphTitle("Standard Frechet probability plot, " + this.title + ":   gradient = " + Fmath.truncate(this.frechetStandardLine[1], 4) + ", intercept = "  +  Fmath.truncate(this.frechetStandardLine[0], 4) + ",  R = " + Fmath.truncate(this.frechetStandardCorrCoeff, 4));
                pg.setGraphTitle2("  gamma = "  +  Fmath.truncate(this.frechetStandardParam[0], 4));

                // Plot
                pg.plot();
                this.probPlotDone = true;
            }

            this.frechetStandardDone = true;
        }

        // Return Frechet mu
        public double frechetStandardMu(){
            if(!this.frechetStandardDone)this.frechetStandardProbabilityPlot();
            return this.frechetStandardParam[0];
        }

        // Return Frechet mu error
        public double frechetStandardMuError(){
            if(!this.frechetStandardDone)this.frechetStandardProbabilityPlot();
            return this.frechetStandardParamErrors[0];
        }

        // Return Frechet sigma
        public double frechetStandardSigma(){
            if(!this.frechetStandardDone)this.frechetStandardProbabilityPlot();
            return this.frechetStandardParam[1];
        }

        // Return Frechet sigma error
        public double frechetStandardSigmaError(){
            if(!this.frechetStandardDone)this.frechetStandardProbabilityPlot();
            return this.frechetStandardParamErrors[1];
        }

        // Return Frechet gamma
        public double frechetStandardGamma(){
            if(!this.frechetStandardDone)this.frechetStandardProbabilityPlot();
            return this.frechetStandardParam[2];
        }

        // Return Frechet gamma error
        public double frechetStandardGammaError(){
            if(!this.frechetStandardDone)this.frechetStandardProbabilityPlot();
            return this.frechetStandardParamErrors[2];
        }

        // Return Frechet order statistic medians
        public double[] frechetStandardOrderStatisticMedians(){
            if(!this.frechetStandardDone)this.frechetStandardProbabilityPlot();
            return this.frechetStandardOrderMedians;
        }

        // Return the Frechet gradient
        public double frechetStandardGradient(){
            if(!this.frechetStandardDone)this.frechetStandardProbabilityPlot();
            return this.frechetStandardLine[1];
        }

        // Return the error of the Frechet gradient
        public double frechetStandardGradientError(){
            if(!this.frechetStandardDone)this.frechetStandardProbabilityPlot();
            return this.frechetStandardLineErrors[1];
        }

        // Return the Frechet intercept
        public double frechetStandardIntercept(){
            if(!this.frechetStandardDone)this.frechetStandardProbabilityPlot();
            return this.frechetStandardLine[0];
        }

        // Return the error of the Frechet intercept
        public double frechetStandardInterceptError(){
            if(!this.frechetStandardDone)this.frechetStandardProbabilityPlot();
            return this.frechetStandardLineErrors[0];
        }

        // Return the Frechet correlation coefficient
        public double frechetStandardCorrelationCoefficient(){
            if(!this.frechetStandardDone)this.frechetStandardProbabilityPlot();
            return this.frechetStandardCorrCoeff;
        }

        // Return the sum of squares at the Frechet minimum
        public double frechetStandardSumOfSquares(){
            if(!this.frechetStandardDone)this.frechetStandardProbabilityPlot();
            return this.frechetStandardSumOfSquares;
        }
        
        // Return the weighted sum of squares at the Standard Frechet minimum
        public double frechetStandardWeightedSumOfSquares(){
            if(!this.frechetStandardDone)this.frechetStandardProbabilityPlot();
            return this.frechetStandardWeightedSumOfSquares;
        }
        

        // GUMBEL (MINIMUM ORDER STATISTIC) PROBABILITY PLOT
        // User supplied initial estimates
        public void gumbelMinUserSuppliedInitialEstimates(double mue, double sigmae){
            this.userSuppliedEstimates[5] = true;
            this.userInitialEstimates[5] = new double[2];
            this.userInitialEstimates[5][0] = mue;
            this.userInitialEstimates[5][1] = sigmae;
        }
        
        // Remove user supplied initial estimates
        public void removeGumbelMinUserSuppliedInitialEstimates(){
            this.userSuppliedEstimates[5] = false;
        }
        
        public void gumbelMinProbabilityPlot(){
            this.lastMethod = 5;

            // Check for suffient data points
            this.gumbelMinNumberOfParameters = 2;
            if(this.numberOfDataPoints<3)throw new IllegalArgumentException("There must be at least three data points - preferably considerably more");

            // Create instance of Regression
            Regression min = new Regression(this.sortedData, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)min.suppressErrorMessages();
            
            // Calculate initial estimates
            double[] start = new double[2];
            if(this.userSuppliedEstimates[5]){
                start[0] = this.userInitialEstimates[5][0];
                start[1] = this.userInitialEstimates[5][1]; 
                if(start[1]<0.0)start[1] = this.standardDeviation/3.0;
            }
            else{
                start[0] = mean;
                start[1] = this.standardDeviation;
            }
            double[] step = {this.stepFactor*start[0], this.stepFactor*start[1]};
            if(step[0]==0.0)step[0] = this.standardDeviation/30.0;
            this.initialEstimates = start;

             // Add constraint; sigma>0
            min.addConstraint(1, -1, 0);

            // Create an instance of Gumbel (minimum order statistic)ProbPlotFunc
            GumbelMinProbPlotFunc gmippf = new GumbelMinProbPlotFunc();
            gmippf.setDataArray(this.numberOfDataPoints);

            // Obtain best probability plot varying mu and sigma
            // by minimizing the sum of squares of the differences between the ordered data and the ordered statistic medians
            min.simplex(gmippf, start, step, this.tolerance);

             // Get mu and sigma values
            this.gumbelMinParam = min.getBestEstimates();

            // Get mu and sigma value errors
            this.gumbelMinParamErrors = min.getBestEstimatesErrors();

            // Get sum of squares
            this.gumbelMinSumOfSquares = min.getSumOfSquares();
            this.gumbelMinWeightedSumOfSquares = this.gumbelMinSumOfSquares;
            if(this.weighted)this.gumbelMinWeightedSumOfSquares = min.getChiSquare();

            // Calculate Gumbel (minimum order statistic) order statistic medians
            this.gumbelMinOrderMedians = Stat.gumbelMinOrderStatisticMedians(this.gumbelMinParam[0], this.gumbelMinParam[1], this.numberOfDataPoints);

            // Regression of the ordered data on the Gumbel (minimum order statistic) order statistic medians
            Regression reg = new Regression(this.gumbelMinOrderMedians, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)reg.suppressErrorMessages();
            reg.linear();

            // Intercept and gradient of best fit straight line
            this.gumbelMinLine = reg.getBestEstimates();

            // Estimated erors of the intercept and gradient of best fit straight line
            this.gumbelMinLineErrors = reg.getBestEstimatesErrors();

            // Correlation coefficient
            this.gumbelMinCorrCoeff = reg.getSampleR();

            // Output results to text file
            if(!this.suppressFile){
                    this.outputFile(this.lastMethod, min, reg, this.gumbelMinNumberOfParameters);
            }

            if(!this.suppressPlot){
                // Initialize data arrays for plotting
                double[][] data = PlotGraph.data(2,this.numberOfDataPoints);

                // Assign data to plotting arrays
                data[0] = this.gumbelMinOrderMedians;
                data[1] = this.sortedData;

                data[2] = gumbelMinOrderMedians;
                for(int i=0; i<this.numberOfDataPoints; i++){
                    data[3][i] = this.gumbelMinLine[0] + this.gumbelMinLine[1]*gumbelMinOrderMedians[i];
                }

                // Create instance of PlotGraph
                PlotGraph pg = new PlotGraph(data);
                int[] points = {4, 0};
                pg.setPoint(points);
                int[] lines = {0, 3};
                pg.setLine(lines);
                pg.setXaxisLegend("Gumbel (minimum order statistic) Order Statistic Medians");
                pg.setYaxisLegend("Ordered Data Values");
                pg.setGraphTitle("Gumbel (minimum order statistic) probability plot, " + this.title + ":   gradient = " + Fmath.truncate(this.gumbelMinLine[1], 4) + ", intercept = "  +  Fmath.truncate(this.gumbelMinLine[0], 4) + ",  R = " + Fmath.truncate(this.gumbelMinCorrCoeff, 4));
                pg.setGraphTitle2("  mu = " + Fmath.truncate(this.gumbelMinParam[0], 4) + ", sigma = "  +  Fmath.truncate(this.gumbelMinParam[1], 4));

                // Plot
                pg.plot();
                this.probPlotDone = true;
            }

            this.gumbelMinDone = true;
        }

        // Return Gumbel (minimum order statistic) mu
        public double gumbelMinMu(){
            if(!this.gumbelMinDone)this.gumbelMinProbabilityPlot();
            return this.gumbelMinParam[0];
        }

        // Return Gumbel (minimum order statistic) mu error
        public double gumbelMinMuError(){
            if(!this.gumbelMinDone) this.gumbelMinProbabilityPlot();
            return this.gumbelMinParamErrors[0];
        }

        // Return Gumbel (minimum order statistic) sigma
        public double gumbelMinSigma(){
            if(!this.gumbelMinDone)this.gumbelMinProbabilityPlot();
            return this.gumbelMinParam[1];
        }

        // Return Gumbel (minimum order statistic) sigma error
        public double gumbelMinSigmaError(){
            if(!this.gumbelMinDone)this.gumbelMinProbabilityPlot();
            return this.gumbelMinParamErrors[1];
        }

        // Return Gumbel (minimum order statistic) order statistic medians
        public double[] gumbelMinOrderStatisticMedians(){
            if(!this.gumbelMinDone)this.gumbelMinProbabilityPlot();
            return this.gumbelMinOrderMedians;
        }

        // Return the Gumbel (minimum order statistic) gradient
        public double gumbelMinGradient(){
            if(!this.gumbelMinDone)this.gumbelMinProbabilityPlot();
            return this.gumbelMinLine[1];
        }

        // Return the error of the Gumbel (minimum order statistic) gradient
        public double gumbelMinGradientError(){
            if(!this.gumbelMinDone)this.gumbelMinProbabilityPlot();
            return this.gumbelMinLineErrors[1];
        }

        // Return the Gumbel (minimum order statistic) intercept
        public double gumbelMinIntercept(){
            if(!this.gumbelMinDone)this.gumbelMinProbabilityPlot();
            return this.gumbelMinLine[0];
        }

        // Return the error of the Gumbel (minimum order statistic) intercept
        public double gumbelMinInterceptError(){
            if(!this.gumbelMinDone)this.gumbelMinProbabilityPlot();
            return this.gumbelMinLineErrors[0];
        }

        // Return the Gumbel (minimum order statistic) correlation coefficient
        public double gumbelMinCorrelationCoefficient(){
            if(!this.gumbelMinDone)this.gumbelMinProbabilityPlot();
            return this.gumbelMinCorrCoeff;
        }

        // Return the sum of squares at the Gumbel (minimum order statistic) minimum
        public double gumbelMinSumOfSquares(){
            if(!this.gumbelMinDone)this.gumbelMinProbabilityPlot();
            return this.gumbelMinSumOfSquares;
        }

        // Return the weighted sum of squares at the Gumbel (minimum order statistic) minimum
        public double gumbelMinWeightedSumOfSquares(){
            if(!this.gumbelMinDone)this.gumbelMinProbabilityPlot();
            return this.gumbelMinWeightedSumOfSquares;
        }
        
        

        // GUMBEL (MAXIMUM ORDER STATISTIC) PROBABILITY PLOT
        // User supplied initial estimates
        public void gumbelMaxUserSuppliedInitialEstimates(double mue, double sigmae){
            this.userSuppliedEstimates[6] = true;
            this.userInitialEstimates[6] = new double[2];
            this.userInitialEstimates[6][0] = mue;
            this.userInitialEstimates[6][1] = sigmae;
        }
        
        // Remove user supplied initial estimates
        public void removeGumbelMaxUserSuppliedInitialEstimates(){
            this.userSuppliedEstimates[6] = false;
        }
        
        public void gumbelMaxProbabilityPlot(){
            this.lastMethod = 6;

            // Check for suffient data points
            this.gumbelMaxNumberOfParameters = 2;
            if(this.numberOfDataPoints<3)throw new IllegalArgumentException("There must be at least three data points - preferably considerably more");

            // Create instance of Regression
            Regression min = new Regression(this.sortedData, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)min.suppressErrorMessages();
            
            // Calculate initial estimates
            double[] start = new double[2];
            if(this.userSuppliedEstimates[6]){
                start[0] = this.userInitialEstimates[6][0];
                start[1] = this.userInitialEstimates[6][1]; 
                if(start[1]<0.0)start[1] = this.standardDeviation/3.0;
            }
            else{
                start[0] = mean;
                start[1] = this.standardDeviation;
            }
            double[] step = {this.stepFactor*start[0], this.stepFactor*start[1]};
            if(step[0]==0.0)step[0] = this.standardDeviation/30.0;
            this.initialEstimates = start;

             // Add constraint; sigma>0
            min.addConstraint(1, -1, 0);

            // Create an instance of Gumbel (maximum order statistic)ProbPlotFunc
            GumbelMaxProbPlotFunc gmappf = new GumbelMaxProbPlotFunc();
            gmappf.setDataArray(this.numberOfDataPoints);

            // Obtain best probability plot varying mu and sigma
            // by minimizing the sum of squares of the differences between the ordered data and the ordered statistic medians
            min.simplex(gmappf, start, step, this.tolerance);

              // Get mu and sigma values
            this.gumbelMaxParam = min.getBestEstimates();

            // Get mu and sigma value errors
            this.gumbelMaxParamErrors = min.getBestEstimatesErrors();

            // Get sum of squares
            this.gumbelMaxSumOfSquares = min.getSumOfSquares();
            this.gumbelMaxWeightedSumOfSquares = this.gumbelMaxSumOfSquares;
            if(this.weighted)this.gumbelMaxWeightedSumOfSquares = min.getChiSquare();


            // Calculate Gumbel (maximum order statistic) order statistic medians
            this.gumbelMaxOrderMedians = Stat.gumbelMaxOrderStatisticMedians(this.gumbelMaxParam[0], this.gumbelMaxParam[1], this.numberOfDataPoints);

            // Regression of the ordered data on the Gumbel (maximum order statistic) order statistic medians
            Regression reg = new Regression(this.gumbelMaxOrderMedians, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)reg.suppressErrorMessages();
            reg.linear();

            // Intercept and gradient of best fit straight line
            this.gumbelMaxLine = reg.getBestEstimates();

            // Correlation coefficient
            this.gumbelMaxCorrCoeff = reg.getSampleR();
            
            // Output results to text file
            if(!this.suppressFile){
                    this.outputFile(this.lastMethod, min, reg, this.gumbelMaxNumberOfParameters);
            }

            if(!this.suppressPlot){
                // Initialize data arrays for plotting
                double[][] data = PlotGraph.data(2,this.numberOfDataPoints);

                // Assign data to plotting arrays
                data[0] = this.gumbelMaxOrderMedians;
                data[1] = this.sortedData;

                data[2] = gumbelMaxOrderMedians;
                for(int i=0; i<this.numberOfDataPoints; i++){
                    data[3][i] = this.gumbelMaxLine[0] + this.gumbelMaxLine[1]*gumbelMaxOrderMedians[i];
                }

                // Create instance of PlotGraph
                PlotGraph pg = new PlotGraph(data);
                int[] points = {4, 0};
                pg.setPoint(points);
                int[] lines = {0, 3};
                pg.setLine(lines);
                pg.setXaxisLegend("Gumbel (maximum order statistic) Order Statistic Medians");
                pg.setYaxisLegend("Ordered Data Values");
                pg.setGraphTitle("Gumbel (maximum order statistic) probability plot, " + this.title + ":   gradient = " + Fmath.truncate(this.gumbelMaxLine[1], 4) + ", intercept = "  +  Fmath.truncate(this.gumbelMaxLine[0], 4) + ",  R = " + Fmath.truncate(this.gumbelMaxCorrCoeff, 4));
                pg.setGraphTitle2("  mu = " + Fmath.truncate(this.gumbelMaxParam[0], 4) + ", sigma = "  +  Fmath.truncate(this.gumbelMaxParam[1], 4));

                // Plot
                pg.plot();
                this.probPlotDone = true;
            }

            this.gumbelMaxDone = true;
        }

        // Return Gumbel (maximum order statistic) mu
        public double gumbelMaxMu(){
            if(!this.gumbelMaxDone)this.gumbelMaxProbabilityPlot();
            return this.gumbelMaxParam[0];
        }

        // Return Gumbel (maximum order statistic) mu error
        public double gumbelMaxMuError(){
            if(!this.gumbelMaxDone)this.gumbelMaxProbabilityPlot();
            return this.gumbelMaxParamErrors[0];
        }

        // Return Gumbel (maximum order statistic) sigma
        public double gumbelMaxSigma(){
            if(!this.gumbelMaxDone)this.gumbelMaxProbabilityPlot();
            return this.gumbelMaxParam[1];
        }

        // Return Gumbel (maximum order statistic) sigma error
        public double gumbelMaxSigmaError(){
            if(!this.gumbelMaxDone)this.gumbelMaxProbabilityPlot();
            return this.gumbelMaxParamErrors[1];
        }

        // Return Gumbel (maximum order statistic) order statistic medians
        public double[] gumbelMaxOrderStatisticMedians(){
            if(!this.gumbelMaxDone)this.gumbelMaxProbabilityPlot();
            return this.gumbelMaxOrderMedians;
        }

        // Return the Gumbel (maximum order statistic) gradient
        public double gumbelMaxGradient(){
            if(!this.gumbelMaxDone)this.gumbelMaxProbabilityPlot();
            return this.gumbelMaxLine[1];
        }

        // Return the error of the Gumbel (maximum order statistic) gradient
        public double gumbelMaxGradientError(){
            if(!this.gumbelMaxDone)this.gumbelMaxProbabilityPlot();
            return this.gumbelMaxLineErrors[1];
        }

        // Return the Gumbel (maximum order statistic) intercept
        public double gumbelMaxIntercept(){
            if(!this.gumbelMaxDone)this.gumbelMaxProbabilityPlot();
            return this.gumbelMaxLine[0];
        }

        // Return the error of the Gumbel (maximum order statistic) intercept
        public double gumbelMaxInterceptError(){
            if(!this.gumbelMaxDone)this.gumbelMaxProbabilityPlot();
            return this.gumbelMaxLineErrors[0];
        }

        // Return the Gumbel (maximum order statistic) correlation coefficient
        public double gumbelMaxCorrelationCoefficient(){
            if(!this.gumbelMaxDone)this.gumbelMaxProbabilityPlot();
            return this.gumbelMaxCorrCoeff;
        }

        // Return the sum of squares at the Gumbel (maximum order statistic) minimum
        public double gumbelMaxSumOfSquares(){
            if(!this.gumbelMaxDone)this.gumbelMaxProbabilityPlot();
            return this.gumbelMaxSumOfSquares;
        }
        
        // Return the weighted sum of squares at the Gumbel (maximum order statistic) minimum
        public double gumbelMaxWeightedSumOfSquares(){
            if(!this.gumbelMaxDone)this.gumbelMaxProbabilityPlot();
            return this.gumbelMaxWeightedSumOfSquares;
        }


        // RAYLEIGH PROBABILITY PLOT
        // User supplied initial estimates
        public void rayleighUserSuppliedInitialEstimates(double betae){
            this.userSuppliedEstimates[3] = true;
            this.userInitialEstimates[3] = new double[1];
            this.userInitialEstimates[3][0] = betae;
        }
        
        // Remove user supplied initial estimates
        public void removeRayleighUserSuppliedInitialEstimates(){
            this.userSuppliedEstimates[3] = false;
        }
        
        public void rayleighProbabilityPlot(){
            this.lastMethod = 3;

            // Check for suffient data points
            this.rayleighNumberOfParameters = 1;
            if(this.numberOfDataPoints<3)throw new IllegalArgumentException("There must be at least three data points - preferably considerably more");

            // Create instance of Regression
            Regression min = new Regression(this.sortedData, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)min.suppressErrorMessages();
            
             // Calculate initial estimates
            double[] start = new double[1];
            if(this.userSuppliedEstimates[3]){
                start[0] = this.userInitialEstimates[3][0];
                if(start[0]<0.0)start[0] = this.standardDeviation;
            }
            else{
                start[0] = this.standardDeviation;
            }
            double[] step = {this.stepFactor*start[0]};
            if(step[0]==0.0)step[0] = this.standardDeviation/30.0;
            this.initialEstimates = start;

             // Add constraint; beta>0
            min.addConstraint(0, -1, 0);

            // Create an instance of RayleighProbPlotFunc
            RayleighProbPlotFunc rppf = new RayleighProbPlotFunc();
            rppf.setDataArray(this.numberOfDataPoints);

            // Obtain best probability plot varying beta
            // by minimizing the sum of squares of the differences between the ordered data and the ordered statistic medians
            min.simplex(rppf, start, step, this.tolerance);

             // Get mu and sigma values
            this.rayleighParam = min.getBestEstimates();

            // Get mu and sigma value errors
            this.rayleighParamErrors = min.getBestEstimatesErrors();

            // Get sum of squares
            this.rayleighSumOfSquares = min.getSumOfSquares();
            this.rayleighWeightedSumOfSquares = this.rayleighSumOfSquares;
            if(this.weighted)this.rayleighWeightedSumOfSquares = min.getChiSquare();


            // Calculate Rayleigh order statistic medians (Weibull with mu = 0, sigma = sqrt(2).beta, gamma = 2)
            this.rayleighOrderMedians = Stat.weibullOrderStatisticMedians(0.0, this.rayleighParam[0]*Math.sqrt(2.0), 2.0, this.numberOfDataPoints);

            // Regression of the ordered data on the Rayleigh order statistic medians
            Regression reg = new Regression(this.rayleighOrderMedians, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)reg.suppressErrorMessages();
            reg.linear();

            // Intercept and gradient of best fit straight line
            this.rayleighLine = reg.getBestEstimates();

            // Estimated erors of the intercept and gradient of best fit straight line
            this.rayleighLineErrors = reg.getBestEstimatesErrors();

            // Correlation coefficient
            this.rayleighCorrCoeff = reg.getSampleR();

            // Output results to text file
            if(!this.suppressFile){
                    this.outputFile(this.lastMethod, min, reg, this.rayleighNumberOfParameters);
            }

            if(!this.suppressPlot){
                // Initialize data arrays for plotting
                double[][] data = PlotGraph.data(2,this.numberOfDataPoints);

                // Assign data to plotting arrays
                data[0] = this.rayleighOrderMedians;
                data[1] = this.sortedData;

                data[2] = rayleighOrderMedians;
                for(int i=0; i<this.numberOfDataPoints; i++){
                    data[3][i] = this.rayleighLine[0] + this.rayleighLine[1]*rayleighOrderMedians[i];
                }

                // Create instance of PlotGraph
                PlotGraph pg = new PlotGraph(data);
                int[] points = {4, 0};
                pg.setPoint(points);
                int[] lines = {0, 3};
                pg.setLine(lines);
                pg.setXaxisLegend("Rayleigh Order Statistic Medians");
                pg.setYaxisLegend("Ordered Data Values");
                pg.setGraphTitle("Rayleigh probability plot, " + this.title + ":   gradient = " + Fmath.truncate(this.rayleighLine[1], 4) + ", intercept = "  +  Fmath.truncate(this.rayleighLine[0], 4) + ",  R = " + Fmath.truncate(this.rayleighCorrCoeff, 4));
                pg.setGraphTitle2("  beta = " + Fmath.truncate(this.rayleighParam[0], 4));

                // Plot
                pg.plot();
                this.probPlotDone = true;
            }

            this.rayleighDone = true; 
        }

        // Return Rayleigh beta
        public double rayleighBeta(){
            if(!this.rayleighDone)this.rayleighProbabilityPlot();
            return this.rayleighParam[0];
        }

        // Return Rayleigh beta error
        public double rayleighBetaError(){
            if(!this.rayleighDone)this.rayleighProbabilityPlot();
            return this.rayleighParamErrors[0];
        }

        // Return Rayleigh order statistic medians
        public double[] rayleighOrderStatisticMedians(){
            if(!this.rayleighDone)this.rayleighProbabilityPlot();
            return this.rayleighOrderMedians;
        }

        // Return the Rayleigh gradient
        public double rayleighGradient(){
            if(!this.rayleighDone)this.rayleighProbabilityPlot();
            return this.rayleighLine[1];
        }

        // Return the error of the Rayleigh gradient
        public double rayleighGradientError(){
            if(!this.rayleighDone)this.rayleighProbabilityPlot();
            return this.rayleighLineErrors[1];
        }

        // Return the Rayleigh intercept
        public double rayleighIntercept(){
            if(!this.rayleighDone)this.rayleighProbabilityPlot();
            return this.rayleighLine[0];
        }

        // Return the error of the Rayleigh intercept
        public double rayleighInterceptError(){
            if(!this.rayleighDone)this.rayleighProbabilityPlot();
            return this.rayleighLineErrors[0];
        }

        // Return the Rayleigh correlation coefficient
        public double rayleighCorrelationCoefficient(){
            if(!this.rayleighDone)this.rayleighProbabilityPlot();
            return this.rayleighCorrCoeff;
        }

        // Return the sum of squares at the Rayleigh minimum
        public double rayleighSumOfSquares(){
            if(!this.rayleighDone)this.rayleighProbabilityPlot();
            return this.rayleighSumOfSquares;
        }
        
        // Return the weighted sum of squares at the Rayleigh minimum
        public double rayleighMinWeightedSumOfSquares(){
            if(!this.rayleighDone)this.rayleighProbabilityPlot();
            return this.rayleighWeightedSumOfSquares;
        }
        

        // PARETO PROBABILITY PLOT
        // User supplied initial estimates
        public void paretoUserSuppliedInitialEstimates(double alphae, double betae){
            this.userSuppliedEstimates[4] = true;
            this.userInitialEstimates[4] = new double[2];
            this.userInitialEstimates[4][0] = alphae;
            this.userInitialEstimates[4][1] = betae;
        }
        
        // Remove user supplied initial estimates
        public void removeParetoUserSuppliedInitialEstimates(){
            this.userSuppliedEstimates[4] = false;
        }
        
        public void paretoProbabilityPlot(){
            this.lastMethod = 4;

            // Check for suffient data points
            this.paretoNumberOfParameters = 2;
            if(this.numberOfDataPoints<3)throw new IllegalArgumentException("There must be at least three data points - preferably considerably more");

            // Create instance of Regression
            Regression min = new Regression(this.sortedData, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)min.suppressErrorMessages();
            
             // Calculate initial estimates
            double[] start = new double[2];
            if(this.userSuppliedEstimates[4]){
                start[0] = this.userInitialEstimates[4][0];
                start[1] = this.userInitialEstimates[4][1];
                if(start[1]<this.minimum)start[1] = this.minimum + this.standardDeviation/30.0;
            }
            else{
                start[0] = this.mean/(this.mean - this.minimum);
                start[1] = this.minimum + this.standardDeviation/30.0;
            }
            double[] step = {Math.abs(this.stepFactor*start[0]), Math.abs(this.stepFactor*start[1])};
            if(step[0]==0.0)step[0] = this.standardDeviation/30.0;
            if(step[1]==0.0)step[1] = this.standardDeviation/30.0;          
            this.initialEstimates = start;

            // Create an instance of ParetoProbPlotFunc
            ParetoProbPlotFunc pppf = new ParetoProbPlotFunc();
            pppf.setDataArray(this.numberOfDataPoints);

            // Obtain best probability plot varying alpha and beta
            // by minimizing the sum of squares of the differences between the ordered data and the ordered statistic medians
            min.simplex(pppf, start, step, this.tolerance);

            // Get alpha and beta values
            this.paretoParam = min.getBestEstimates();

            // Get alpha and beta value errors
            this.paretoParamErrors = min.getBestEstimatesErrors();

            // Get sum of squares
            this.paretoSumOfSquares = min.getSumOfSquares();
            this.paretoWeightedSumOfSquares = this.paretoSumOfSquares;
            if(this.weighted)this.paretoWeightedSumOfSquares = min.getChiSquare();


            // Calculate Pareto order statistic medians
            this.paretoOrderMedians = Stat.paretoOrderStatisticMedians(this.paretoParam[0], this.paretoParam[1], this.numberOfDataPoints);

            // Regression of the ordered data on the Pareto order statistic medians
            Regression reg = new Regression(this.paretoOrderMedians, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)reg.suppressErrorMessages();
            reg.linear();

            // Intercept and gradient of best fit straight line
            this.paretoLine = reg.getBestEstimates();

            // Estimated erors of the intercept and gradient of best fit straight line
            this.paretoLineErrors = reg.getBestEstimatesErrors();

            // Correlation coefficient
            this.paretoCorrCoeff = reg.getSampleR();

            // Output results to text file
            if(!this.suppressFile){
                    this.outputFile(this.lastMethod, min, reg, this.paretoNumberOfParameters);
            }

            if(!this.suppressPlot){
                // Initialize data arrays for plotting
                double[][] data = PlotGraph.data(2,this.numberOfDataPoints);

                // Assign data to plotting arrays
                data[0] = this.paretoOrderMedians;
                data[1] = this.sortedData;

                data[2] = paretoOrderMedians;
                for(int i=0; i<this.numberOfDataPoints; i++){
                    data[3][i] = this.paretoLine[0] + this.paretoLine[1]*paretoOrderMedians[i];
                }

                // Create instance of PlotGraph
                PlotGraph pg = new PlotGraph(data);
                int[] points = {4, 0};
                pg.setPoint(points);
                int[] lines = {0, 3};
                pg.setLine(lines);
                pg.setXaxisLegend("Pareto Order Statistic Medians");
                pg.setYaxisLegend("Ordered Data Values");
                pg.setGraphTitle("Pareto probability plot, " + this.title + ":   gradient = " + Fmath.truncate(this.paretoLine[1], 4) + ", intercept = "  +  Fmath.truncate(this.paretoLine[0], 4) + ",  R = " + Fmath.truncate(this.paretoCorrCoeff, 4));
                pg.setGraphTitle2("  alpha = " + Fmath.truncate(this.paretoParam[0], 4) + ", beta = "  +  Fmath.truncate(this.paretoParam[1], 4));

                // Plot
                pg.plot();
                this.probPlotDone = true;
            }

            this.paretoDone = true;
        }

        // Return Pareto alpha
        public double paretoAlpha(){
            if(!this.paretoDone)this.paretoProbabilityPlot();
            return this.paretoParam[0];
        }

        // Return Pareto alpha error
        public double paretoAlphaError(){
            if(!this.paretoDone)this.paretoProbabilityPlot();
            return this.paretoParamErrors[0];
        }

        // Return Pareto beta
        public double paretoBeta(){
            if(!this.paretoDone)this.paretoProbabilityPlot();
            return this.paretoParam[1];
        }

        // Return Pareto beta error
        public double paretoBetaError(){
            if(!this.paretoDone)this.paretoProbabilityPlot();
            return this.paretoParamErrors[1];
        }

        // Return Pareto order statistic medians
        public double[] paretoOrderStatisticMedians(){
            if(!this.paretoDone)this.paretoProbabilityPlot();
            return this.paretoOrderMedians;
        }

        // Return the Pareto gradient
        public double paretoGradient(){
            if(!this.paretoDone)this.paretoProbabilityPlot();
            return this.paretoLine[1];
        }

        // Return the error of the Pareto gradient
        public double paretoGradientError(){
            if(!this.paretoDone)this.paretoProbabilityPlot();
            return this.paretoLineErrors[1];
        }

        // Return the Pareto intercept
        public double paretoIntercept(){
            if(!this.paretoDone)this.paretoProbabilityPlot();
            return this.paretoLine[0];
        }

        // Return the error of the Pareto intercept
        public double paretoInterceptError(){
            if(!this.paretoDone)this.paretoProbabilityPlot();
            return this.paretoLineErrors[0];
        }

        // Return the Pareto correlation coefficient
        public double paretoCorrelationCoefficient(){
            if(!this.paretoDone)this.paretoProbabilityPlot();
            return this.paretoCorrCoeff;
        }

        // Return the sum of squares at the Pareto minimum
        public double paretoSumOfSquares(){
            if(!this.paretoDone)this.paretoProbabilityPlot();
            return this.paretoSumOfSquares;
        }
        
        // Return the weighted sum of squares at the Pareto minimum
        public double paretoWeightedSumOfSquares(){
            if(!this.paretoDone)this.paretoProbabilityPlot();
            return this.paretoWeightedSumOfSquares;
        }


        // F-DISTRIBUTION PROBABILITY PLOT
        public void fDistributionProbabilityPlot(int nu1, int nu2){
            this.lastMethod = 15;
            this.fDistributionNu1 = nu1;  
            this.fDistributionNu2 = nu2;

            // Check for suffient data points
            this.fDistributionNumberOfParameters = 0;
            if(this.numberOfDataPoints<3)throw new IllegalArgumentException("There must be at least three data points - preferably considerably more");

            // Calculate Exponential order statistic medians
            this.fDistributionOrderMedians = Stat.fDistributionOrderStatisticMedians(nu1, nu2, this.numberOfDataPoints);

            // Regression of the ordered data on the F-distribution order statistic medians
            Regression reg = new Regression(this.fDistributionOrderMedians, this.sortedData, this.sortedWeights);
            if(this.suppressErrorMessages)reg.suppressErrorMessages();
            reg.linear();

            // Intercept and gradient of best fit straight line
            this.fDistributionLine = reg.getBestEstimates();

            // Estimated erors of the intercept and gradient of best fit straight line
            this.fDistributionLineErrors = reg.getBestEstimatesErrors();

            // Correlation coefficient
            this.fDistributionCorrCoeff = reg.getSampleR();
            
            // Get sum of squares
            this.fDistributionSumOfSquares = reg.getSumOfSquares();
            this.fDistributionSumOfSquares = reg.getSumOfSquares();
            this.fDistributionWeightedSumOfSquares = this.paretoSumOfSquares;
            if(this.weighted)this.fDistributionWeightedSumOfSquares = reg.getChiSquare();
            
            // Output results to text file
            if(!this.suppressFile){
                this.outputFile(this.lastMethod, null, reg, 0);
            }

            if(!this.suppressPlot){
                // Initialize data arrays for plotting
                double[][] data = PlotGraph.data(2,this.numberOfDataPoints);

                // Assign data to plotting arrays
                data[0] = this.fDistributionOrderMedians;
                data[1] = this.sortedData;

                data[2] = fDistributionOrderMedians;
                for(int i=0; i<this.numberOfDataPoints; i++){
                    data[3][i] = this.fDistributionLine[0] + this.fDistributionLine[1]*fDistributionOrderMedians[i];
                }

                // Create instance of PlotGraph
                PlotGraph pg = new PlotGraph(data);
                int[] points = {4, 0};
                pg.setPoint(points);
                int[] lines = {0, 3};
                pg.setLine(lines);
                pg.setXaxisLegend("F-distribution Order Statistic Medians");
                pg.setYaxisLegend("Ordered Data Values");
                pg.setGraphTitle("F-distribution probability plot, " + this.title + ":   gradient = " + Fmath.truncate(this.fDistributionLine[1], 4) + ", intercept = "  +  Fmath.truncate(this.fDistributionLine[0], 4) + ",  R = " + Fmath.truncate(this.fDistributionCorrCoeff, 4));
                pg.setGraphTitle2("  nu1 = " + nu1 + ", nu2 = " + nu2);

                // Plot
                pg.plot();
                this.probPlotDone = true;
            }

            this.fDistributionDone = true;
        }

        // Return F-distribution order statistic medians
        public double[] fDistributionOrderStatisticMedians(){
            if(!this.fDistributionDone)throw new IllegalArgumentException("F-distribution Probability Plot method has not been called");
            return this.fDistributionOrderMedians;
        }

        // Return the F-distribution gradient
        public double fDistributionGradient(){
            if(!this.fDistributionDone)throw new IllegalArgumentException("F-distribution Probability Plot method has not been called");
            return this.fDistributionLine[1];
        }

        // Return the error of the F-distribution gradient
        public double fDistributionGradientError(){
            if(!this.fDistributionDone)throw new IllegalArgumentException("F-distribution Probability Plot method has not been called");
            return this.fDistributionLineErrors[1];
        }

        // Return the F-distribution intercept
        public double fDistributionIntercept(){
            if(!this.fDistributionDone)throw new IllegalArgumentException("F-distribution Probability Plot method has not been called");
            return this.fDistributionLine[0];
        }

        // Return the error of the F-distribution intercept
        public double fDistributionInterceptError(){
            if(!this.fDistributionDone)throw new IllegalArgumentException("F-distribution Probability Plot method has not been called");
            return this.fDistributionLineErrors[0];
        }

        // Return the F-distribution correlation coefficient
        public double fDistributionCorrelationCoefficient(){
            if(!this.fDistributionDone)throw new IllegalArgumentException("F-distribution Probability Plot method has not been called");
            return this.fDistributionCorrCoeff;
        }

        // Return the sum of squares at the F-distribution minimum
        public double fDistributionSumOfSquares(){
            if(!this.fDistributionDone)throw new IllegalArgumentException("F-distribution Probability Plot method has not been called");
            return this.fDistributionSumOfSquares;
        }

        // Return the weighted sum of squares at the F-distribution minimum
        public double fDistributionWeightedSumOfSquares(){
            if(!this.fDistributionDone)throw new IllegalArgumentException("F-distribution Probability Plot method has not been called");
            return this.fDistributionWeightedSumOfSquares;
        }


        // COMMON METHODS

        // Return the ordered data
        public double[] orderedData(){
            return this.sortedData;
        }

        // Return the number of data points
        public int numberOfDataPoints(){
            return this.numberOfDataPoints;
        }

        // Return the data mean
        public double mean(){
            return this.mean;
        }

        // Return the data standard deviation
        public double standardDeviation(){
            if(!this.probPlotDone)throw new IllegalArgumentException("no probability plot method has been called");
            return this.standardDeviation;
        }

        // Return the data minimum
        public double minimum(){
            return this.minimum;
        }

        // Return the data maximum
        public double maximum(){
             return this.maximum;
        }

        // Return the numerical differentiation step, delta
        public double delta(){
            return this.delta;
        }

        // Reset the numerical differentiation step, delta
        public void resetDelta(double delta){
            this.delta = delta;
        }
        
         // Return the step factor, stepFactor
        public double stepFactor(){
            return this.stepFactor;
        }

        // Reset the step factor, stepFactor
        public void resetStepFactor(double stepFactor){
            this.stepFactor = stepFactor;
        }

        // Set standard deviation denominator to n
        public void setDenominatorToN(){
            this.nFactorOptionI = true;
            this.arrayAsStat.setDenominatorToN();
            this.standardDeviation = arrayAsStat.standardDeviation();
            this.nFactorReset = true;
        }

        // Set standard deviation denominator to n-1
        public void setDenominatorToNminusOne(){
            this.nFactorOptionI = false;
            arrayAsStat.setDenominatorToNminusOne();
            this.standardDeviation = arrayAsStat.standardDeviation();
            this.nFactorReset = true;
        }

        // Return initial estimates used in last call to a probability plot method
        public double[] getInitialEstimates(){
            if(this.lastMethod==-1)throw new IllegalArgumentException("No probability plot method has been called");
            return this.initialEstimates;
        }
        

        public void outputFile(int methodNumber, Regression regMain, Regression regLin, int numberOfParameters){
            
            String plotName = this.methodNames[this.lastMethod];
            String fileTitle = plotName + "ProbabilityPlotOutput" + title + ".txt";
            fileTitle = this.removeSpacesEtc(fileTitle);
            double[] bestLin = regLin.getBestEstimates();
            double[] bestLinErrors = regLin.getBestEstimatesErrors();
            double[] bestMain = null;
            double[] bestMainErrors = null;
            double[][] corrCoeffs = null;
            if(numberOfParameters>0){
                bestMain = regMain.getBestEstimates();
                bestMainErrors = regMain.getBestEstimatesErrors();
                corrCoeffs = regMain.getCorrCoeffMatrix();
            }  
            
            FileOutput fout0 = new FileOutput(fileTitle);
            
            fout0.println(plotName + " Probability Plot");
            fout0.println();
            fout0.println(this.tad.getFullDate() + ", " + this.tad.getShortTime24());
            fout0.println();
            switch(this.weightingOption){
                case 0: fout0.println("Unweighted constrained simplex regression");
                        break;
                case 1: fout0.println("Weighted constrained simplex regression - user supplied weights");
                        break;
                case 2: fout0.println("Weighted constrained simplex regression - weights = responses");
                        break;
                case 3: fout0.println("Weighted constrained simplex regression - weights = square root of the responses");
                        break;  
            }
            fout0.println();
            
            fout0.print("Gradient:  " + Fmath.truncate(bestLin[1], this.trunc), 25);
            fout0.println("s.e. " + Fmath.truncate(bestLinErrors[1], this.trunc));
            fout0.print("Intercept: " + Fmath.truncate(bestLin[0], this.trunc), 25);
            fout0.println("s.e. " + Fmath.truncate(bestLinErrors[0], this.trunc));
            fout0.println();
            
            fout0.println("Correlation Coefficient, r: " + Fmath.truncate(regLin.getSampleR(), this.trunc));
            fout0.println("Critical Value for r:       " + Fmath.truncate(correlationCoefficientCriticalValue(), this.trunc));
            fout0.println("Significance level used:    " + this.significance*100 + "%");
            fout0.println("Sum of squares:             " + Fmath.truncate(regLin.getSumOfSquares(), this.trunc));
            
            fout0.println();
                    
            if(numberOfParameters>0){
                fout0.println("Best Estimates of the Parameters");
                for(int i=0; i<numberOfParameters; i++){
                    fout0.print(this.methodParameters[methodNumber][i] + ": ", 8);
                    fout0.print(Fmath.truncate(bestMain[i], this.trunc), 12);
                    fout0.println("'pseudo-linear' s.e. " + Fmath.truncate(bestMainErrors[i], this.trunc));
                }
                fout0.println();
                fout0.println("The 'pseudo-linear' s.e.s are a lower limit to the s.e. and may significantly underestimate the s.e.");
                fout0.println("See documentation, http://www.ee.ucl.ac.uk/~mflanaga/java/ProbabilityPlot.html, for details");
                fout0.println();
                
                fout0.println("'Pseudo-linear' Parameter-Parameter Correlation Coefficients");
                fout0.print(" ", 8);
                for(int i=0; i<numberOfParameters; i++)fout0.print(this.methodParameters[methodNumber][i], 12);
                fout0.println();
                 for(int i=0; i<numberOfParameters; i++){
                    fout0.print(this.methodParameters[methodNumber][i], 8);
                    for(int j=0; j<numberOfParameters; j++)fout0.print(Fmath.truncate(corrCoeffs[i][j], this.trunc),12);
                    fout0.println();
                 }   
            }
            else{
                if(methodNumber==15){
                    fout0.println("Degrees of freedom, nu1: " + this.fDistributionNu1);
                    fout0.println("Degrees of freedom, nu2: " + this.fDistributionNu2);
                }
            }
            
            fout0.close();
            
            
        }
        
        public String removeSpacesEtc(String name){
            int n = name.length();
            String ret = "";
            for(int i=0; i<n; i++){
                char ch = name.charAt(i);
                if(ch==' ' || ch=='(' ||  ch==')'  ||  ch=='-'){
                    // skip
                }
                else{
                    ret += ch;
                }
            }
            return ret;
        }
        
        // Start of plot title - for use ith Outlier class
        public void setStartOfGraphTitle(String stitle){
            this.titleStart = stitle;
        }
            
            
}



// PROBABILITY PLOT FUNCTIONS
// Gaussian Probability plot function
class GaussProbPlotFunc implements RegressionFunction{

    private int nPoints = 0;
    private int index = 0;
    private double[] medians = null;

    public double function(double[] p, double[] x){

        // Calculate Gaussian order statistic medians
        if(index==0)medians = Stat.gaussianOrderStatisticMedians(p[0], p[1], nPoints);

        // return median value
        double y = medians[index];
        index++;
        if(index==nPoints)index=0;
        return y;
    }

    public void setDataArray(int nPoints){
        this.nPoints = nPoints;
    }
}



// Exponential Probability plot function
class ExponentialProbPlotFunc implements RegressionFunction{

    private int nPoints = 0;
    private int index = 0;
    private double[] medians = null;

    public double function(double[] p, double[] x){

        // Calculate Exponential order statistic medians
        if(index==0)medians = Stat.exponentialOrderStatisticMedians(p[0], p[1], nPoints);

        // return median value
        double y = medians[index];
        index++;
        if(index==nPoints)index=0;
        return y;
    }

    public void setDataArray(int nPoints){
        this.nPoints = nPoints;
    }
}




// Frechet Probability plot function
// Three parameters
class FrechetProbPlotFunc implements RegressionFunction{

    private int nPoints = 0;
    private int index = 0;
    private double[] medians = null;

    public double function(double[] p, double[] x){

        // Calculate Frechet order statistic medians
        if(index==0)medians = Stat.frechetOrderStatisticMedians(p[0], p[1], p[2], nPoints);

        // return median value
        double y = medians[index];
        index++;
        if(index==nPoints)index=0;
        return y;
    }

    public void setDataArray(int nPoints){
        this.nPoints = nPoints;
    }
}

// Frechet Probability plot function
// Two parameters
class FrechetTwoParProbPlotFunc implements RegressionFunction{

    private int nPoints = 0;
    private int index = 0;
    private double[] medians = null;

    public double function(double[] p, double[] x){

        // Calculate Frechet order statistic medians
        if(index==0)medians = Stat.frechetOrderStatisticMedians(p[0], p[1], nPoints);

        // return median value
        double y = medians[index];
        index++;
        if(index==nPoints)index=0;
        return y;
    }

    public void setDataArray(int nPoints){
        this.nPoints = nPoints;
    }
}

// Frechet Probability plot function
// Standard
class FrechetStandardProbPlotFunc implements RegressionFunction{

    private int nPoints = 0;
    private int index = 0;
    private double[] medians = null;

    public double function(double[] p, double[] x){

        // Calculate Frechet order statistic medians
        if(index==0)medians = Stat.frechetOrderStatisticMedians(p[0], nPoints);

        // return median value
        double y = medians[index];
        index++;
        if(index==nPoints)index=0;
        return y;
    }

    public void setDataArray(int nPoints){
        this.nPoints = nPoints;
    }
}


// Gumbel (minimum order statistic) Probability plot function
class GumbelMinProbPlotFunc implements RegressionFunction{

    private int nPoints = 0;
    private int index = 0;
    private double[] medians = null;


    public double function(double[] p, double[] x){

        // Calculate Gumbel order statistic medians
        if(index==0)medians = Stat.gumbelMinOrderStatisticMedians(p[0], p[1], nPoints);

        // return median value
        double y = medians[index];
        index++;
        if(index==nPoints)index=0;
        return y;
    }

    public void setDataArray(int nPoints){
        this.nPoints = nPoints;
    }
}



// Gumbel (maximum order statistic) Probability plot function
class GumbelMaxProbPlotFunc implements RegressionFunction{

    private int nPoints = 0;
    private int index = 0;
    private double[] medians = null;

    public double function(double[] p, double[] x){

         // Calculate Gumbel order statistic medians
        if(index==0)medians = Stat.gumbelMaxOrderStatisticMedians(p[0], p[1], nPoints);

        // return median value
        double y = medians[index];
        index++;
        if(index==nPoints)index=0;
        return y;
    }

    public void setDataArray(int nPoints){
        this.nPoints = nPoints;
    }
}


// Logistic Probability plot function
class LogisticProbPlotFunc implements RegressionFunction{

    private int nPoints = 0;
    private int index = 0;
    private double[] medians = null;

    public double function(double[] p, double[] x){

        // Calculate Logistic order statistic medians
        if(index==0)medians = Stat.logisticOrderStatisticMedians(p[0], p[1], nPoints);

        // return median value
        double y = medians[index];
        index++;
        if(index==nPoints)index=0;
        return y;
    }

    public void setDataArray(int nPoints){
        this.nPoints = nPoints;
    }
}



// Pareto Probability plot function
class ParetoProbPlotFunc implements RegressionFunction{

    private int nPoints = 0;
    private int index = 0;
    private double[] medians = null;

    public double function(double[] p, double[] x){

        // Calculate Pareto order statistic medians
        if(index==0)medians = Stat.paretoOrderStatisticMedians(p[0], p[1], nPoints);

        // return median value
        double y = medians[index];
        index++;
        if(index==nPoints)index=0;
        return y;
    }

    public void setDataArray(int nPoints){
        this.nPoints = nPoints;
    }
}

// Rayleigh Probability plot function
class RayleighProbPlotFunc implements RegressionFunction{

    private int nPoints = 0;
    private int index = 0;
    private double[] medians = null;


    public double function(double[] p, double[] x){

        // Calculate Rayleigh order statistic medians
        if(index==0)medians = Stat.rayleighOrderStatisticMedians(p[0], nPoints);

        // return median value
        double y = medians[index];
        index++;
        if(index==nPoints)index=0;
        return y;
    }

    public void setDataArray(int nPoints){
        this.nPoints = nPoints;
    }
}

// Weibull Probability plot function
// Three parameter
class WeibullProbPlotFunc implements RegressionFunction{

    private int nPoints = 0;
    private int index = 0;
    private double[] medians = null;

    public double function(double[] p, double[] x){

        // Calculate Weibull order statistic medians
        if(index==0)medians = Stat.weibullOrderStatisticMedians(p[0], p[1], p[2], nPoints);

        // return median value
        double y = medians[index];
        index++;
        if(index==nPoints)index=0;
        return y;
    }

    public void setDataArray(int nPoints){
        this.nPoints = nPoints;
    }
}

// Weibull Probability plot function
// Two parameter
class WeibullTwoParProbPlotFunc implements RegressionFunction{

    private int nPoints = 0;
    private int index = 0;
    private double[] medians = null;

    public double function(double[] p, double[] x){

        // Calculate Weibull order statistic medians
        if(index==0)medians = Stat.weibullOrderStatisticMedians(p[0], p[1], nPoints);

        // return median value
        double y = medians[index];
        index++;
        if(index==nPoints)index=0;
        return y;

    }

    public void setDataArray(int nPoints){
        this.nPoints = nPoints;
    }
}

// Weibull Probability plot function
// Standard (one parameter)
class WeibullStandardProbPlotFunc implements RegressionFunction{

    private int nPoints = 0;
    private int index = 0;
    private double[] medians = null;

    public double function(double[] p, double[] x){

        // Calculate Weibull order statistic medians
        if(index==0)medians = Stat.weibullOrderStatisticMedians(p[0], nPoints);

        // return median value
        double y = medians[index];
        index++;
        if(index==nPoints)index=0;
        return y;

    }

    public void setDataArray(int nPoints){
        this.nPoints = nPoints;
    }
    
}

