/*
*   Class   Outliers
*
*   USAGE:  Statistical functions
*
*   WRITTEN BY: Dr Michael Thomas Flanagan
*
*   DATE:    Part from June 2002 as part of Stat
*   SEPARATED: 4 March 2015 Outliers separated out from Stat as a new class
*   AMENDED: 6-30 March 2015
*
*   DOCUMENTATION:
*   See Michael Thomas Flanagan's Java library on-line web page:
*   http://www.ee.ucl.ac.uk/~mflanaga/java/Outlier.html
*   http://www.ee.ucl.ac.uk/~mflanaga/java/
*
*   Copyright (c) 2015 Michael Thomas Flanagan
*
*   PERMISSION TO COPY:
*
* Permission to use, copy and modify this software and its documentation for NON-COMMERCIAL purposes is granted, without fee,
* provided that an acknowledgement to the author, Dr Michael Thomas Flanagan at www.ee.ucl.ac.uk/~mflanaga, appears in all copies333
* and associated documentation or publications.
*
* Public listing of the source codes on the internet is not permitted.
*
* Redistribution of the source codes or of the flanagan.jar file is not permitted.
*
* Redistribution in binary form of all or parts of these classes is not permitted.
*
* Dr Michael Thomas Flanagan makes no representations about the suitability or fitness of the software for any or for a particular purpose.
*
* Dr Michael Thomas Flanagan shall not be liable for any damages suffered as a result of using, modifying or distributing this software
* or its derivatives.
*
***************************************************************************************/

package flanagan.analysis;

import java.util.ArrayList;
import java.math.*;

import flanagan.analysis.Normality;
import flanagan.analysis.ProbabilityPlot;
import flanagan.analysis.Stat;
import flanagan.io.FileChooser;
import flanagan.io.FileOutput;
import flanagan.io.PrintToScreen;
import flanagan.math.ArrayMaths;
import flanagan.math.DeepCopy;
import flanagan.math.Fmath;
import flanagan.math.TimeAndDate;
import flanagan.util.Strings;


public class Outliers{

        private double[] data = null;                                           // data array
        private double[] orderedData = null;                                    // data array sorted into ascending order
        private int[] originalIndices = null;                                   // original indices of sorted data
        private double[] orderStatisticMedians = null;                          // order statistic medians for original data
        private double[] strippedData = null;                                   // original data with any outliers removed
        private double[] strippedOrderStatisticMedians = null;                  // order statistic medians for stripped data
        private int nPoints = 0;                                                // number of data points
        private double meanAll = Double.NaN;                                    // sample mean of data
        private double sdAll = Double.NaN;                                      // sample standard deviation of data  
        
        private double significance = 0.05;                                     // significance level
        
        private int nTietjenMooreSimulations = 10000;                           // number of Tietjen & Moore simulations, instance methods
        private static int nTietjenMooreSimulationsStatic = 10000;              // number of Tietjen & Moore simulations, static methods
        private int nDixonSimulations = 10000;                                  // number of Dixon simulations, instance methods
        private static int nDixonSimulationsStatic = 10000;                     // number of Dixon simulations, static methods
        
        
        private int bluOption = -1;                                             // = -1: no detion method yet called
                                                                                // = 0: lower and upper
                                                                                // = 1: lower
                                                                                // = 2: upper
        private int luOption = -1;                                              // = -1: no detion method yet called
                                                                                // = 0: lower (including result in lower or upper method)
                                                                                // = 1: upper (including result in lower or upper method)
        private boolean pointsIgnored = false;                                  // = true if point/s ignored in Dixon's Q test
        private int nPointsIgnored = 0;                                         // number of points ignored in Dixon's Q test
        private double[] ignoredPoints = null;                                  // points ignored in Dixon's Q test
   
        
        private double[] esdLambdas = null;                                     // ESD lambda array
        private double[] esdRvalues = null;                                     // ESD R values
        private double[] esdMaxValues = null;                                   // data value giving maximum esd on each iteration
        private boolean[] esdTests = null;                                      // ESD test results array
        
        private boolean suppressPrint = false;                                  // if = true printing to file of results suppressed
        private boolean suppressDisplay = false;                                // if = true display of probability plots suppressed
        
        private String filenameout = "OutlierDetectionResults.txt";             // default output text file name
        private String filenamein = null;                                       // name of input file if data read from text file
        private TimeAndDate tad = new TimeAndDate();                            // instance of TimeAndDate
        private String time_date;                                               // time and date of program run
        private int trunc = 4;                                                  // output truncation value
        private int field0 = 30;                                                // field for 1st column in text files
        private int field = 15;                                                 // field for all but the 1st column in text files
        


        // CONSTRUCTORS
                    
        public Outliers(double[] data){
            this.data = data;
            this.initialise();
        }
        
        public Outliers(float[] data){
            ArrayMaths amc = new ArrayMaths(data);
            this.data = amc.array_as_double();
            this.initialise();
        }
        
        public Outliers(long[] data){
            ArrayMaths amc = new ArrayMaths(data);
            this.data = amc.array_as_double();
            this.initialise();
        }
        
        public Outliers(int[] data){
            ArrayMaths amc = new ArrayMaths(data);
            this.data = amc.array_as_double();
            this.initialise();
        }
        
        public Outliers(BigDecimal[] data){
            ArrayMaths amc = new ArrayMaths(data);
            this.data = amc.array_as_double();
            this.initialise();
        }
        
        public Outliers(BigInteger[] data){
            ArrayMaths amc = new ArrayMaths(data);
            this.data = amc.array_as_double();
            this.initialise();
        }
        
        public Outliers(){
            this.data = null;
            this.initialise();
        }
        
        // Initialisation
        private void initialise(){
            if(this.data!=null){
                this.nPoints = this.data.length;
                this.meanAll = Stat.mean(this.data);
                this.sdAll = Stat.standardDeviation(this.data);
            }
            this.time_date = this.tad.getShortTime24() + ", ";
            this.time_date += tad.getDate();
        }
        
        // Read from text file
        public void readDataFromTextFile(){
            
            FileChooser fin0 = new FileChooser();
            this.filenamein = fin0.selectFile();
            ArrayList<Double> alS = new ArrayList<Double>();
            int nLines = fin0.numberOfLines();
            for(int i=0; i<nLines; i++){
                String line = fin0.readLine();
                line = this.replacements(line);
                String[] tokens = Strings.tokens(line);
                int nTokens = tokens.length;
                for(int j=0; j<nTokens; j++){
                    double holdD = Double.parseDouble(tokens[j]);
                    alS.add(holdD);
                }
            }
            fin0.close();
            
            this.nPoints = alS.size();
            this.data = new double[this.nPoints];
            for(int i=0; i<this.nPoints; i++)this.data[i] = alS.get(i);   
            this.meanAll = Stat.mean(this.data);
            this.sdAll = Stat.standardDeviation(this.data);
        }
        
        // Line replacements
        public String replacements(String line){
            int nChar = line.length();
            char[] cline = new char[nChar];
            String rline = "";
            for(int i=0; i<nChar; i++){
                cline[i] = line.charAt(i);
                if(cline[i]==',')cline[i] = ' ';
                if(cline[i]==';')cline[i] = ' ';
                if(cline[i]==':')cline[i] = ' ';
                if(cline[i]=='\t')cline[i] = ' ';   
                rline += cline[i];
            } 
            return rline;
        }
        
        // Reset the significance level
        public void resetSignificanceLevel(double significance){
            this.significance = significance;
        }
        
        // Get the significance level
        public double getSignificanceLevel(){
            return this.significance;
        }
      
        // Set number of points
        // For internal use
        private void setNpoints(int n){
            this.nPoints = n;
        }
        
        // Get the number of points
        public int getNpoints(){
            return this.nPoints;
        }
        
        // Suppress print to file in instance methods#
        public void suppressPrint(){
            this.suppressPrint = true;
        }
        
        // Restore print to file in instance methods#
        public void restorePrint(){
            this.suppressPrint = false;
        }
         
        // Reset text file name
        public void resetTextFileName(String filenameout){
            this.filenameout = filenameout;
            int pos0 = filenameout.indexOf(".");
            if(pos0==-1)filenameout += ".txt";
        }
        
        // Suppress display of probability plots
        public void suppressDisplay(){
            this.suppressDisplay = true;
        }
        
        // Restore display of probability plots
        public void restoreDisplay(){
            this.suppressDisplay = false;
        }
               
        // Order data
        private ArrayList<Object> orderData(double[] datah, int option){
            ArrayList<Object> ret = null;
            switch(option){
                case 0: ret = orderAbsoluteResiduals(datah);
                        break;
                case 1: ret = orderDescending(datah);
                        break;
                case 2: ret = orderAscending(datah);
                        break;
            }
            return ret;
        }
        
        // Order absolute values of the residuals
        public ArrayList<Object> orderAbsoluteResiduals(double[] datah){
            int np = datah.length;
            double[] dataw = DeepCopy.copy(datah);
            double mean = Stat.mean(dataw);
            ArrayMaths alh = new ArrayMaths(dataw);
            alh = alh.minus(mean);
            alh = alh.abs();
            alh = alh.sort();
            int[] indices = alh.originalIndices();
            double[] retd = new double[np];
            for(int i=0; i<np; i++){
                retd[i] = datah[indices[i]];
            }
            ArrayList<Object> ret = new ArrayList<Object>();
            ret.add(retd);
            ret.add(indices);
            return ret;   
        }
        
                
        // Order data in descending order
        public ArrayList<Object> orderDescending(double[] datah){
            ArrayMaths alh = new ArrayMaths(datah);
            alh = alh.sort();
            double[] asc = alh.array(); 
            int[] ind = alh.originalIndices();
            double[] holdD = DeepCopy.copy(asc);
            int[] holdI = DeepCopy.copy(ind);
            int n = datah.length;
            for(int i=0; i<n; i++){
                asc[i] = holdD[n-1-i];
                ind[i] = holdI[n-1-i];
            }
            ArrayList<Object> ret = new ArrayList<>();
            ret.add(asc);
            ret.add(ind);
            return ret;   
        }
      
        // Order data in ascending order
        public ArrayList<Object> orderAscending(double[] datah){
            ArrayMaths alh = new ArrayMaths(datah);
            alh = alh.sort();
            double[] asc = alh.array(); 
            int[] ind = alh.originalIndices();
            ArrayList<Object> ret = new ArrayList<>();
            ret.add(asc);
            ret.add(ind);
            return ret;   
        }
        
        // DATA
        // Original data as double[]
        public double[] getOriginalData(){
            return this.data;
        }
        
        // Ordered original data as double[]
        public double[] getOrderedOriginalData(){
            ArrayMaths alO = new ArrayMaths(this.data);
            alO.sort();
            return alO.array();
        }
        
        //  Original Data Gaussian Order Statistic Medians
        public double[] getDataOrderStatisticMedians(){
            return this.orderStatisticMedians;
        }
        
        // Data stripped of outlier/s
        public double[] getStrippedData(){
            return this.strippedData;
        }
   
        // Data stripped of outlier/s
        public double[] getOrderedStrippedData(){
            ArrayMaths alO = new ArrayMaths(this.strippedData);
            alO.sort();
            return alO.array();
        }
        
        //  Stripped Data Gaussian Order Statistic Medians
        public double[] getStrippedDataOrderStatisticMedians(){
            return this.strippedOrderStatisticMedians;
        }

   
        
        // GRUBBS' TEST
        
        // Grubbs' critical test criterion, T; single tailed
        // Static method
        public static double getGrubbsOneTailedCriticalT(double significance, int nObservations){
            return Outliers .getGrubbsCriticalT(significance, nObservations, 1);
        }
  
        public static double getGrubbsOneSidedCriticalT(double significance, int nObservations){
            return Outliers .getGrubbsCriticalT(significance, nObservations, 1);
        }
        
        // Grubbs' critical test criterion, T; single tailed
        // instance method
         public double getGrubbsOneTailedCriticalT(){
            return Outliers .getGrubbsCriticalT(this.significance, this.nPoints, 1);
        }
         
        public double getGrubbsOneSidedCriticalT(){
            return Outliers .getGrubbsCriticalT(this.significance, this.nPoints, 1);
        }
       
        // Grubbs' critical test criterion, T; double tailed
        public static double getGrubbsTwoTailedCriticalT(double significance, int nObservations){
            return Outliers .getGrubbsCriticalT(significance, nObservations, 2);
        }
         
        public static double getGrubbsTwoSidedCriticalT(double significance, int nObservations){
            return Outliers .getGrubbsCriticalT(significance, nObservations, 2);
        }
               
        // Grubbs' critical test criterion, T; double tailed
        // instance method
        public double getGrubbsTwoTailedCriticalT(){
            return Outliers .getGrubbsCriticalT(this.significance, this.nPoints, 2);
        }
        
        public double getGrubbsTwoSidedCriticalT(){
            return Outliers .getGrubbsCriticalT(this.significance, this.nPoints, 2);
        }
        
        // Grubbs' critical test criterion, T, 
        private static double getGrubbsCriticalT(double significance, int nObservations, int tailChoice){
      
            // get student's t value
            double tv =  Stat.studentstValue(significance/(nObservations*tailChoice), nObservations-2);
            double tv2 = tv*tv;
        
            return (nObservations - 1)*Math.sqrt(((tv2/(nObservations - 2 + tv2))/nObservations)); 
        }
        
        // Grubbs' test for an upper or lower outlier
        // static method
        public ArrayList<Object> outlierGrubbs(double[] data){
            Outliers outl0 = new Outliers(data);
            return outl0.outlierGrubbs();
        }
       
            
        // Grubbs' test for an upper or lower outlier
        // instance method
        public ArrayList<Object> outlierGrubbs(){
                  
            // Arrange data
            Stat.denominatorSwap();
            this.bluOption = 0;
            
            ArrayMaths am = new ArrayMaths(this.data);
            am = am.minus(this.meanAll);
            am = am.abs();
            am = am.sort();
            int[] indices = am.originalIndices();
            this.orderedData = am.getArray_as_double();
            double possibleOutlier = data[indices[this.nPoints-1]];
            
            // Get Grubbs T value
            double grubbsT =  getGrubbsTwoTailedCriticalT(this.significance, this.nPoints);

            // Calculate Grubb's G value
            double grubbsG = (this.orderedData[this.nPoints-1])/this.sdAll;
            
            Stat.denominatorUnswap();
            return this.outlierGrubbsCore(possibleOutlier, grubbsT, grubbsG);
        }
        
        // Grubbs' test for an upper outlier
        // static method
        public static ArrayList<Object> upperOutlierGrubbs(double[] data){
            Outliers outl0 = new Outliers(data);
            return outl0.upperOutlierGrubbs();
        }
        
        // Grubbs' test for a upper outlier
        // instance method
        public ArrayList<Object> upperOutlierGrubbs(){
                  
            // Arrange data
            Stat.denominatorSwap();
            this.bluOption = 2;
            
            ArrayMaths am = new ArrayMaths(this.data);
            am = am.sort();
            this.orderedData = am.getArray_as_double();
            double possibleOutlier = orderedData[this.nPoints-1];
        
            // Get Grubbs T value
            double grubbsT =  getGrubbsOneTailedCriticalT(significance, this.nPoints);

            // Calculate Grubb's G value
            double grubbsG = (possibleOutlier - this.meanAll)/this.sdAll;
            
            Stat.denominatorUnswap();
            return this.outlierGrubbsCore(possibleOutlier, grubbsT, grubbsG);
            
        }
        
        // Grubbs' test for an lower outlier
        // static method
        public static ArrayList<Object> lowerOutlierGrubbs(double[] data){
            Outliers outl0 = new Outliers(data);
            return outl0.lowerOutlierGrubbs();
        }
        
        // Grubbs' test for a lower outlier
        // instance method
        public ArrayList<Object> lowerOutlierGrubbs(){
                  
            // Arrange data
            Stat.denominatorSwap();
            this.bluOption = 1;
            
            ArrayMaths am = new ArrayMaths(data);
            am = am.sort();
            this.orderedData = am.getArray_as_double();
            double possibleOutlier = this.orderedData[0];
        
            // Get Grubbs T value
            double grubbsT =  getGrubbsOneTailedCriticalT(this.significance, this.nPoints);

            // Calculate Grubb's G value
            double grubbsG = (this.meanAll - possibleOutlier)/this.sdAll;
            
            Stat.denominatorUnswap();
            return this.outlierGrubbsCore(possibleOutlier, grubbsT, grubbsG);
        }
        
            
        private ArrayList<Object> outlierGrubbsCore(double possibleOutlier, double grubbsT,  double grubbsG){
            
            ArrayList<Object>  ret = new ArrayList<Object>();
            
            double outlier = Double.NaN;
            int index = -1;
            boolean found = false;
            if(grubbsG>grubbsT){
                found = true;
                outlier = possibleOutlier;
                
                // identify outlier index
                ArrayList<Double> arrayout = new ArrayList<Double>();
                for(int i=0; i<this.nPoints; i++){
                    if(this.data[i]==possibleOutlier){
                        index = i;
                    }else{
                        arrayout.add(data[i]);
                    }
                }
                int naout = arrayout.size();
                this.strippedData = new double[naout];
                for(int i=0; i<naout; i++)this.strippedData[i] = arrayout.get(i); 
                ret.add(found);
                ret.add(outlier);
                ret.add(index);
                ret.add(this.strippedData);
            }
            else{
                ret.add(found);
                ret.add(outlier);
                ret.add(index);   
                ret.add(this.data);
            }
            ret.add(grubbsG);
            ret.add(grubbsT);
            ret.add(significance);
            ret.add(nPoints);
            
            ArrayList<Double> alD = this.probabilityPlots(found, this.data, this.strippedData);
            ret.add(alD.get(0));
            ret.add(alD.get(1));
            ret.add(alD.get(2));
            ret.add(alD.get(3));
            
            if(found){
                ret.add(alD.get(4));
                ret.add(alD.get(5));
                ret.add(alD.get(6));
                ret.add(alD.get(7));
            }
            else{
                ret.add(Double.NaN);
                ret.add(Double.NaN);
                ret.add(Double.NaN);
                ret.add(Double.NaN);
            }

            ArrayList<Object> alswA = this.shapiroWilkTest(this.data);
            double wValueA = (Double)alswA.get(0);
            double wCritA = (Double)alswA.get(1);
            double pValueA = (Double)alswA.get(2);
            String swWtestA = (String)alswA.get(3);
            double wValueB = Double.NaN;
            double wCritB = Double.NaN;
            double pValueB = Double.NaN;
            String swWtestB = null;
            if(found){
                ArrayList<Object> alswB = this.shapiroWilkTest(this.strippedData);
                wValueB = (Double)alswB.get(0);
                wCritB = (Double)alswB.get(1);
                pValueB = (Double)alswB.get(2);
                swWtestB = (String)alswB.get(3);
            }
            
            ret.add(wValueA);
            ret.add(wCritA);
            ret.add(pValueA);
            if(found){
                ret.add(wValueB);
                ret.add(wCritB);
                ret.add(pValueB);
            }
            else{
                ret.add(Double.NaN);
                ret.add(Double.NaN);
                ret.add(Double.NaN);
            }
            
            // Print
            if(!this.suppressPrint){
                FileOutput fout0 = new FileOutput(this.filenameout);
                fout0.println("Outlier Detection Results");
                fout0.println("Outlier Detection Method: Grubbs' Test");
                switch(this.bluOption){
                    case 0: fout0.println("Lower or upper outlier");
                            break;
                    case 1: fout0.println("Lower outlier");
                            break;
                    case 2: fout0.println("Upper outlier");
                            break;
                }
                if(this.filenamein!=null)fout0.println("Input file: " + this.filenamein);
                
                fout0.println();
                fout0.println("File name: " + filenameout);
                fout0.println("Run date: " + this.time_date);
                fout0.println();
                if(found){
                    fout0.println("Extreme outlier indicated");
                }
                else{
                     fout0.println("No outlier indicated");
                }
                fout0.println();
                fout0.println("Significance level: " + this.significance + " [" + this.significance*100 + "%]");
                fout0.println("Test Statistic, G:  " +  Fmath.truncate(grubbsG, this.trunc));
                fout0.println("Critical value, T:  " +  Fmath.truncate(grubbsT, this.trunc));   
             
                fout0.println();
                if(found){
                    fout0.print("Outlier: ");
                    fout0.println(Fmath.truncate(outlier, trunc));           
                    fout0.print("Outlier index in the original inputted data [indices start at 0]: ");
                    fout0.println(index);
                    fout0.println();
                }     

                fout0.print(" ", this.field0);
                fout0.print("Original data ", this.field);
                if(found)fout0.println("Data with outlier removed");
                else fout0.println();
                fout0.print("Number of data points: ", this.field0);
                fout0.print(this.nPoints, this.field);
                if(found)fout0.println((this.nPoints - 1));
                else fout0.println();
                fout0.print("Sample mean: ", this.field0);
                fout0.print(Fmath.truncate(Stat.mean(this.data), trunc), this.field);
                if(found)fout0.println(Fmath.truncate(Stat.mean(this.strippedData), trunc));
                else fout0.println();
                fout0.print("Sample standard deviation: ", this.field0);
                fout0.print(Fmath.truncate(Stat.standardDeviation(this.data), trunc), this.field);
                if(found)fout0.println(Fmath.truncate(Stat.standardDeviation(this.strippedData), trunc));
                else fout0.println();
                fout0.print("Maximum: ", this.field0);
                fout0.print(Fmath.truncate(Fmath.maximum(this.data), trunc), this.field);
                if(found)fout0.println(Fmath.truncate(Fmath.maximum(this.strippedData), trunc));
                else fout0.println();
                fout0.print("Minimum: ", this.field0);
                fout0.print(Fmath.truncate(Fmath.minimum(this.data), trunc), this.field);
                if(found)fout0.println(Fmath.truncate(Fmath.minimum(this.strippedData), trunc));
                else fout0.println();
                
                fout0.println();
                fout0.println("Shapiro-Wilk Test:");
                fout0.print(" Shapiro-Wilk W value: ", this.field0);
                fout0.print(Fmath.truncate(wValueA, trunc), this.field);
                if(found){
                    fout0.println(Fmath.truncate(wValueB, trunc));
                }
                else fout0.println(); 
                fout0.println(" Critical value for W: ");
                fout0.print("  (" + this.significance*100 + "% Significance level)", this.field0);
                fout0.print(Fmath.truncate(wCritA, trunc), this.field);
                if(found){
                    fout0.println(Fmath.truncate(wCritB, trunc));
                }
                else fout0.println(); 
                fout0.print(" Shapiro-Wilk P value: ", this.field0);
                fout0.print(Fmath.truncate(pValueA, trunc), this.field);
                if(found){
                    fout0.println(Fmath.truncate(pValueB, trunc));
                }
                else fout0.println(); 
                fout0.print(" Data possibly Gaussian: ", this.field0);
                fout0.print(swWtestA, this.field);
                if(found){
                    fout0.println(swWtestB);
                }
                else fout0.println(); 
                fout0.println();
                
                fout0.println("Probability plot:");
                fout0.print(" Correlation coefficient, r: ", this.field0);
                double corrCoeffA = alD.get(2);
                double corrCoeffB = Double.NaN;
                fout0.print(Fmath.truncate(corrCoeffA, trunc), this.field);
                if(found){
                    corrCoeffB = alD.get(6);
                    fout0.println(Fmath.truncate(corrCoeffB, trunc));
                }
                else fout0.println(); 
                fout0.println(" Critical value for r: ");
                fout0.print("  (" + this.significance*100 + "% Significance level)", this.field0);
                double critValueA = alD.get(3);
                double critValueB = Double.NaN;
                fout0.print(Fmath.truncate(critValueA, trunc), this.field);
                if(found){
                    critValueB = alD.get(7);
                    fout0.println(Fmath.truncate(critValueB, trunc));
                }
                else fout0.println(); 
                fout0.print(" Data possibly Gaussian: ", this.field0);
                String gTest = "Rejected";
                if(corrCoeffA>=critValueA)gTest = "Accepted";
                fout0.print(gTest, this.field);
                if(found){
                    gTest = "Rejected";
                    if(corrCoeffB>=critValueB)gTest = "Accepted";
                    fout0.println(gTest);
                }
                else fout0.println(); 
                fout0.print(" Gradient: ", this.field0);
                fout0.print(Fmath.truncate(alD.get(0), trunc), this.field);
                if(found)fout0.println(Fmath.truncate(alD.get(4), trunc));
                else fout0.println();
                fout0.print(" Intercept: ", this.field0);
                fout0.print(Fmath.truncate(alD.get(1), trunc), this.field);
                if(found)fout0.println(Fmath.truncate(alD.get(5), trunc));
                else fout0.println();
               
           
                fout0.println();
                
                fout0.println("Input data: ");
                int kk = 0;
                for(int i=0; i<this.nPoints; i++){
                    fout0.print(this.data[i] + "  ");
                    kk++;
                    if(kk==10){
                        fout0.println();
                        kk = 0;
                    }
                }
                fout0.close();
            }
            return ret;
        }
        
        
        // TIETJEN AND MOORE TEST
        
        // Number of Tietjen and Moore's simulations
        // Reset number of Tietjen and Moore's simulations
        public void resetNumberTietjenMooreSimulations(int n){
            this.nTietjenMooreSimulations = n;
        } 
        
        // Get number of Tietjen and Moore's simulations
        public int getNumberTietjenMooreSimulations(){
            return this.nTietjenMooreSimulations;
        } 
         
        // Tietjen and Moore's test for lower or upper outliers
        // instance method
        public  ArrayList<Object> outliersTietjenMoore(int nPossibleOutliers){
            return  outliersTietjenMooreCore(nPossibleOutliers, 0);
        }
        
        // Tietjen and Moore's test for lower or upper outliers
        // static method
        public  static ArrayList<Object> outliersTietjenMoore(double[] data, int nPossibleOutliers){
            Outliers outl0 = new Outliers(data);  
            return outl0.outliersTietjenMoore(nPossibleOutliers);
        }
        
        // Tietjen and Moore's test for upper outliers
        // instance method
        public  ArrayList<Object> upperOutliersTietjenMoore(int nPossibleOutliers){
            return outliersTietjenMooreCore(nPossibleOutliers, 2);
        }
        
        // Tietjen and Moore's test for upper outliers
        // static method
        public  static ArrayList<Object> upperOutliersTietjenMoore(double[] data, int nPossibleOutliers){
            Outliers outl0 = new Outliers(data);  
            return outl0.upperOutliersTietjenMoore(nPossibleOutliers);
        }
        
        // Tietjen and Moore's test for lower outliers
        // instance method
        public  ArrayList<Object> lowerOutliersTietjenMoore(int nPossibleOutliers){
            return outliersTietjenMooreCore(nPossibleOutliers, 1);
        }        
              
        // Tietjen and Moore's test for lower outliers
        // static method
        public  static ArrayList<Object> lowerOutliersTietjenMoore(double[] data, int nPossibleOutliers){
            Outliers outl0 = new Outliers(data);  
            return outl0.lowerOutliersTietjenMoore(nPossibleOutliers);
        }
        
        // Core method for Tietjen and Moore Test
        private  ArrayList<Object> outliersTietjenMooreCore(int nPossibleOutliers, int option){
                  
            // Arrange data
            ArrayList<Object> ret = new ArrayList<Object>();
            Stat.denominatorSwap();
            
            // Calculate Tietjen and Moore's L value
            ArrayList<Object> retal = this.orderData(this.data, option);
            this.orderedData = (double[])retal.get(0);
            this.originalIndices = (int[])retal.get(1);
            double[] possibleOutliers = new double[nPossibleOutliers];
            int[] possibleOutlierIndices = new int[nPossibleOutliers];
            for(int i=0; i<nPossibleOutliers; i++){
                possibleOutliers[i] = this.orderedData[this.nPoints - nPossibleOutliers + i];
                possibleOutlierIndices[i] = this.originalIndices[this.nPoints - nPossibleOutliers + i];
            }
            double tietjenMooreLvalue = this.getTietjenMooreLvalue(this.orderedData, this.nPoints, nPossibleOutliers);
            
            //  Simulations
            double[] simulLs = new double[this.nTietjenMooreSimulations];
            double[] orderedRandom = null;
            double nAbove = 0;
            for(int i=0; i<this.nTietjenMooreSimulations; i++){
                double[] random = Stat.gaussianRand(0.0, 1.0, this.nPoints); 
                ArrayList retal2 = this.orderData(random, 0);
                orderedRandom = DeepCopy.copy((double[])retal2.get(0));
                simulLs[i] = this.getTietjenMooreLvalue(orderedRandom, this.nPoints, nPossibleOutliers);
                if(simulLs[i]>=tietjenMooreLvalue)nAbove++;    
            }
            double cdf  = nAbove/this.nTietjenMooreSimulations;
            double prob = 1.0 - cdf;
            ArrayMaths am1 = new ArrayMaths(simulLs);
            am1 = am1.sort();
            simulLs = am1.array();
            int iCritical = (int)(this.significance*this.nTietjenMooreSimulations);
            double tietjenMooreCriticalLvalue = simulLs[iCritical];
            boolean found = false;
            if(tietjenMooreLvalue<tietjenMooreCriticalLvalue)found = true;
            ret.add(found);
            ret.add(possibleOutliers);
            ret.add(possibleOutlierIndices);
            
            if(found){
                this.strippedData = new double[this.nPoints - nPossibleOutliers];
                int k = 0;
                for(int i=0; i<this.nPoints; i++){
                    boolean test = true;
                    for(int j=0; j<nPossibleOutliers; j++)if(i==possibleOutlierIndices[j])test = false;
                    if(test){
                        this.strippedData[k] = this.data[i];
                        k++;  
                    }
                }
                ret.add(this.strippedData);
            }
            else{
                ret.add(this.data);
            }
            ret.add(tietjenMooreLvalue);
            ret.add(tietjenMooreCriticalLvalue);
            ret.add(this.significance);
            ret.add(prob);
            ret.add(this.nPoints);
            ret.add(nPossibleOutliers);

            ArrayList<Double> alD = this.probabilityPlots(found, this.data, this.strippedData);
            ret.add(alD.get(0));
            ret.add(alD.get(1));
            ret.add(alD.get(2));
            ret.add(alD.get(3));
            
            if(found){
                ret.add(alD.get(4));
                ret.add(alD.get(5));
                ret.add(alD.get(6));
                ret.add(alD.get(7));
            }
            else{
                ret.add(Double.NaN);
                ret.add(Double.NaN);
                ret.add(Double.NaN);
                ret.add(Double.NaN);
            }
           
            ArrayList<Object> alswA = this.shapiroWilkTest(this.data);
            double wValueA = (Double)alswA.get(0);
            double wCritA = (Double)alswA.get(1);
            double pValueA = (Double)alswA.get(2);
            String swWtestA = (String)alswA.get(3);
            double wValueB = Double.NaN;
            double wCritB = Double.NaN;
            double pValueB = Double.NaN;
            String swWtestB = null;
            if(found){
                ArrayList<Object> alswB = this.shapiroWilkTest(this.strippedData);
                wValueB = (Double)alswB.get(0);
                wCritB = (Double)alswB.get(1);
                pValueB = (Double)alswB.get(2);
                swWtestB = (String)alswB.get(3);
            }
            
            ret.add(wValueA);
            ret.add(wCritA);
            ret.add(pValueA);
            if(found){
                ret.add(wValueB);
                ret.add(wCritB);
                ret.add(pValueB);
            }
            else{
                ret.add(Double.NaN);
                ret.add(Double.NaN);
                ret.add(Double.NaN);
            }
            
            // Print
            if(!this.suppressPrint){
                FileOutput fout0 = new FileOutput(this.filenameout);
                fout0.println("Outlier Detection Results");
                fout0.println("Outlier Detection Method: Tietjen-Moore Test");
                switch(option){
                    case 0: fout0.println("Lower and upper outliers");
                            break;
                    case 1: fout0.println("Lower outliers");
                            break;
                    case 2: fout0.println("Upper outliers");
                            break;
                }
                if(this.filenamein!=null)fout0.println("Input file: " + this.filenamein);
                
                fout0.println();
                fout0.println("File name: " + filenameout);
                fout0.println("Run date: " + this.time_date);
                fout0.println();
                fout0.println("Number of outliers checked: " + nPossibleOutliers);
                if(found){
                    fout0.println(nPossibleOutliers + " outliers indicated");
                }
                else{
                     fout0.println("No outliers indicated");
                }
                fout0.println();
                fout0.println("Significance level:    " + this.significance + " [" + this.significance*100 + "%]");
                fout0.println("Test Statistic, Lk:    " +  Fmath.truncate(tietjenMooreLvalue, this.trunc));
                fout0.println("Critical value, Lcrit: " +  Fmath.truncate(tietjenMooreCriticalLvalue, this.trunc));
                fout0.println("Significance level at which Lk = Lcrit:  " +  Fmath.truncate(prob, this.trunc) + " [" + Fmath.truncate(prob*100, 8) + "%]");
                
             
                fout0.println();
                if(found){
                    fout0.println("Outliers: ");
                    for(int i=0; i<nPossibleOutliers; i++)fout0.print(Fmath.truncate(possibleOutliers[i], trunc) + "  ");
                    fout0.println();             
                    fout0.println("Outlier indices in the original inputted data [indices start at 0]: ");
                    for(int i=0; i<nPossibleOutliers; i++)fout0.print(possibleOutlierIndices[i] + "  ");
                    fout0.println();
                }     
                   
                fout0.print(" ", this.field0);
                fout0.print("Original data ", this.field);
                if(found)fout0.println("Data with outlier/s removed");
                else fout0.println();
                fout0.print("Number of data points: ", this.field0);
                fout0.print(this.nPoints, this.field);
                if(found)fout0.println((this.nPoints - nPossibleOutliers));
                else fout0.println();
                fout0.print("Sample mean: ", this.field0);
                fout0.print(Fmath.truncate(Stat.mean(this.data), trunc), this.field);
                if(found)fout0.println(Fmath.truncate(Stat.mean(this.strippedData), trunc));
                else fout0.println();
                fout0.print("Sample standard deviation: ", this.field0);
                fout0.print(Fmath.truncate(Stat.standardDeviation(this.data), trunc), this.field);
                if(found)fout0.println(Fmath.truncate(Stat.standardDeviation(this.strippedData), trunc));
                else fout0.println();
                fout0.print("Maximum: ", this.field0);
                fout0.print(Fmath.truncate(Fmath.maximum(this.data), trunc), this.field);
                if(found)fout0.println(Fmath.truncate(Fmath.maximum(this.strippedData), trunc));
                else fout0.println();
                fout0.print("Minimum: ", this.field0);
                fout0.print(Fmath.truncate(Fmath.minimum(this.data), trunc), this.field);
                if(found)fout0.println(Fmath.truncate(Fmath.minimum(this.strippedData), trunc));
                else fout0.println();
                
                fout0.println();
                fout0.println("Shapiro-Wilk Test:");
                fout0.print(" Shapiro-Wilk W value: ", this.field0);
                fout0.print(Fmath.truncate(wValueA, trunc), this.field);
                if(found){
                    fout0.println(Fmath.truncate(wValueB, trunc));
                }
                else fout0.println(); 
                fout0.println(" Critical value for W: ");
                fout0.print("  (" + this.significance*100 + "% Significance level)", this.field0);
                fout0.print(Fmath.truncate(wCritA, trunc), this.field);
                if(found){
                    fout0.println(Fmath.truncate(wCritB, trunc));
                }
                else fout0.println(); 
                fout0.print(" Shapiro-Wilk P value: ", this.field0);
                fout0.print(Fmath.truncate(pValueA, trunc), this.field);
                if(found){
                    fout0.println(Fmath.truncate(pValueB, trunc));
                }
                else fout0.println(); 
                fout0.print(" Data possibly Gaussian: ", this.field0);
                fout0.print(swWtestA, this.field);
                if(found){
                    fout0.println(swWtestB);
                }
                else fout0.println(); 
                fout0.println();
                
                fout0.println("Probability plot:");
                fout0.print(" Correlation coefficient, r: ", this.field0);
                double corrCoeffA = alD.get(2);
                double corrCoeffB = Double.NaN;
                fout0.print(Fmath.truncate(corrCoeffA, trunc), this.field);
                if(found){
                    corrCoeffB = alD.get(6);
                    fout0.println(Fmath.truncate(corrCoeffB, trunc));
                }
                else fout0.println(); 
                fout0.println(" Critical value for r: ");
                fout0.print("  (" + this.significance*100 + "% Significance level)", this.field0);
                double critValueA = alD.get(3);
                double critValueB = Double.NaN;
                fout0.print(Fmath.truncate(critValueA, trunc), this.field);
                if(found){
                    critValueB = alD.get(7);
                    fout0.println(Fmath.truncate(critValueB, trunc));
                }
                else fout0.println(); 
                fout0.print(" Data possibly Gaussian: ", this.field0);
                String gTest = "Rejected";
                if(corrCoeffA>=critValueA)gTest = "Accepted";
                fout0.print(gTest, this.field);
                if(found){
                    gTest = "Rejected";
                    if(corrCoeffB>=critValueB)gTest = "Accepted";
                    fout0.println(gTest);
                }
                else fout0.println(); 
                fout0.print(" Gradient: ", this.field0);
                fout0.print(Fmath.truncate(alD.get(0), trunc), this.field);
                if(found)fout0.println(Fmath.truncate(alD.get(4), trunc));
                else fout0.println();
                fout0.print(" Intercept: ", this.field0);
                fout0.print(Fmath.truncate(alD.get(1), trunc), this.field);
                if(found)fout0.println(Fmath.truncate(alD.get(5), trunc));
                else fout0.println();
                

            
                fout0.println();
                fout0.println("Input data: ");
                int kk = 0;
                for(int i=0; i<this.nPoints; i++){
                    fout0.print(this.data[i] + "  ");
                    kk++;
                    if(kk==10){
                        fout0.println();
                        kk = 0;
                    }
                }
                fout0.close();
            }
            
            Stat.denominatorUnswap(); 
            return ret;
        }
        
                
        // Tietjen and Moore's Critical value for lower or upper outliers
        // instance method
        public double getTietjenMooreCriticalL(int nPossibleOutliers){
            return getTietjenMooreCriticalLcore(nPossibleOutliers, 0);
        }
        
        // Tietjen and Moore's Critical value for lower or upper outliers
        // static method
        
        public static double getTietjenMooreCriticalL(int nPossibleOutliers, int nPoints, int nSimulations, double significance){ 
            Outliers outl0 = new Outliers();
            outl0.setNpoints(nPoints);
            outl0.resetNumberTietjenMooreSimulations(nSimulations);
            outl0.resetSignificanceLevel(significance);
            return outl0.getTietjenMooreCriticalL(nPossibleOutliers);
        }
                
        public static double getTietjenMooreCriticalL(int nPossibleOutliers, int nPoints, double significance){ 
            Outliers outl0 = new Outliers();
            outl0.setNpoints(nPoints);
            outl0.resetSignificanceLevel(significance);
            return outl0.getTietjenMooreCriticalL(nPossibleOutliers);
        }
        
        // Tietjen and Moore's Critical value for lower outliers
        // instance method
        public double getTietjenMooreLowerCriticalL(int nPossibleOutliers){
            return getTietjenMooreCriticalLcore(nPossibleOutliers, 1);
        }
        
        // Tietjen and Moore's Critical value for lower outliers
        // static method
        public static double getTietjenMooreLowerCriticalL(int nPossibleOutliers, int nPoints, int nSimulations, double significance){
            Outliers outl0 = new Outliers();
            outl0.setNpoints(nPoints);
            outl0.resetNumberTietjenMooreSimulations(nSimulations);
            outl0.resetSignificanceLevel(significance);
            return outl0.getTietjenMooreLowerCriticalL(nPossibleOutliers);
        }
        
        public static double getTietjenMooreLowerCriticalL(int nPossibleOutliers, int nPoints, double significance){
            Outliers outl0 = new Outliers();
            outl0.setNpoints(nPoints);
            outl0.resetSignificanceLevel(significance);
            return outl0.getTietjenMooreLowerCriticalL(nPossibleOutliers);
        }
        
        // Tietjen and Moore's Critical value for upper outliers
        // instance method
        public double getTietjenMooreUpperCriticalL(int nPossibleOutliers){
            return getTietjenMooreCriticalLcore(nPossibleOutliers, 2);
        }

        // Tietjen and Moore's Critical value for upper outliers
        // static method
        public static double getTietjenMooreUpperCriticalL(int nPossibleOutliers, int nPoints, int nSimulations, double significance){
            Outliers outl0 = new Outliers();
            outl0.setNpoints(nPoints);
            outl0.resetNumberTietjenMooreSimulations(nSimulations);
            outl0.resetSignificanceLevel(significance);
            return outl0.getTietjenMooreUpperCriticalL(nPossibleOutliers);
        }
        
        public static double getTietjenMooreUpperCriticalL(int nPossibleOutliers, int nPoints, double significance){
            Outliers outl0 = new Outliers();
            outl0.setNpoints(nPoints);
            outl0.resetSignificanceLevel(significance);
            return outl0.getTietjenMooreUpperCriticalL(nPossibleOutliers);
        }
        
        // Core method 
        private double getTietjenMooreCriticalLcore(int nPossibleOutliers, int option){
            double criticalL = Double.NaN;
            Stat.denominatorSwap();
            //  Simulations
            double[] simulLs = new double[this.nTietjenMooreSimulations];
            double[] orderedRandom = null;
            
            for(int i=0; i<this.nTietjenMooreSimulations; i++){
                double[] random = Stat.gaussianRand(0.0, 1.0, this.nPoints); 
                ArrayList<Object> retal = this.orderData(random, option);
                orderedRandom = (double[])retal.get(0);
                simulLs[i] = this.getTietjenMooreLvalue(orderedRandom, this.nPoints, nPossibleOutliers);   
            }
            ArrayMaths am1 = new ArrayMaths(simulLs);
            am1 = am1.sort();
            simulLs = am1.array();
            int iCritical = (int)(this.significance*this.nTietjenMooreSimulations);
            criticalL = simulLs[iCritical];    

            Stat.denominatorUnswap();
            return criticalL;
        }
           
        // Tietjen and Moore's L value for lower or upper outliers
        // instance method
        private double getTietjenMooreLvalue(double[] ordData, int nP, int nPossibleOutliers){
            double tmLvalue = Double.NaN;
            
            int k = nP - nPossibleOutliers;
            double[] nonOutliers = new double[k];
           
            for(int i=0; i<k; i++){
                 nonOutliers[i] = ordData[i]; 
            }
            double meanNonOuliers = Stat.mean(nonOutliers);
            
            double numer = 0.0;
            double denom = 0.0;
            double mean = Stat.mean(ordData);
            for(int i=0; i<k; i++)numer += Fmath.square(ordData[i] - meanNonOuliers);
            for(int i=0; i<nPoints; i++)denom += Fmath.square(ordData[i] - mean);
            tmLvalue = numer/denom;
            
            return tmLvalue;
        }
        
        // BACKWARD COMPATABILITY
        public ArrayList<Object> outliersTeitjenMoore(int nOutliers){
            return this.outliersTietjenMoore(nOutliers);
        } 
       
        public ArrayList<Object> lowerOutliersTeitjenMoore(int nOutliers){
            return this.lowerOutliersTietjenMoore(nOutliers);
        } 

        public ArrayList<Object> upperOutliersTeitjenMoore(int nOutliers){
            return this.upperOutliersTietjenMoore(nOutliers);
        } 
            
        public double getTeitjenMooreCriticalL(int nOutliers){
            return this.getTietjenMooreCriticalL(nOutliers);
        } 
        
        public double getTeitjenMooreLowerCriticalL(int nOutliers){
            return this.getTietjenMooreLowerCriticalL(nOutliers);
        }

        public double getTeitjenMooreUpperCriticalL(int nOutliers){
            return this.getTietjenMooreUpperCriticalL(nOutliers);
        }
        
        public void resetNumberTeitjenMooreSimulations(int nSimulations){
            this.resetNumberTietjenMooreSimulations(nSimulations);
        }

        public int getNumberTeitjenMooreSimulations(){
            return this.getNumberTietjenMooreSimulations();
        }
        
        public static ArrayList<Object> outliersTeitjenMoore(double[] data, int nOutliers){
            return Outliers.outliersTietjenMoore(data, nOutliers);

        }  
        
        public static ArrayList<Object> lowerOutliersTeitjenMoore(double[] data, int nOutliers){
            return Outliers.lowerOutliersTietjenMoore(data, nOutliers);
        } 

        public static ArrayList<Object> upperOutliersTeitjenMoore(double[] data, int nOutliers){
            return Outliers.upperOutliersTietjenMoore(data, nOutliers);
        }
        
        public static double getTeitjenMooreCriticalL(int nOutliers, int nObservations, int nSimulations, double significance){
            return Outliers.getTietjenMooreCriticalL(nOutliers, nObservations, nSimulations, significance);
        } 
            
        public static double getTeitjenMooreCriticalL(int nOutliers, int nObservations, double significance){
            return Outliers.getTietjenMooreCriticalL(nOutliers, nObservations, significance);
        } 

        public static double getTeitjenMooreLowerCriticalL(int nOutliers, int nObservations, int nSimulations, double significance){
            return Outliers.getTietjenMooreLowerCriticalL(nOutliers, nObservations, nSimulations, significance);
        } 

        public static double getTeitjenMooreLowerCriticalL(int nOutliers, int nObservations, double significance){
            return Outliers.getTietjenMooreLowerCriticalL(nOutliers, nObservations, significance);
        }

        public static double getTeitjenMooreUpperCriticalL(int nOutliers, int nObservations, int nSimulations, double significance){
            return Outliers.getTietjenMooreUpperCriticalL(nOutliers, nObservations, nSimulations, significance);
        } 

        public static double getTeitjenMooreUpperCriticalL(int nOutliers, int nObservations, double significance){
            return Outliers.getTietjenMooreUpperCriticalL(nOutliers, nObservations, significance);
        }

        
        // GENERALISED ESD (EXTREME STUDENTISED DEVIATE) TEST
        
        // ESD upper or lower outliers
        // instance method
        public ArrayList<Object> outliersESD(int nOutliers){
            return this.outliersESDcore(nOutliers, 0);

        }

        // ESD core method
        public ArrayList<Object> outliersESDcore(int nOutliers, int option){
            if(this.nPoints-nOutliers<=1){
                System.out.println("Number of suggested outliers, " + nOutliers + ", must be less than the number of points minus two, " + (this.nPoints - 2));
                nOutliers--;
                System.out.println("The number of outliers has been reduced to " + nOutliers);
            }
            this.esdLambdas = this.getESDlambdas(nOutliers);
            this.esdRvalues = new double[nOutliers];
            this.esdTests = new boolean[nOutliers];
            this.esdMaxValues = new double[nOutliers];
            double[] runningData = (double[])(this.orderData(this.data, option)).get(0);
            boolean found = false;
            int nFound = 0;
            for(int i=0; i<nOutliers; i++){
                this.esdRvalues[i] = Math.abs(runningData[nPoints-i-1] - Stat.mean(runningData))/Stat.standardDeviation(runningData);
                this.esdTests[i] = false;
                if(this.esdRvalues[i]>this.esdLambdas[i]){
                    this.esdTests[i] = true;
                    found = true;
                    nFound = i+1;
                }              
                double[] hold = DeepCopy.copy(runningData);
                this.esdMaxValues[i] = runningData[nPoints - 1 - i];
                runningData = new double[nPoints-i-1];
                for(int j=0; j<nPoints-i-1; j++)runningData[j] = hold[j];
                runningData = (double[])(this.orderData(runningData, option)).get(0);
            }
            double[] foundValues = null;
            int[] foundIndices = null;
            boolean[] indexCheck = null;
            
            if(found){
                // outlier values and indices
                foundValues = new double[nFound];
                foundIndices = new int[nFound];
                indexCheck = new boolean[this.nPoints];
                for(int i=0; i<this.nPoints; i++)indexCheck[i] = true;
                for(int i=0; i<nFound; i++){
                    foundValues[i] = this.esdMaxValues[i];
                    for(int j=0; j<this.nPoints; j++){
                        if(indexCheck[j] && foundValues[i]==this.data[j]){
                            foundIndices[i] = j;
                            indexCheck[j] = false;
                            break;
                        }
                    }
                }
                // data with outliers removed
                int nRem = this.nPoints - nFound;
                this.strippedData = new double[nRem];
                int jj = 0;
                for(int i=0; i<this.nPoints;i++){
                    if(indexCheck[i]){
                        this.strippedData[jj] = this.data[i];
                         jj++;
                    }
                }
            }
            
            ArrayList<Object> ret = new ArrayList<Object>();
            ret.add(found);
            ret.add(foundValues);
            ret.add(foundIndices);
            ret.add(this.strippedData);
            ret.add(this.esdRvalues);
            ret.add(this.esdLambdas);
            ret.add(this.esdTests);
            ret.add(this.esdMaxValues);
            ret.add(this.significance);
            ret.add(this.nPoints);
            ret.add(nOutliers);
            
            ArrayList<Double> alD = this.probabilityPlots(found, this.data, this.strippedData);
            
            ret.add(alD.get(0));
            ret.add(alD.get(1));
            ret.add(alD.get(2));
            ret.add(alD.get(3));
            
            if(found){
                ret.add(alD.get(4));
                ret.add(alD.get(5));
                ret.add(alD.get(6));
                ret.add(alD.get(7));
            }
            else{
                ret.add(Double.NaN);
                ret.add(Double.NaN);
                ret.add(Double.NaN);
                ret.add(Double.NaN);
            }

            ArrayList<Object> alswA = this.shapiroWilkTest(this.data);
            double wValueA = (Double)alswA.get(0);
            double wCritA = (Double)alswA.get(1);
            double pValueA = (Double)alswA.get(2);
            String swWtestA = (String)alswA.get(3);
            double wValueB = Double.NaN;
            double wCritB = Double.NaN;
            double pValueB = Double.NaN;
            String swWtestB = null;
            if(found){
                ArrayList<Object> alswB = this.shapiroWilkTest(this.strippedData);
                wValueB = (Double)alswB.get(0);
                wCritB = (Double)alswB.get(1);
                pValueB = (Double)alswB.get(2);
                swWtestB = (String)alswB.get(3);
            }
            
            ret.add(wValueA);
            ret.add(wCritA);
            ret.add(pValueA);
            if(found){
                ret.add(wValueB);
                ret.add(wCritB);
                ret.add(pValueB);
            }
            else{
                ret.add(Double.NaN);
                ret.add(Double.NaN);
                ret.add(Double.NaN);
            }
            
            // Print
            if(!this.suppressPrint){
                FileOutput fout0 = new FileOutput(this.filenameout);
                fout0.println("Outlier Detection Results");
                fout0.println("Outlier Detection Method: Generalised ESD (Extreme Studentised Deviate) Test");
                fout0.println("Lower and upper outliers");
                if(this.filenamein!=null)fout0.println("Input file: " + this.filenamein);
                fout0.println();
                fout0.println("File name: " + filenameout);
                fout0.println("Run date: " + this.time_date);
                fout0.println();
                if(found){
                    fout0.println(nFound + " outliers indicated");
                }
                else{
                     fout0.println("No outliers indicated");
                }
                fout0.println();
                fout0.println("Significance level: " + this.significance + " [" + this.significance*100 + "%]");
                fout0.println();
                int field2 = 12;
                int trunc = 4;
                fout0.println("Potential   Test        Critical    R>lambda    Maximum");
                fout0.println("number of   Statistic,  Value,                  deviate");
                fout0.println("outliers    R           lambda                  data value");
                for(int i=0; i<nOutliers; i++){
                    fout0.print(i+1, field2);
                    fout0.print(Fmath.truncate(this.esdRvalues[i], trunc), field2);
                    fout0.print(Fmath.truncate(this.esdLambdas[i], trunc), field2);
                    fout0.print(this.esdTests[i], field2);
                    fout0.print(Fmath.truncate(this.esdMaxValues[i], trunc), field2);
                    fout0.println();
                }
                fout0.println();
                if(found){
                    fout0.println("Outliers: ");
                    for(int i=0; i<nFound; i++)fout0.print(Fmath.truncate(foundValues[i], trunc) + "  ");
                    fout0.println();             
                    fout0.println("Outlier indices in the original inputted data [indices start at 0]: ");
                    for(int i=0; i<nFound; i++)fout0.print(foundIndices[i] + "  ");
                    fout0.println();
                    fout0.println();
                }
                                
                fout0.print(" ", this.field0);
                fout0.print("Original data ", this.field);
                if(found)fout0.println("Data with outlier/s removed");
                else fout0.println();
                fout0.print("Number of data points: ", this.field0);
                fout0.print(this.nPoints, this.field);
                if(found)fout0.println((this.nPoints - nFound));
                else fout0.println();
                fout0.print("Sample mean: ", this.field0);
                fout0.print(Fmath.truncate(Stat.mean(this.data), trunc), this.field);
                if(found)fout0.println(Fmath.truncate(Stat.mean(this.strippedData), trunc));
                else fout0.println();
                fout0.print("Sample standard deviation: ", this.field0);
                fout0.print(Fmath.truncate(Stat.standardDeviation(this.data), trunc), this.field);
                if(found)fout0.println(Fmath.truncate(Stat.standardDeviation(this.strippedData), trunc));
                else fout0.println();
                fout0.print("Maximum: ", this.field0);
                fout0.print(Fmath.truncate(Fmath.maximum(this.data), trunc), this.field);
                if(found)fout0.println(Fmath.truncate(Fmath.maximum(this.strippedData), trunc));
                else fout0.println();
                fout0.print("Minimum: ", this.field0);
                fout0.print(Fmath.truncate(Fmath.minimum(this.data), trunc), this.field);
                if(found)fout0.println(Fmath.truncate(Fmath.minimum(this.strippedData), trunc));
                else fout0.println();
                
                fout0.println();
                fout0.println("Shapiro-Wilk Test:");
                fout0.print(" Shapiro-Wilk W value: ", this.field0);
                fout0.print(Fmath.truncate(wValueA, trunc), this.field);
                if(found){
                    fout0.println(Fmath.truncate(wValueB, trunc));
                }
                else fout0.println(); 
                fout0.println(" Critical value for W: ");
                fout0.print("  (" + this.significance*100 + "% Significance level)", this.field0);
                fout0.print(Fmath.truncate(wCritA, trunc), this.field);
                if(found){
                    fout0.println(Fmath.truncate(wCritB, trunc));
                }
                else fout0.println(); 
                fout0.print(" Shapiro-Wilk P value: ", this.field0);
                fout0.print(Fmath.truncate(pValueA, trunc), this.field);
                if(found){
                    fout0.println(Fmath.truncate(pValueB, trunc));
                }
                else fout0.println(); 
                fout0.print(" Data possibly Gaussian: ", this.field0);
                fout0.print(swWtestA, this.field);
                if(found){
                    fout0.println(swWtestB);
                }
                else fout0.println(); 
                fout0.println();
                
                fout0.println("Probability plot:");
                fout0.print(" Correlation coefficient, r: ", this.field0);
                double corrCoeffA = alD.get(2);
                double corrCoeffB = Double.NaN;
                fout0.print(Fmath.truncate(corrCoeffA, trunc), this.field);
                if(found){
                    corrCoeffB = alD.get(6);
                    fout0.println(Fmath.truncate(corrCoeffB, trunc));
                }
                else fout0.println(); 
                fout0.println(" Critical value for r: ");
                fout0.print("  (" + this.significance*100 + "% Significance level)", this.field0);
                double critValueA = alD.get(3);
                double critValueB = Double.NaN;
                fout0.print(Fmath.truncate(critValueA, trunc), this.field);
                if(found){
                    critValueB = alD.get(7);
                    fout0.println(Fmath.truncate(critValueB, trunc));
                }
                else fout0.println(); 
                fout0.print(" Data possibly Gaussian: ", this.field0);
                String gTest = "Rejected";
                if(corrCoeffA>=critValueA)gTest = "Accepted";
                fout0.print(gTest, this.field);
                if(found){
                    gTest = "Rejected";
                    if(corrCoeffB>=critValueB)gTest = "Accepted";
                    fout0.println(gTest);
                }
                else fout0.println(); 
                fout0.print(" Gradient: ", this.field0);
                fout0.print(Fmath.truncate(alD.get(0), trunc), this.field);
                if(found)fout0.println(Fmath.truncate(alD.get(4), trunc));
                else fout0.println();
                fout0.print(" Intercept: ", this.field0);
                fout0.print(Fmath.truncate(alD.get(1), trunc), this.field);
                if(found)fout0.println(Fmath.truncate(alD.get(5), trunc));
                else fout0.println();
                
                
                fout0.println();
                fout0.println("Input data: ");
                int kk = 0;
                for(int i=0; i<this.nPoints; i++){
                    fout0.print(this.data[i] + "  ");
                    kk++;
                    if(kk==10){
                        fout0.println();
                        kk = 0;
                    }
                }
                fout0.close();
            }
            return ret;
        }
        
        // ESD upper or lower outliers
        // static method
        public static ArrayList<Object> outliersESD(double[] data, int nOutliers){
            Outliers outl0 = new Outliers(data);
            return outl0.outliersESD(nOutliers);
        }
        
        // ESD lambda array
        // instance method
        public double[] getESDlambdas(int r){
            this.esdLambdas = new double[r];
            for(int i=1; i<=r; i++){
                this.esdLambdas[i-1] = this.getESDlambda(i);
            }
            return this.esdLambdas;
        }
       
        // Calculate ESD lambda
        // lower and upper outliers
        private double getESDlambda(int i){
            return this.getESDlambdaCore(i, 2);
        }
            
        // Calculate ESD lambda
        // core method
        private double getESDlambdaCore(int i, int tailChoice){
            double esdl = Double.NaN;
            
            int nn = this.nPoints - i;
            double p = 1.0 - (this.significance/(nn + 1))/tailChoice;
            double studT = Stat.studentstValue(p, nn - 1);
            esdl = nn*studT/Math.sqrt((nn - 1 + studT*studT)*(nn + 1));  
            return esdl;
        }
        
        // ESD lambda array
        // static method
        public static double[] getESDlambdas(int r, int nPoints, double significance){
            Outliers outl0 = new Outliers();
            outl0.resetSignificanceLevel(significance);
            outl0.setNpoints(nPoints);
            return outl0.getESDlambdas(r);
        }
        
       // DIXON Q TEST
        // Points ignored
        public double[] calcIgnoredPoints(int a, int b, int[] indices, int luOption){
            ArrayList<Double> alig = new ArrayList<Double>();
            if(luOption==0){
                for(int i=1; i<a; i++)alig.add(this.data[indices[i]]);
                for(int i=this.nPoints-b; i<this.nPoints; i++)alig.add(this.data[indices[i]]);
            }
            else{
                for(int i=0; i<b; i++)alig.add(this.data[indices[i]]);
                for(int i=this.nPoints-a; i<this.nPoints-1; i++)alig.add(this.data[indices[i]]);
            }
            this.pointsIgnored = false;
            this.ignoredPoints = null;
            this.nPointsIgnored = alig.size();
           
            if(this.nPointsIgnored>0){
               this.ignoredPoints = new double[this.nPointsIgnored];
               this.pointsIgnored = true; 
               for(int i=0; i<this.nPointsIgnored; i++)this.ignoredPoints[i] = alig.get(i);
            }
 
            return this.ignoredPoints;
        }
        
        // Points ignored
        // indices for a lower test
        public static int[] ignoredIndicesLowerTest(int a, int b, int nPoints){
            int[] ignoredIndices = null;
            
            ArrayList<Integer> alig = new ArrayList<Integer>();
            for(int i=1; i<a; i++)alig.add(i+1);
            for(int i=nPoints-b; i<nPoints; i++)alig.add(i+1);
    
            int nPointsIgnored = alig.size();
            if(nPointsIgnored>0){
               ignoredIndices = new int[nPointsIgnored];
               for(int i=0; i<nPointsIgnored; i++)ignoredIndices[i] = alig.get(i);
            }
            return ignoredIndices;
        }
        
        // Points ignored
        // indices for an upper test
        public static int[] ignoredIndicesUpperTest(int a, int b, int nPoints){
            int[] ignoredIndices = null;
            
            ArrayList<Integer> alig = new ArrayList<Integer>();
            for(int i=0; i<b; i++)alig.add(i+1);
            for(int i=nPoints-a; i<nPoints-1; i++)alig.add(i+1);
            
            int nPointsIgnored = alig.size();
            if(nPointsIgnored>0){
               ignoredIndices = new int[nPointsIgnored];
               for(int i=0; i<nPointsIgnored; i++)ignoredIndices[i] = alig.get(i);
            }
            return ignoredIndices;
        }
        
        // Number of Dixon simulations
        // Reset number of Dixon simulations
        public void resetNumberDixonSimulations(int n){
            this.nDixonSimulations = n;
        } 
        
        // Get number of Dixon  simulations
        public int getNumberDixonSimulations(){
            return this.nDixonSimulations;
        } 
        
        // Dixon Q test
        // Get rab subscripts
        // luOption = lowere, luOption = 1 upper
        // instance method
        private int[] getSubscripts(int a, int b, int luOption){
            int[] sub = new int[4];
            if(luOption==0){
                sub[0] = a;
                sub[1]= 0;
                sub[2] = this.nPoints - b - 1;
                sub[3] = 0;
            }
            else{
                sub[0] = this.nPoints - 1;
                sub[1] = this.nPoints - a - 1;
                sub[2] = this.nPoints - 1;
                sub[3] = b;       
            }
            return sub;
        }
        
        // Get rab subscripts
        // luOption = lower, luOption = 1 upper
        // static method
        private static int[] getSubscripts(int a, int b, int luOption, int nPoints){
            int[] sub = new int[4];
            if(luOption==0){
                sub[0] = a;
                sub[1]= 0;
                sub[2] = nPoints - b - 1;
                sub[3] = 0;
            }
            else{
                sub[0] = nPoints - 1;
                sub[1] = nPoints - a - 1;
                sub[2] = nPoints - 1;
                sub[3] = b;       
            }
            return sub;
        }
        
        
        // Dixon's Q rab test for a single lower or upper outer oulier
        // instance metod
        public ArrayList<Object> outlierDixon(int a, int b){
            this.bluOption = 0;
            ArrayList<Object> ret = new ArrayList<Object>();
            
            double[] datah = (double[])this.orderData(DeepCopy.copy(this.data), 2).get(0);
            int[] indh = (int[])this.orderData(datah, 2).get(1);
            
            int luOption = 0;
            int[] subsL = this.getSubscripts(a, b, luOption);
            double gap = datah[subsL[0]] - datah[subsL[1]];
            double width = datah[subsL[2]] - datah[subsL[3]];
            double dixonQvalueL = gap/width;
            
            luOption = 1;
            int[] subsU = this.getSubscripts(a, b, luOption);
            gap = datah[subsU[0]] - datah[subsU[1]];
            width = datah[subsU[2]] - datah[subsU[3]];
            double dixonQvalueU = gap/width;
            
            luOption = 0;
            if(dixonQvalueU>dixonQvalueL)luOption = 1;
            
            int dixonQindex = -1;
            double dixonQcrit = Double.NaN;
            double dixonQvalue = Double.NaN;
            boolean found = false;
            if(luOption==0){
                dixonQvalue = dixonQvalueL;
                dixonQindex = indh[0];
                dixonQcrit = Outliers.getDixonCritical(subsL, nPoints, this.nDixonSimulations, this.significance, 2);
                if(dixonQvalueL>=dixonQcrit)found = true;
            }
            else{
                dixonQvalue = dixonQvalueU;
                dixonQindex = indh[this.nPoints-1];
                dixonQcrit = Outliers.getDixonCritical(subsU, nPoints, this.nDixonSimulations, this.significance, 2);
                if(dixonQvalueU>=dixonQcrit)found = true;
            }
            double outlier = this.data[dixonQindex];
            this.calcIgnoredPoints(a, b, indh, luOption);
            this.strippedData(found, dixonQindex);
            ArrayList<Double> alD = this.probabilityPlots(found, this.data, this.strippedData);
            ArrayList<Object> alswA = this.shapiroWilkTest(this.data);
            ArrayList<Object> alswB = null;
            if(found)alswB = this.shapiroWilkTest(this.strippedData);
            if(!this.suppressPrint)this.dixonTextFile(found, a, b, dixonQvalue, dixonQcrit, dixonQindex, outlier, alD, alswA, alswB);
            return this.dixonRetArray(found, dixonQvalue, dixonQcrit, dixonQindex, a, b, alD, alswA, alswB);

        }
        
        
        // Dixon's Q rab test for a single lower or upper outer outlier
        // static metod
        public static ArrayList<Object> outlierDixon(double data[], int a, int b){
            Outliers outl = new Outliers(data);
            return outl.outlierDixon(a, b);
        }
        
        
        // Dixon's Q test for a single lower or upper outer outlier ignoring no points
        // instance method
        public ArrayList<Object> outlierDixon(){
            return this.outlierDixon(1, 0);
        }
        
        // Dixon's Q test for a single lower or upper outer oulier ignoring no points
        // static metod
        public static ArrayList<Object> outlierDixon(double data[]){
            Outliers outl = new Outliers(data);
            return outl.outlierDixon(1, 0);
        }
        
        
        // Dixon's Q rab test for a single lower outer oulier
        // instance metod
        public ArrayList<Object> lowerOutlierDixon(int a, int b){
            this.bluOption = 1;
            int luOption = 0;
            int[] subs = this.getSubscripts(a, b, luOption);
            ArrayList<Object> ret = new ArrayList<Object>();
            double[] datah = (double[])this.orderData(DeepCopy.copy(this.data), 2).get(0);
            int[] indh = (int[])this.orderData(datah, 2).get(1);
            double gap = datah[subs[0]] - datah[subs[1]];
            double width = datah[subs[2]] - datah[subs[3]];
            double dixonQvalue = gap/width;
            int dixonQindex = indh[0];
            double dixonQcrit = Outliers.getDixonCritical(subs, nPoints, this.nDixonSimulations, this.significance, 1);
            boolean found = false;
            if(dixonQvalue>=dixonQcrit)found = true;
            double outlier = this.data[dixonQindex];        
            this.calcIgnoredPoints(a, b, indh, luOption);
            this.strippedData(found, dixonQindex);
            ArrayList<Double> alD = this.probabilityPlots(found, this.data, this.strippedData); 
            ArrayList<Object> alswA = this.shapiroWilkTest(this.data);
            ArrayList<Object> alswB = null;
            if(found)alswB = this.shapiroWilkTest(this.strippedData);
            if(!this.suppressPrint)this.dixonTextFile(found, a, b, dixonQvalue, dixonQcrit, dixonQindex, outlier, alD, alswA, alswB);
            return this.dixonRetArray(found, dixonQvalue, dixonQcrit, dixonQindex, a, b, alD, alswA, alswB);
        }
        
        private ArrayList<Object> dixonRetArray(boolean found, double dixonQvalue, double dixonQcrit, int dixonQindex, int a, int b, ArrayList<Double> alD, ArrayList<Object> alswA, ArrayList<Object> alswB){
            ArrayList<Object> ret = new ArrayList<Object>();
            
            double wValueA = (Double)alswA.get(0);
            double wCritA = (Double)alswA.get(1);
            double pValueA = (Double)alswA.get(2);
            String swWtestA = (String)alswA.get(3);
            double wValueB = Double.NaN;
            double wCritB = Double.NaN;
            double pValueB = Double.NaN;
            String swWtestB = null;
            if(found){       
                wValueB = (Double)alswB.get(0);
                wCritB = (Double)alswB.get(1);
                pValueB = (Double)alswB.get(2);
                swWtestB = (String)alswB.get(3);
            }
            
            double outlier = Double.NaN;
            double[] stripped = null;
            int foundIndex = -1;
            if(found){
                outlier = data[dixonQindex];
                stripped = new double[this.nPoints-1];
                int jj = 0;
                for(int i=0; i<this.nPoints-2; i++){
                    if(i!=dixonQindex)stripped[jj] = this.data[i];
                    jj++;
                }
                foundIndex = dixonQindex;
            }
            else{
                stripped = DeepCopy.copy(this.data);
            }
            ret.add(found);
            ret.add(outlier);
            ret.add(foundIndex);
            ret.add(stripped);
            ret.add(dixonQvalue);
            ret.add(dixonQcrit);
            ret.add(this.significance);
            ret.add(this.nPoints);
            ret.add(a);
            ret.add(b);
            ret.add(this.nPointsIgnored);
            ret.add(this.ignoredPoints);
            
            ret.add(alD.get(0));
            ret.add(alD.get(1));
            ret.add(alD.get(2));
            ret.add(alD.get(3));
            
            if(found){
                ret.add(alD.get(4));
                ret.add(alD.get(5));
                ret.add(alD.get(6));
                ret.add(alD.get(7));
            }
            else{
                ret.add(Double.NaN);
                ret.add(Double.NaN);
                ret.add(Double.NaN);
                ret.add(Double.NaN);
            }
            
            ret.add(wValueA);
            ret.add(wCritA);
            ret.add(pValueA);
            if(found){
                ret.add(wValueB);
                ret.add(wCritB);
                ret.add(pValueB);
            }
            else{
                ret.add(Double.NaN);
                ret.add(Double.NaN);
                ret.add(Double.NaN);
            }
            

            return ret;
        }
        
        // Dixon's Q rab test for a single lower outer oulier
        // static metod
        public static ArrayList<Object> lowerOutlierDixon(double data[], int a, int b){
            Outliers outl = new Outliers(data);
            return outl.lowerOutlierDixon(a, b);
        }
        
        
        // Dixon's Q test for a single lower outer outlier ignoring no points
        // instance method
        public ArrayList<Object> lower0outlierDixon(){
            return this.lowerOutlierDixon(1, 0);
        }
        
        // Dixon's Q test for a single lower outer outlier ignoring no points
        // static metod
        public static ArrayList<Object> lowerOutlierDixon(double data[]){
            Outliers outl = new Outliers(data);
            return outl.lowerOutlierDixon(1, 0);
        }
        
   
        
        // Dixon's Q rab test for a single upper outer oulier
        // instance metod
        public ArrayList<Object> upperOutlierDixon(int a, int b){
            this.bluOption = 2;
            int luOption = 1;
            int[] subs = this.getSubscripts(a, b, luOption);
            ArrayList<Object> ret = new ArrayList<Object>();
            double[] datah = (double[])this.orderData(DeepCopy.copy(this.data), 2).get(0);
            int[] indh = (int[])this.orderData(datah, 2).get(1);
            double gap = datah[subs[0]] - datah[subs[1]];
            double width = datah[subs[2]] - datah[subs[3]];
            double dixonQvalue = gap/width;
            int dixonQindex = indh[this.nPoints-1];
            double dixonQcrit = Outliers.getDixonCritical(subs, nPoints, this.nDixonSimulations, this.significance, 1);
            boolean found = false;
            if(dixonQvalue>=dixonQcrit)found = true;
            double outlier = data[dixonQindex];
            this.calcIgnoredPoints(a, b, indh, luOption);
            this.strippedData(found, dixonQindex);
            ArrayList<Double> alD = this.probabilityPlots(found, this.data, this.strippedData);
            ArrayList<Object> alswA = this.shapiroWilkTest(this.data);
            ArrayList<Object> alswB = null;
            if(found)alswB = this.shapiroWilkTest(this.strippedData);
            if(!this.suppressPrint)this.dixonTextFile(found, a, b, dixonQvalue, dixonQcrit, dixonQindex, outlier, alD, alswA, alswB);
    
            return this.dixonRetArray(found, dixonQvalue, dixonQcrit, dixonQindex, a, b, alD, alswA, alswB);        
        }
        
        // Dixon's Q rab test for a single upper outer oulier
        // static metod
        public static ArrayList<Object> upperOutlierDixon(double data[], int a, int b){
            Outliers outl = new Outliers(data);
            return outl.upperOutlierDixon(a, b);
        }
        
        
        // Dixon's Q test for a single upper outer oulier ignoring no points
        // instance method
        public ArrayList<Object> upperOutlierDixon(){
            return this.upperOutlierDixon(1, 0);
        }
        
        // Dixon's Q test for a single upper outer oulier ignoring no points
        // static metod
        public static ArrayList<Object> upperOutlierDixon(double data[]){
            Outliers outl = new Outliers(data);
            return outl.upperOutlierDixon(1, 0);
        }
              
        // Critical Q value
        // one tail test
        // Static method
        public static double getDixonOneTailedCriticalQ(int a, int b, int nPoints, int nSimulations, double significance){
           int[] subs = Outliers.getSubscripts(a, b, 0, nPoints);
           return Outliers.getDixonCritical(subs, nPoints, nSimulations, significance, 1);
        }
        
        // Critical Q value
        // one tail test
        // Static method
        public static double getDixonOneTailedCriticalQ(int a, int b, int nPoints, double significance){
           int[] subs = Outliers.getSubscripts(a, b, 0, nPoints);
           return Outliers.getDixonCritical(subs, nPoints, Outliers.nDixonSimulationsStatic, significance, 1);
        }
        
        // Critical Q value
        // one tail test, r10
        // Static method
        public static double getDixonOneTailedCriticalQ(int nPoints, int nSimulations, double significance){
           int[] subs = Outliers.getSubscripts(1, 0, 0, nPoints);
           return Outliers.getDixonCritical(subs, nPoints, nSimulations, significance, 1);
        }
        
        // Critical Q value
        // one tail test, r10
        // Static method
        public static double getDixonOneTailedCriticalQ(int nPoints, double significance){
           int[] subs = Outliers.getSubscripts(1, 0, 0, nPoints);
           return Outliers.getDixonCritical(subs, nPoints, Outliers.nDixonSimulationsStatic, significance, 1);
        }
        
        // Critical Q value
        // Two tail test
        // Static method
        public static double getDixonTwoTailedCriticalQ(int a, int b, int nPoints, int nSimulations, double significance){
           int[] subs = Outliers.getSubscripts(a, b, 0, nPoints);
           return Outliers.getDixonCritical(subs, nPoints, nSimulations, significance, 2);
        }
        
        // Critical Q value
        // Two tail test
        // Static method
        public static double getDixonTwoTailedCriticalQ(int a, int b, int nPoints, double significance){
           int[] subs = Outliers.getSubscripts(a, b, 0, nPoints);
           return Outliers.getDixonCritical(subs, nPoints, Outliers.nDixonSimulationsStatic, significance, 2);
        }
        
        
        // Critical Q value
        // Two tail test, r10
        // Static method
        public static double getDixonTwoTailedCriticalQ(int nPoints, int nSimulations, double significance){
           int[] subs = Outliers.getSubscripts(1, 0, 0, nPoints);
           return Outliers.getDixonCritical(subs, nPoints, nSimulations, significance, 2);
        }
        
        // Critical Q value
        // Two tail test, r10
        // Static method
        public static double getDixonTwoTailedCriticalQ(int nPoints, double significance){
           int[] subs = Outliers.getSubscripts(1, 0, 0, nPoints);
           return Outliers.getDixonCritical(subs, nPoints, Outliers.nDixonSimulationsStatic, significance, 2);
        }
        
        // Critical Q value
        // Core method 0
        // tailOption = 1 one-tailed, = 2 two-tailed
        // Static method
        private static double getDixonCritical(int a, int b, int nPoints, int nSimulations, double significance, int tailOption){
           int[] subs = Outliers.getSubscripts(a, b, 0, nPoints);
           return Outliers.getDixonCritical(subs, nPoints, nSimulations, significance, tailOption);
        }
        
        // Critical Q value
        // Core method 1
        // luOption = 0 lower; luOption = 1 upper
        // Static method
        private static double getDixonCritical(int a, int b, int luOption, int nPoints, int nSimulations, double significance, int tailOption){
           int[] subs = Outliers.getSubscripts(a, b, luOption, nPoints);
           return Outliers.getDixonCritical(subs, nPoints, nSimulations, significance, tailOption);
        }
            
        // Critical Q value
        // Core method 2
        // Static method
        private static double getDixonCritical(int[] sub, int nPoints, int nSimulations, double significance, int tailOption){
       
            double[] qSimul = new double[nSimulations];
            
            for(int i=0; i<nSimulations; i++){
                double[] ySimul = Stat.gaussianRand(0.0, 1.0, nPoints);
                ArrayMaths am0 = new ArrayMaths(ySimul);
                ySimul = (am0.sort()).array();
                double gap = ySimul[sub[0]] - ySimul[sub[1]];
                double width = ySimul[sub[2]] - ySimul[sub[3]];
                qSimul[i] = gap/width;
                
            }
            
            ArrayMaths am1 = new ArrayMaths(qSimul);
            qSimul = (am1.sort()).array();
            int nCrit = (int)Math.round(nSimulations*(1.0 - significance/tailOption));
            double qCrit = qSimul[nCrit];
            return qCrit;
        }
        
        
        private void dixonTextFile(boolean found, int a, int b, double dixonQvalue, double dixonQcrit, int dixonQindex, double outlier, ArrayList<Double> alD, ArrayList<Object> alswA, ArrayList<Object> alswB){
            String rab = "r" + a + b;
            
            double wValueA = (Double)alswA.get(0);
            double wCritA = (Double)alswA.get(1);
            double pValueA = (Double)alswA.get(2);
            String swWtestA = (String)alswA.get(3);
            double wValueB = Double.NaN;
            double wCritB = Double.NaN;
            double pValueB = Double.NaN;
            String swWtestB = null;
            if(found){       
                wValueB = (Double)alswB.get(0);
                wCritB = (Double)alswB.get(1);
                pValueB = (Double)alswB.get(2);
                swWtestB = (String)alswB.get(3);
            }
            
            FileOutput fout0 = new FileOutput(this.filenameout);
            fout0.println("Outlier Detection Results");
            fout0.println("Outlier Detection Method: Dixon's Q Test with " + rab);
            String ulS = "";
            switch(this.bluOption){
                    case 0: fout0.println("Lower or upper outlier");
                            break;
                    case 1: fout0.println("Lower outlier");
                            ulS = " lower ";
                            break;
                    case 2: fout0.println("Upper outlier");
                            ulS = " upper ";
                            break;
            }
            if(this.filenamein!=null)fout0.println("Input file: " + this.filenamein);
            
            fout0.println();
            fout0.println("File name: " + filenameout);
            fout0.println("Run date: " + this.time_date);
            fout0.println();
            
            if(this.pointsIgnored){
                if(this.nPointsIgnored==1){
                    fout0.print("1 point ignored: ");
                }
                else{
                    fout0.print(this.nPointsIgnored + " points ignored: ");
                }
                for(int i=0; i<this.nPointsIgnored; i++)fout0.print(this.ignoredPoints[i] + "   ");
                fout0.println();
                fout0.println();
            }
            else{
                fout0.println("No points ignored");
                fout0.println();
            }
                      
            if(found){
                    fout0.println("Extreme outlier indicated");
            }
            else{
                     fout0.println("No" + ulS + "outlier indicated");
            }
            fout0.println();
            fout0.println("Significance level:    " + this.significance + " [" + this.significance*100 + "%]");
            fout0.println("Test Statistic, " + rab  + ":   " +  Fmath.truncate(dixonQvalue, this.trunc));
            fout0.println("Critical value, Qcrit: " +  Fmath.truncate(dixonQcrit, this.trunc));   
             
            fout0.println();
            if(found){
                    fout0.print("Outlier: ");
                    fout0.println(Fmath.truncate(outlier, trunc));           
                    fout0.print("Outlier index in the original inputted data [indices start at 0]: ");
                    fout0.println(dixonQindex);
                    fout0.println();
            }     
                

            fout0.print(" ", this.field0);
            fout0.print("Original data ", this.field);
            if(found)fout0.println("Data with outlier removed");
            else fout0.println();
            fout0.print("Number of data points: ", this.field0);
            fout0.print(this.nPoints, this.field);
            if(found)fout0.println((this.nPoints - 1));
            else fout0.println();
            fout0.print("Sample mean: ", this.field0);
            fout0.print(Fmath.truncate(Stat.mean(this.data), trunc), this.field);
            if(found)fout0.println(Fmath.truncate(Stat.mean(this.strippedData), trunc));
            else fout0.println();
            fout0.print("Sample standard deviation: ", this.field0);
            fout0.print(Fmath.truncate(Stat.standardDeviation(this.data), trunc), this.field);
            if(found)fout0.println(Fmath.truncate(Stat.standardDeviation(this.strippedData), trunc));
            else fout0.println();
            fout0.print("Maximum: ", this.field0);
            fout0.print(Fmath.truncate(Fmath.maximum(this.data), trunc), this.field);
            if(found)fout0.println(Fmath.truncate(Fmath.maximum(this.strippedData), trunc));
            else fout0.println();
            fout0.print("Minimum: ", this.field0);
            fout0.print(Fmath.truncate(Fmath.minimum(this.data), trunc), this.field);
            if(found)fout0.println(Fmath.truncate(Fmath.minimum(this.strippedData), trunc));
            else fout0.println();
            
            fout0.println();
                fout0.println("Shapiro-Wilk Test:");
                fout0.print(" Shapiro-Wilk W value: ", this.field0);
                fout0.print(Fmath.truncate(wValueA, trunc), this.field);
                if(found){
                    fout0.println(Fmath.truncate(wValueB, trunc));
                }
                else fout0.println(); 
                fout0.println(" Critical value for W: ");
                fout0.print("  (" + this.significance*100 + "% Significance level)", this.field0);
                fout0.print(Fmath.truncate(wCritA, trunc), this.field);
                if(found){
                    fout0.println(Fmath.truncate(wCritB, trunc));
                }
                else fout0.println(); 
                fout0.print(" Shapiro-Wilk P value: ", this.field0);
                fout0.print(Fmath.truncate(pValueA, trunc), this.field);
                if(found){
                    fout0.println(Fmath.truncate(pValueB, trunc));
                }
                else fout0.println(); 
                fout0.print(" Data possibly Gaussian: ", this.field0);
                fout0.print(swWtestA, this.field);
                if(found){
                    fout0.println(swWtestB);
                }
                else fout0.println(); 
                fout0.println();
                
            fout0.println("Probability plot:");
            fout0.print(" Correlation coefficient, r: ", this.field0);
            double corrCoeffA = alD.get(2);
            double corrCoeffB = Double.NaN;
            fout0.print(Fmath.truncate(corrCoeffA, trunc), this.field);
            if(found){
                corrCoeffB = alD.get(6);
                fout0.println(Fmath.truncate(corrCoeffB, trunc));
            }
            else fout0.println(); 
            fout0.println(" Critical value for r: ");
            fout0.print("  (" + this.significance*100 + "% Significance level)", this.field0);
            double critValueA = alD.get(3);
            double critValueB = Double.NaN;
            fout0.print(Fmath.truncate(critValueA, trunc), this.field);
            if(found){
                critValueB = alD.get(7);
                fout0.println(Fmath.truncate(critValueB, trunc));
            }
            else fout0.println(); 
            fout0.print(" Data possibly Gaussian: ", this.field0);
            String gTest = "Rejected";
            if(corrCoeffA>=critValueA)gTest = "Accepted";
            fout0.print(gTest, this.field);
            if(found){
                gTest = "Rejected";
                if(corrCoeffB>=critValueB)gTest = "Accepted";
                fout0.println(gTest);
            }
            else fout0.println(); 
            fout0.print(" Gradient: ", this.field0);
            fout0.print(Fmath.truncate(alD.get(0), trunc), this.field);
            if(found)fout0.println(Fmath.truncate(alD.get(4), trunc));
            else fout0.println();
            fout0.print(" Intercept: ", this.field0);
            fout0.print(Fmath.truncate(alD.get(1), trunc), this.field);
            if(found)fout0.println(Fmath.truncate(alD.get(5), trunc));
            else fout0.println();
            
            
            fout0.println();
            fout0.println("Input data: ");
            int kk = 0;
            for(int i=0; i<this.nPoints; i++){
                    fout0.print(this.data[i] + "  ");
                    kk++;
                    if(kk==10){
                        fout0.println();
                        kk = 0;
                    }
            }
            fout0.close();
        }
        
        
        // Strip data of any outliers
        public double[] strippedData(boolean outliersFound, int possibleOutlierIndex){
            int[] possibleOutlierIndices = {possibleOutlierIndex};
            return this.strippedData(outliersFound, 1, possibleOutlierIndices);

        }
        
        public double[] strippedData(boolean outliersFound, int nPossibleOutliers, int[] possibleOutlierIndices){
            
            this.strippedData = null;
            if(outliersFound){
                this.strippedData = new double[this.nPoints - nPossibleOutliers];
                int k = 0;
                for(int i=0; i<this.nPoints; i++){
                    boolean test = true;
                    for(int j=0; j<nPossibleOutliers; j++)if(i==possibleOutlierIndices[j])test = false;
                    if(test){
                        this.strippedData[k] = this.data[i];
                        k++;  
                    }
                }
            }
            else{
                this.strippedData = DeepCopy.copy(this.data);
            }
            return this.strippedData;
        }
        
        
        // Probability Plots
        public ArrayList<Double> probabilityPlots(boolean found, double[] dataOrig, double[] dataRemoved){
            ArrayList<Double> ret = new ArrayList<Double>();
            ProbabilityPlot pp = new ProbabilityPlot(dataOrig);
            pp.setStartOfGraphTitle("Original data: ");
            pp.resetSignificance(this.significance);
            pp.suppressFileOutput();
            pp.suppressErrorMessages();
            if(this.suppressDisplay)pp.suppressDisplay();
            pp.gaussianProbabilityPlot();
            ret.add(pp.gaussianGradient());
            ret.add(pp.gaussianIntercept());
            ret.add(pp.gaussianCorrelationCoefficient());
            ret.add(pp.correlationCoefficientCriticalValue());
            this.orderStatisticMedians = pp.gaussianOrderStatisticMedians();
            this.strippedOrderStatisticMedians = this.orderStatisticMedians;
            
            if(found){
                pp = new ProbabilityPlot(dataRemoved);
                pp.setStartOfGraphTitle("Stripped data: ");
                pp.resetSignificance(this.significance);
                pp.suppressFileOutput();
                pp.suppressErrorMessages();
                if(this.suppressDisplay)pp.suppressDisplay();
                pp.gaussianProbabilityPlot();
                ret.add(pp.gaussianGradient());
                ret.add(pp.gaussianIntercept());
                ret.add(pp.gaussianCorrelationCoefficient());
                ret.add(pp.correlationCoefficientCriticalValue());
                this.strippedOrderStatisticMedians =  pp.gaussianOrderStatisticMedians();
            }
            return ret;
        }
        
        // Shapiro-Wilk W Test
        public ArrayList<Object> shapiroWilkTest(double[] datasw){
            ArrayList<Object> alsw = new ArrayList<Object>();
            Normality norm = new Normality(datasw);
            double wValue = norm.shapiroWilkWvalue();
            alsw.add(wValue);
            double wCrit = norm.shapiroWilkCriticalW();
            alsw.add(wCrit);
            double wPvalue = norm.shapiroWilkPvalue();
            alsw.add(wPvalue);
            String ar = "Rejected";
            if(wValue>=wCrit)ar = "Accepted";
            alsw.add(ar);
            return alsw;    
        }
        
}