/*
*   Class   Normality
*
*   USAGE:  Methods for examining departure from normality
*
*   WRITTEN BY: Dr Michael Thomas Flanagan
*
*   DATE:    28-31 March 2015
*   AMENDED: 
*
*   DOCUMENTATION:
*   See Michael Thomas Flanagan's Java library on-line web page:
*   http://www.ee.ucl.ac.uk/~mflanaga/java/Normality.html
*   http://www.ee.ucl.ac.uk/~mflanaga/java/
*
*   Copyright (c) 2015 Michael Thomas Flanagan
*
*   PERMISSION TO COPY:
*
* Permission to use, copy and modify this software and its documentation for NON-COMMERCIAL purposes is granted, without fee,
* provided that an acknowledgement to the author, Dr Michael Thomas Flanagan at www.ee.ucl.ac.uk/~mflanaga, appears in all copies
* and associated documentation or publications.
*
* Public listing of the source codes on the internet is not permitted.
*
* Redistribution of the source codes or of the flanagan.jar file is not permitted.
*
* Redistribution in binary form of all or parts of these classes is not permitted.
*
* Dr Michael Thomas Flanagan makes no representations about the suitability or fitness of the software for any or for a particular purpose.
*
* Dr Michael Thomas Flanagan shall not be liable for any damages suffered as a result of using, modifying or distributing this software
* or its derivatives.
*
***************************************************************************************/

package flanagan.analysis;

import java.util.ArrayList;
import java.math.*;

import flanagan.analysis.ProbabilityPlot;
import flanagan.analysis.Stat;
import flanagan.io.FileChooser;
import flanagan.io.FileOutput;
import flanagan.io.PrintToScreen;
import flanagan.math.ArrayMaths;
import flanagan.math.Fmath;
import flanagan.math.TimeAndDate;
import flanagan.util.Strings;


public class Normality{
    
    private double[] data = null;                               // entered data
    private double[] orderedData = null;                        // data sorted into ascending order
    private int[] originalindices = null;                       // indices of unsorted data in order of sorted data
    private double[] orderStatisticMedians = null;              // order statistic medians 
 
    private int nPoints = 0;                                    // number of data points
    
    private double mean = Double.NaN;                           // sample mean of the data
    private double standardDeviation = Double.NaN;              // sample standard deviation of the data
    private double median = Double.NaN;                         // sample median
    private double momentSkewness = Double.NaN;                 // moment skewness 
    private double medianSkewness = Double.NaN;                 // median skewness 
    private double quartileSkewness = Double.NaN;               // quartile skewness 
    
    private double kurtosis = Double.NaN;                       // kurtosis
    private double curtosis = Double.NaN;                       // kurtosis
    private double excessKurtosis = Double.NaN;                 // excess kurtosis
    private double excessCurtosis = Double.NaN;                 // excess kurtosis
    
    private double maximum = Double.NaN;                        // maximum value
    private double minimum = Double.NaN;                        // minimum value
    private double range = Double.NaN;                          // range
    
  
       
    private double sumOfSquares = Double.NaN;                   // sum of squares of the data
    
    private double[] shapiroWilkCoeff = null;                   // Shapiro-Wilk coefficients
    private int nSWcoeff = 0;                                   // number of Shapiro-Wilk coefficients
    private boolean swCoeffDone = false;                        // = true when Shapiro-Wilk coefficients calculated
    private double shapiroWilkWvalue = Double.NaN;              // Shapiro-Wilk W value
    private boolean swWvalueDone = false;                       // = true when Shapiro-Wilk W value calculated
    private double shapiroWilkPvalue = Double.NaN;              // Shapiro-Wilk P value
    private boolean swPvalueDone = false;                       // = true when Shapiro-Wilk P value calculated
    private double[] swWvalues = null;                          // Shapiro-Wilk W value obtained by iteration
    private double shapiroWilkCriticalWvalue = Double.NaN;      //Shapiro-Wilk critical W value at default significance level
    private boolean swPCritDone = false;                        // = true when Shapiro-Wilk Critical value at default significance calculated
    
    private double significance = 0.05;                         // default significance level
    private int nSimulations = 10000;                           // number of iterations in simulation calculation of P values
    
    private ProbabilityPlot pp = null;                          // instance of ProbabilityPlot
    
    private String filenameout = "NormalityAnalysis.txt";       // Full analysis output file name
    private String filenamein = null;                           // intput file name if data read from file
    private TimeAndDate tad = new TimeAndDate();                // instance of TimeAndDate
    private String time_date;                                   // time and date of program run
    private int trunc = 4;                                      // output truncation value
    private int field0 = 35;                                    // field for 1st column in text files
    private int field = 15;                                     // field for all but the 1st column in text files
       

    
    // Constructors
    public Normality(double[] data){
        this.data = data;
        this.initialise();
    }
    
    public Normality(float[] data){
        ArrayMaths am0 = new ArrayMaths(data);
        this.data = am0.array();
        this.initialise();
    }
    
    public Normality(long[] data){
        ArrayMaths am0 = new ArrayMaths(data);
        this.data = am0.array();
        this.initialise();
    }
    
     public Normality(int[] data){
        ArrayMaths am0 = new ArrayMaths(data);
        this.data = am0.array();
        this.initialise();
    }
     
    public Normality(BigDecimal[] data){
        ArrayMaths amc = new ArrayMaths(data);
        this.data = amc.array_as_double();
        this.initialise();
    }
        
    public Normality(BigInteger[] data){
        ArrayMaths amc = new ArrayMaths(data);
        this.data = amc.array_as_double();
        this.initialise();
    }
     
    public Normality(){
        this.data = null;
    }
     
             
    // Read from text file
    public void readDataFromTextFile(){
            
            FileChooser fin0 = new FileChooser();
            this.filenamein = fin0.selectFile();
            ArrayList<Double> alS = new ArrayList<Double>();
            int nLines = fin0.numberOfLines();
            for(int i=0; i<nLines; i++){
                String line = fin0.readLine();
                line = this.replacements(line);
                String[] tokens = Strings.tokens(line);
                int nTokens = tokens.length;
                for(int j=0; j<nTokens; j++){
                    double holdD = Double.parseDouble(tokens[j]);
                    alS.add(holdD);
                }
            }
            fin0.close();
            
            this.nPoints = alS.size();
            this.data = new double[this.nPoints];
            for(int i=0; i<this.nPoints; i++)this.data[i] = alS.get(i);   
            this.initialise();
    }
        
    // Line replacements
    public String replacements(String line){
            int nChar = line.length();
            char[] cline = new char[nChar];
            String rline = "";
            for(int i=0; i<nChar; i++){
                cline[i] = line.charAt(i);
                if(cline[i]==',')cline[i] = ' ';
                if(cline[i]==';')cline[i] = ' ';
                if(cline[i]==':')cline[i] = ' ';
                if(cline[i]=='\t')cline[i] = ' ';   
                rline += cline[i];
            } 
            return rline;
    }
    
    // Initialisation
    private void initialise(){
        this.nPoints = data.length;
        ArrayMaths am1 = new ArrayMaths(this.data);
        am1 = am1.sort();
        this.orderedData = am1.array();
        this.originalindices = am1.originalIndices();
        this.mean = Stat.mean(this.data);
        this.median = Stat.median(this.data);
        this.minimum = Fmath.minimum(this.data);
        this.maximum = Fmath.maximum(this.data);
        this.range = this.maximum - this.minimum;
        this.standardDeviation = Stat.standardDeviation(this.data);
        this.momentSkewness = Stat.momentSkewness(this.data); 
        this.medianSkewness = Stat.medianSkewness(this.data); 
        this.quartileSkewness = Stat.quartileSkewness(this.data); 
        this.kurtosis = Stat.kurtosis(this.data);
        this.curtosis = Stat.kurtosis(this.data);
        this.excessKurtosis = Stat.excessKurtosis(this.data);
        this.excessCurtosis = Stat.excessKurtosis(this.data);
        this.sumOfSquares = 0.0;
        for(int i=0; i<this.nPoints; i++){
            double hold = this.data[i] - this.mean;
            this.sumOfSquares += hold*hold;
        }
        this.pp = new ProbabilityPlot(this.data);
        pp.suppressErrorMessages();
        pp.suppressFileOutput();
        this.time_date = this.tad.getShortTime24() + ", ";
        this.time_date += tad.getDate();
    }
    
    // Reset the significance level
    public void resetSignificanceLevel(double significance){
        this.significance = significance;
    }
        
    // Get the significance level
    public double getSignificanceLevel(){
        return this.significance;
    }
       
    // Return mean
    public double mean(){
        return this.mean;
    }
    
    // Return standard deviation
    public double standardDeviation(){
        return this.standardDeviation;
    }
    
    // Return moment skewness
    public double momentSkewness(){
        return this.momentSkewness;
    }
    
    // Return median skewness
    public double medianSkewness(){
        return this.medianSkewness;
    }
    
    // Return quartile skewness
    public double quartileSkewness(){
        return this.quartileSkewness;
    }
    
    // Return kurtosis
    public double kurtosis(){
        return this.kurtosis;
    }
    
    // Return curtosis
    public double curtosis(){
        return this.curtosis;
    }
    
    // Return excess kurtosis
    public double excessKurtosis(){
        return this.excessKurtosis;
    }
    
    // Return excess curtosis
    public double excessCurtosis(){
        return this.excessCurtosis;
    }
    
    // DATA
    // Original data as double[]
    public double[] getData(){
        return this.data;
    }
        
    // Ordered original data as double[]
    public double[] getOrderedOriginalData(){
        return this.orderedData;
    }
        
    //  Original Data Gaussian Order Statistic Medians
    public double[] getDataOrderStatisticMedians(){
        return this.orderStatisticMedians;
    }
  
    // SHAPIRO-WILK W TEST
    
    // Returns Shapiro-Wilk W value
    public double shapiroWilkWvalue(){
        if(!this.swCoeffDone)this.shapiroWilkCoeff();
        double swB = 0;
        for(int i=0; i<this.nSWcoeff; i++){
            int ind = this.nPoints - i - 1;
            swB += this.shapiroWilkCoeff[i]*(this.orderedData[ind] - this.orderedData[i]);  
        }
        swB = swB*swB;
        
        this.shapiroWilkWvalue = swB/this.sumOfSquares;
        this.swWvalueDone = true;
        return this.shapiroWilkWvalue;
    }
    
    // Returns Shapiro-Wilk p value
    public double shapiroWilkPvalue(){
        if(!this.swCoeffDone)this.shapiroWilkCoeff();
        if(!this.swWvalueDone)this.shapiroWilkWvalue();
        this.swWvalues = new double[this.nSimulations];
        double[] sData = new double[this.nPoints];
        int above = 0;
        for(int i=0; i<this.nSimulations; i++){
            sData = Stat.gaussianRand(this.mean, this.standardDeviation, this.nPoints);
            Normality normS = new Normality(sData);
            this.swWvalues[i] = normS.shapiroWilkWvalue();
            if(this.shapiroWilkWvalue>this.swWvalues[i])above++; 
        }
        ArrayMaths amS = new ArrayMaths(this.swWvalues);
        this.swWvalues = (amS.sort()).array();
        this.shapiroWilkPvalue = (double)above/(double)nSimulations;
        this.swPvalueDone = true;
        return this.shapiroWilkPvalue;
    }
    
    // Returns Shapiro-Wilk critical W value
    // Default significance level
    public double shapiroWilkCriticalW(){
        if(!this.swCoeffDone)this.shapiroWilkCoeff();
        if(!this.swWvalueDone)this.shapiroWilkWvalue();
        if(!this.swPvalueDone)this.shapiroWilkPvalue();
        int indx = (int)Math.round((double)this.nSimulations*this.significance);
        this.shapiroWilkCriticalWvalue = this.swWvalues[indx];
        this.swPCritDone = true; 
        return this.shapiroWilkCriticalWvalue;
    }
    
    // Returns Shapiro-Wilk critical W value
    // Entered significance level
    public double shapiroWilkCriticalW(double signif){
        if(!this.swCoeffDone)this.shapiroWilkCoeff();
        if(!this.swWvalueDone)this.shapiroWilkWvalue();
        if(!this.swPvalueDone)this.shapiroWilkPvalue();
        int indx = (int)Math.round((double)this.nSimulations*signif);
        return this.swWvalues[indx];
    }
    
    // Returns Shapiro-Wilk critical W value
    // Entered significance level and number of observation
    public double shapiroWilkCriticalW(double signif, int nObservations){
        double shapiroWilkPvalueS = Double.NaN;
        if(nObservations==this.nPoints){
            shapiroWilkPvalueS = this.shapiroWilkCriticalW(signif);
        }
        else{
            double[] swWvaluesS = new double[this.nSimulations];
            double[] sDataS = new double[nObservations];
            for(int i=0; i<this.nSimulations; i++){
                sDataS = Stat.gaussianRand(0.0, 1.0, nObservations);
                Normality normS = new Normality(sDataS);
                swWvaluesS[i] = normS.shapiroWilkWvalue();
            }
            ArrayMaths amS = new ArrayMaths(swWvaluesS);
            swWvaluesS = (amS.sort()).array();
            int indx = (int)Math.round((double)this.nSimulations*signif);
            shapiroWilkPvalueS = swWvaluesS[indx]; 
        }
        return shapiroWilkPvalueS;
    }
    
    // Returns the Shapiro-Wilk coefficients for the number of data points entered via the constructor
    // Java version of the coefficient calculation section of Algorithm AS R94 Appl. Statist. (1995) Vol.44, NO.4
    // Method of Patrick Royston, Statistics and Computing (1992) 2, 117-119
    public double[] shapiroWilkCoeff(){
        return this.shapiroWilkCoeff(this.nPoints);
    }
     
    // Returns the Shapiro-Wilk coefficients for n observations
    // Java version of the coefficient calculation section of Algorithm AS R94 Appl. Statist. (1995) Vol.44, NO.4
    // Method of Patrick Royston, Statistics and Computing (1992) 2, 117-119
    public double[] shapiroWilkCoeff(int n){
       
        int an = n; 
        int n2 = n/2;
        this.nSWcoeff = n2;
        if(n<2)n2 = 1;
        double[] a = new double[n2+1];
        a[0] = Double.NaN;
        int i1;
        double summ2 = 0.0, ssumm2 = 0.0;
        double fac, rsn, an25, a1, a2;
        
        double[] c1 = {Double.NaN, 0.0E0, 0.221157E0, -0.147981E0, -0.207119E1, 0.4434685E1, -0.2706056E1};
        double[] c2 = {Double.NaN, 0.0E0, 0.42981E-1, -0.293762E0, -0.1752461E1, 0.5682633E1, -0.3582633E1};

        double zero = 0.0;
        double one = 1.0;
        double two = 2.0;
        double sqrth = 0.70711E0;
        double qtr = 0.25E0;
        double th = 0.375E0;

        if(n<2){
            a[1] = 0.0;
        }
        else{
            if(n==2 || n==3){
                a[1] = sqrth;
            }
            else{
                an25 = an + qtr;
                summ2 = zero;
                for(int i=1; i<=n2; i++){
                    a[i] = this.ppnd(((double)i - th)/an25);
                    summ2 = summ2 + Math.pow(a[i],2);
                }               
                summ2 = summ2 * two;
                ssumm2 = Math.sqrt(summ2);
                rsn = one/Math.sqrt(an);
                a1 = this.poly(c1, 6, rsn) - a[1]/ssumm2;

                //  Normalize coefficients

                if(n>5){
                    i1 = 3;
                    a2 = -a[2]/ssumm2 + this.poly(c2,6,rsn);
                    fac = Math.sqrt((summ2 - two*Math.pow(a[1],2) - two*Math.pow(a[2],2))/(one - two*Math.pow(a1,2) - two*Math.pow(a2,2)));
                    a[1] = a1;
                    a[2] = a2;
                }
                else{
                    i1 = 2;
                    fac = Math.sqrt((summ2 - two*Math.pow(a[1],2))/(one - two*Math.pow(a1,2)));
                    a[1] = a1;
                }
                for(int i=i1; i<=n2; i++){
                    a[i] = -a[i]/fac;
                }
            }
        }
        this.shapiroWilkCoeff = this.shiftDown(a);
        this.swCoeffDone = true;
        return this.shapiroWilkCoeff;
    }
    
    // Produces normal deviate corresponding to lower tail area = p
    // Java version of Algorithm AS 111, Appl. Statist., Vol.26, 118-121, 1977.
    public double ppnd(double p){
      
        double split = 0.42;
        double a0 = 2.50662823884;
        double a1 = -18.61500062529; 
        double a2 = 41.39119773534; 
        double a3 = -25.44106049637;
                
        double b1 = -8.47351093090; 
        double b2 = 23.08336743743; 
        double b3 = -21.06224101826; 
        double b4 = 3.13082909833; 

        double c0 = -2.78718931138;
        double c1 = -2.29796479134;
        double c2 = 4.85014127135;
        double c3 = 2.32121276858;
        
        double d1 = 3.54388924762;
        double d2 = 1.63706781897;
     
        double zero = 0.0;
        double one = 1.0;
        double half = 0.5;

        int ier = 0;
        
        double q = p - half;
        boolean goto10 = false;
        if(Math.abs(q)>split)goto10 = true;
        if(!goto10){
            // 0.08 < p < 0.92
            double r = q*q;
            double ppnd = q*(((a3*r + a2)*r + a1)*r + a0)/((((b4*r + b3)*r + b2)*r + b1)*r + one);
            return ppnd;
        }
        
        // p < 0.08 or p > 0.92, Set r = min(p,1-p)
        double r = p;
        if(q>zero) r = one-p;
        boolean goto20 = false;
        if(r<=zero)goto20 = true;
        if(!goto20){
            r = Math.sqrt(-Math.log(r));
            double ppnd = (((c3*r + c2)*r + c1)*r + c0)/((d2*r + d1)*r + one);
            if(q<zero) ppnd = -ppnd;
            return ppnd;
        }  
        
        double ppnd = zero;
        return ppnd;
    }

    //  Calculates the algebraic polynomial of order nored-1 with array of coefficients c.  Zero order coefficient is c(1)
    //  Java version of Algorithm AS 181.2   Appl. Statist.  (1982) Vol. 31, No. 2
    public double poly(double[] c, int nord, double x){

        double poly = c[1];
        if(nord==1) return poly;
        double p = x*c[nord];
        boolean goto20 = false;
        if(nord==2)goto20 = true;
        if(!goto20){
            int n2 = nord-2;
            int j = n2+1;
            for(int i=1; i<=n2; i++){
                p = (p+c[j])*x;
                j = j-1;
            }
        }
        poly = poly + p;
        return poly;
    }
    
    // Evaluates the tail area of the standardised normal curve from x to infinity if upper is .true. or from minus infinity to x if upper is .false.
    // Java version of Algorithm AS66 Applied Statistics (1973) vol22 no.3
    public double  alnorm(double x, boolean upper){

        double z,y;
        boolean up;
        
        double zero =0.0; 
        double one = 1.0;
        double half = 0.5;
        double ltone = 7.0;
        double utzero = 18.66;
        double con = 1.28;
        double p = 0.398942280444;
        double q = 0.39990348504;
        double r = 0.398942280385;  
        double a1 = 5.75885480458;
        double a2 = 2.62433121679;
        double a3 = 5.92885724438;  
        double b1 = -29.8213557807;
        double b2 = 48.6959930692;
        double c1 = -3.8052e-8;
        double c2 = 3.98064794e-4;
        double c3 = -0.151679116635;
        double c4 = 4.8385912808;
        double c5 = 0.742380924027;
        double c6 = 3.99019417011; 
        double d1 = 1.00000615302;
        double d2 = 1.98615381364;
        double d3 = 5.29330324926;  
        double d4 = -15.1508972451;
        double d5 = 30.789933034;

        up = upper;
        z = x;
        boolean goto10 = false;
        if(z>=zero)goto10 = true;
        if(!goto10){
            if(up==false){
                up = true;
            }
            else{
                up=true;
            }
            z = -z;
        }
        double alnorm = 0.0;
        boolean goto40 = false;
        boolean goto30 = false;
        boolean goto20 = false;
        if(z<=ltone || up && z<=utzero)goto20 = true;
        if(!goto20){
            alnorm = zero;
            goto40 = true;
        }
        if(!goto40){
            y = half*z*z;
            if(z>con)goto30 = true;
            if(!goto30){
                alnorm=half-z*(p-q*y/(y+a1+b1/(y+a2+b2/(y+a3))));
                goto40 = true;
            }
            if(!goto40){
                alnorm=r*Math.exp(-y)/(z+c1+d1/(z+c2+d2/(z+c3+d3/(z+c4+d4/(z+c5+d5/(z+c6))))));
            }
        }
        if(!up)alnorm = one-alnorm;
        return alnorm;
    }
      
    
    // returns the absolute value of a times the sign of b
    public double swapSign(double a, double b){
        double c = Math.abs(a);
        if(b<0)c = -c;
        return c;
    }
    
    public int swapSign(int a, int b){
        int c = Math.abs(a);
        if(b<0)c = -c;
        return c;
    }
     
    // Shift to convert array starting at index 0 to an array starting at index 1
    public double[] shiftUp(double[] a){
        int n = a.length;
        double[] b = new double[n+1];
        b[0] = Double.NaN;
        for(int i=0; i<n; i++)b[i+1] = a[i];
        return b;
    }
    
    // Shift to convert array starting at index 1 to an array starting at index 0
    public double[] shiftDown(double[] a){
        int n = a.length;
        double[] b = new double[n-1];
        for(int i=0; i<n-1; i++)b[i] = a[i+1];
        return b;
    }
    
    // Reset number of Shapiro-Wilk critical W iterations
    public void resetNsimulation(int nSimul){
        this.nSimulations = nSimul;
    }
    
    // Reset number of Shapiro-Wilk critical W iterations
    public int getNsimulation(){
        return this.nSimulations;
    }
    
    
    // GAUSSIAN PROBABILITY PLOT
    // user supplied initial estimates
    public void gaussianUserSuppliedInitialEstimates(double mue, double sigmae){
         this.pp.gaussianUserSuppliedInitialEstimates(mue, sigmae);
    }
       
    public void normalUserSuppliedInitialEstimates(double mue, double sigmae){
        this.pp.normalUserSuppliedInitialEstimates(mue, sigmae);
    }
        
    // Remove user supplied initial estimates
    public void removeGaussianUserSuppliedInitialEstimates(){
        this.pp.removeGaussianUserSuppliedInitialEstimates();
    }
        
    public void removeNormalUserSuppliedInitialEstimates(){
        this.pp.removeNormalUserSuppliedInitialEstimates();
    }
        
    // Gaussian probability plot
    public void gaussianProbabilityPlot(){
        this.pp.gaussianProbabilityPlot();
    }
    
    public void normalProbabilityPlot(){
        this.pp.normalProbabilityPlot();
    }

    // Return Gaussian mu
    public double gaussianMu(){
        return this.pp.gaussianMu();
    }

    // Return Gaussian mu error
    public double gaussianMuError(){
        return this.pp.gaussianMuError();
    }

    // Return Gaussian sigma
    public double gaussianSigma(){
        return this.pp.gaussianSigma();
    }
    
    // Return Gaussian sigma error
    public double gaussianSigmaError(){
        return this.pp.gaussianSigmaError();
    }

    // Return the Gaussian gradient
    public double gaussianGradient(){
        return this.pp.gaussianGradient();
    }

    // Return the error of the Gaussian gradient
    public double gaussianGradientError(){
        return this.pp.gaussianGradientError();
    }

    // Return the Gaussian intercept
    public double gaussianIntercept(){
        return this.pp.gaussianIntercept();
    }

    // Return the error of the Gaussian intercept
    public double gaussianInterceptError(){
        return this.pp.gaussianInterceptError();
    }

    // Return the Gaussian correlation coefficient
    public double gaussianCorrelationCoefficient(){
        return this.pp.gaussianCorrelationCoefficient();
    }

    // Return the sum of squares at the Gaussian minimum
    public double gaussianSumOfSquares(){
        return this.pp.gaussianSumOfSquares();
    }

    // Return the Weighted sum of squares at the Gaussian minimum
    public double gaussianWeightedSumOfSquares(){
        return this.pp.gaussianWeightedSumOfSquares();
    }

    // Return Gaussian order statistic medians
    public double[] gaussianOrderStatisticMedians(){
        return this.pp.gaussianOrderStatisticMedians();
    }

    // Return Gaussian mu
    public double normalMu(){
        return this.pp.normalMu();
    }

    // Return Gaussian mu error
    public double normalMuError(){
        return this.pp.normalMuError();
    }

    // Return Gaussian sigma
    public double normalSigma(){
        return this.pp.normalSigma();
    }
    
    // Return Gaussian sigma error
    public double normalSigmaError(){
        return this.pp.normalSigmaError();
    }

    // Return the Gaussian gradient
    public double normalGradient(){
        return this.pp.normalGradient();
    }

    // Return the error of the Gaussian gradient
    public double normalGradientError(){
        return this.pp.normalGradientError();
    }

    // Return the Gaussian intercept
    public double normalIntercept(){
        return this.pp.normalIntercept();
    }

    // Return the error of the Gaussian intercept
    public double normalInterceptError(){
        return this.pp.normalInterceptError();
    }

    // Return the Gaussian correlation coefficient
    public double normalCorrelationCoefficient(){
        return this.pp.normalCorrelationCoefficient();
    }

    // Return the sum of squares at the Gaussian minimum
    public double normalSumOfSquares(){
        return this.pp.normalSumOfSquares();
    }

    // Return the Weighted sum of squares at the Gaussian minimumme
    public double normalWeightedSumOfSquares(){
        return this.pp.normalWeightedSumOfSquares();
    }

    // Return Gaussian order statistic dians
    public double[] normalOrderStatisticMedians(){
        return this.pp.normalOrderStatisticMedians();
    }

    // STANDARD GAUSSIAN PROBABILITY PLOT
    public void gaussianStandardProbabilityPlot(){
        this.pp.gaussianStandardProbabilityPlot();
    }
   
    public void normalStandardProbabilityPlot(){
        this.pp.normalStandardProbabilityPlot();
    }

    // Return the Standard Gaussian gradient
    public double gaussianStandardGradient(){
        return this.pp.gaussianStandardGradient();
    }

    // Return the error of the Standard Gaussian gradient
    public double gaussianStandardGradientError(){
        return this.pp.gaussianStandardGradientError();
    }

    // Return the Standard Gaussian intercept
    public double gaussianStandardIntercept(){
        return this.pp.gaussianStandardIntercept();
    }

    // Return the error of the Standard Gaussian intercept
    public double gaussianStandardInterceptError(){
        return this.pp.gaussianStandardInterceptError();
    }

    // Return the Standard Gaussian correlation coefficient
    public double gaussianStandardCorrelationCoefficient(){
        return this.pp.gaussianStandardCorrelationCoefficient();
    }

    // Return the sum of squares at the Standard Gaussian minimum
    public double gaussianStandardSumOfSquares(){
        return this.pp.gaussianStandardSumOfSquares();
    }

    // Return Standard Gaussian order statistic medians
    public double[] gaussianStandardOrderStatisticMedians(){
        return this.pp.gaussianStandardOrderStatisticMedians();
    }

    // Return the Standard Gaussian gradient
    public double normalStandardGradient(){
        return this.pp.normalStandardGradient();
    }

    // Return the error of the Standard Gaussian gradient
    public double normalstandardGradientError(){
        return this.pp.normalstandardGradientError();
    }

    // Return the Standard Gaussian intercept
    public double normalStandardInterceptError(){
        return this.pp.normalStandardInterceptError();
    }

    // Return the Standard Gaussian correlation coefficient
    public double normalStandardCorrelationCoefficient(){
        return this.pp.normalStandardCorrelationCoefficient();
    }

    // Return the sum of squares at the Standard Gaussian minimum
    public double normalStandardSumOfSquares(){
        return this.pp.normalStandardSumOfSquares();
    }
        
    // Return the weighted sum of squares at the Standard Gaussian minimum
    public double normalStandardWeightedSumOfSquares(){
        return this.pp.normalStandardWeightedSumOfSquares();
    }

    // Return Standard Gaussian order statistic medians
    public double[] normalStandardOrderStatisticMedians(){
        return this.pp.normalStandardOrderStatisticMedians();
    }
    
    // FULL ANALYSIS
    public void fullAnalysis(){
        this.fullAnalysis(this.filenameout);        
    }

    public void fullAnalysis(String filename){
        int p0 = filename.indexOf(".");
        if(p0==-1)filename += ".txt";
        this.filenameout = filename;
         
        FileOutput fout0 = new FileOutput(this.filenameout);
        fout0.println("Normality Analysis");
        if(this.filenamein!=null)fout0.println("Input file: " + this.filenamein);
        fout0.println();
        fout0.println("File name: " + filenameout);
        fout0.println("Run date: " + this.time_date);
        fout0.println();
        
        fout0.print("Parameter / statistic", field0);
        fout0.print("Value", field);
        fout0.println("Error");
        fout0.print("Number of data points: ", field0);
        fout0.println(this.nPoints);
        fout0.print("Sample mean: ", field0);
        fout0.println(Fmath.truncate(this.mean, trunc));
        fout0.print("Sample median: ", field0);
        fout0.println(Fmath.truncate(this.median, trunc));
        fout0.print("Sample minimum: ", field0);
        fout0.println(Fmath.truncate(this.minimum, trunc));
        fout0.print("Sample maximum: ", field0);
        fout0.println(Fmath.truncate(this.maximum, trunc));
        fout0.print("Sample range: ", field0);
        fout0.println(Fmath.truncate(this.range, trunc));
        fout0.print("Sample standard deviation: ", field0);
        fout0.println(Fmath.truncate(this.standardDeviation, trunc));
        fout0.print("Sample moment skewnesss: ", field0);
        fout0.println(Fmath.truncate(this.momentSkewness, trunc));
        fout0.print("Sample median skewnesss: ", field0);
        fout0.println(Fmath.truncate(this.medianSkewness, trunc));
        fout0.print("Sample quartile skewnesss: ", field0);
        fout0.println(Fmath.truncate(this.quartileSkewness, trunc));
        fout0.print("Sample kurtosis: ", field0);
        fout0.println(Fmath.truncate(this.kurtosis, trunc));
        fout0.print("Sample excess kurtosis: ", field0);
        fout0.println(Fmath.truncate(this.excessKurtosis, trunc));
        fout0.println();
        fout0.println("Shapiro-Wilk W test");
        if(!this.swWvalueDone)this.shapiroWilkWvalue = this.shapiroWilkWvalue();
        fout0.print(" Shapiro-Wilk W value: ", field0);
        fout0.println(Fmath.truncate(this.shapiroWilkWvalue, trunc));
        if(!this.swPCritDone)this.shapiroWilkCriticalWvalue = this.shapiroWilkCriticalW();
        fout0.print(" Shapiro-Wilk W critical value: ", field0);
        fout0.println(Fmath.truncate(this.shapiroWilkCriticalWvalue, trunc));
        if(!this.swPvalueDone)this.shapiroWilkPvalue  = this.shapiroWilkPvalue();
        fout0.print(" Shapiro-Wilk P value: ", field0);
        fout0.println(Fmath.truncate(this.shapiroWilkPvalue , trunc));
        String accrej = "Accepted";
        if(this.shapiroWilkWvalue<this.shapiroWilkCriticalWvalue)accrej = "Rejected";
        fout0.print(" Data possibly normal: ", field0);
        fout0.println(accrej);
        fout0.println();
        

        fout0.println("Probability plot:");
        pp.gaussianProbabilityPlot();
        fout0.print(" Correlation coefficient, r: ", this.field0);
        double corrCoeff = pp.gaussianCorrelationCoefficient();
        fout0.println(Fmath.truncate(corrCoeff , trunc));            
        fout0.println(" Critical value for r: ");
        double critValue = pp.correlationCoefficientCriticalValue();
        fout0.print("  (" + this.significance*100 + "% Significance level)", this.field0);
        fout0.println(Fmath.truncate(critValue, trunc));
        fout0.print(" Data possibly normal: ", this.field0);
        String gTest = "Rejected";
        if(corrCoeff>=critValue)gTest = "Accepted";
        fout0.println(gTest);
        fout0.print(" Gradient: ", this.field0);
        fout0.print(Fmath.truncate(pp.gaussianGradient(), trunc), this.field);
        fout0.println(Fmath.truncate(pp.gaussianGradientError(), trunc));
        fout0.print(" Intercept: ", this.field0);
        fout0.print(Fmath.truncate(pp.gaussianIntercept(), trunc), this.field);
        fout0.println(Fmath.truncate(pp.gaussianInterceptError(), trunc));
        fout0.print(" Mean (best estimate): ", this.field0);
        fout0.print(Fmath.truncate(pp.gaussianMu(), trunc), this.field);
        fout0.println(Fmath.truncate(pp.gaussianMuError(), trunc));
        fout0.print(" Stand. devn. (best estimate): ", this.field0);
        fout0.print(Fmath.truncate(pp.gaussianSigma(), trunc), this.field);
        fout0.println(Fmath.truncate(pp.gaussianSigmaError(), trunc));
                
        fout0.println();
                
        fout0.println("Input data: ");
        int kk = 0;
        for(int i=0; i<this.nPoints; i++){
            fout0.print(this.data[i] + "  ");
            kk++;
            if(kk==10){
            fout0.println();
                kk = 0;
            }
        }
        fout0.close();
    }
        
}