/*
*   Class  Strings
*
*   Methods for editing Strings
*
*   WRITTEN BY: Dr Michael Thomas Flanagan
*
*   DATE:       March 2013
*   REVISED:    April 2013, 2 May 2013, 30 November 2013, 20 July 2015, 15 June to 1 July2020
*
*   DOCUMENTATION:
*   See Michael Thomas Flanagan's Java library on-line web page:
*   http://www.ee.ucl.ac.uk/~mflanaga/java/Strings.html
*   http://www.ee.ucl.ac.uk/~mflanaga/java/
*
*   Copyright (c)2013 Michael Thomas Flanagan
*
*
***************************************************************************************/

package flanagan.util;

import java.util.ArrayList;

import flanagan.math.Fmath;
import flanagan.io.*;

public class Strings{
        
    private String enteredString = null;                        // Entered string
    private String originalString = null;                       // Original entered String
    private String editedString = null;                         // Edited string
    private int nEntered = 0;                                   // Number of characters in the entered string after any html to ASCII conversions
    private int nOriginal = 0;                                  // Number of characters in the original entered string
    private String upperCaseString = null;                      // Upper case string
    private String lowerCaseString = null;                      // Lower case string
    private String allTitleCaseString = null;                   // All title string
    private String titleCaseString = null;                      // Title string
    private String sentenceCaseString = null;                   // Sentence case string
    private String unAccentedString = null;                     // Unaccented string
    private boolean[] whiteSpaces = null;                       // true if corresponding char is a white space
    private String[] tokens = null;                             // tokens separated by white space/s
    private int nTokens = 0;                                    // number of tokens
    private String separator = null;                            // tokens separatop
    private int[] tokenInitialIndices = null;                   // initial index of each token
    private int[] tokenFinalIndices = null;                     // final index of each token
    private boolean tokensDone = false;                         // = true when tokens found
    private String[] principalTokens = null;                    // tokens, other tan the first token, that are not articles, prepostions or conjunctions
    private int nPrincipalTokens = 0;                           // number of principal tokens
    private int[] principalTokenInitialIndices = null;          // initial index of each principal token
    private int[] principalTokenFinalIndices = null;            // final index of each principaltoken
    private boolean suppressDialogue = false;                   // Suppress dialogue indicator
    private boolean tabastab = false;                           //  if false tab replaced by white space
    private boolean tabsReplaced = false;                       // = true if tabs replaced by spaces
    
    // Accent  arrays
    private String[] symbolsA = {"", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", ""};
    private String[] replSymbolsA = {"A", "A", "A", "A", "A", "A", "AE", "C", "E", "E", "E", "E", "I", "I", "I", "I", "DH", "N", "O", "O", "O", "O", "O", "O", "U", "U", "U", "U", "Y", "TH", "ss", "a", "a", "a", "a", "a", "a", "ae", "c", "e", "e", "e", "e", "i", "i", "i", "i", "dh", "n", "o", "o", "o", "o", "o", "o", "u", "u", "u", "u", "y", "th", "y", "OE", "oe", "s", "s", "y", "f"};
    private String[] htmlNamesA = {"&Agrave;", "&Aacute;", "&Acirc;", "&Atilde;", "&Auml;", "&Aring;", "&AElig;", "&Ccedil;", "&Egrave;", "&Eacute;", "&Ecirc;", "&Euml;", "&Igrave;", "&Iacute;", "&Icirc;", "&Iuml;", "&ETH;", "&Ntilde;", "&Ograve;", "&Oacute;", "&Ocirc;", "&Otilde;", "&Ouml;", "&Oslash;", "&Ugrave;", "&Uacute;", "&Ucirc;", "&Uuml;", "&Yacute;", "&THORN;", "&szlig;", "&agrave;", "&aacute;", "&acirc;", "&atilde;", "&auml;", "&aring;", "&aelig;", "&ccedil;", "&egrave;", "&eacute;", "&ecirc;", "&euml;", "&igrave;", "&iacute;", "&icirc;", "&iuml;", "&eth;", "&ntilde;", "&ograve;", "&oacute;", "&ocirc;", "&otilde;", "&ouml;", "&oslash;", "&ugrave;", "&uacute;", "&ucirc;", "&uuml;", "&yacute;", "&thorn;", "&yuml;", "&OE;", "&oe;", "&Scaron;", "&scaron;", "&Ydia;", "&fhook;"}; 
    private String[] htmlNumbersA = {"&#192;", "&#193;", "&#194;", "&#195;", "&#196;", "&#197;", "&#198;", "&#199;", "&#200;", "&#201;", "&#202;", "&#203;", "&#204;", "&#205;", "&#206;", "&#207;", "&#208;", "&#209;", "&#210;", "&#211;", "&#212;", "&#213;", "&#214;", "&#216;", "&#217;", "&#218;", "&#219;", "&#220;", "&#221;", "&#222;", "&#223;", "&#224;", "&#225;", "&#226;", "&#227;", "&#228;", "&#229;", "&#230;", "&#231;", "&#232;", "&#233;", "&#234;", "&#235;", "&#236;", "&#237;", "&#238;", "&#239;", "&#240;", "&#241;", "&#242;", "&#243;", "&#244;", "&#245;", "&#246;", "&#248;", "&#249;", "&#250;", "&#251;", "&#252;", "&#253;", "&#254;", "&#255;", "&#338;", "&#339;", "&#352;", "&#353;", "&#376;", "&#402;"};
    private int[] decNumbersA = {192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 209, 210, 211, 212, 213, 214, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 240, 241, 242, 243, 244, 245, 246, 248, 249, 250, 251, 252, 253, 254, 255, 338, 339, 352, 353, 376, 402}; 
    private int nSymbolsA = symbolsA.length;
    
    // Quotation mark  arrays   
    private String[] symbolQ = {"\"", "'", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", ""};  
    private String[] htmlNamesQ = {"&quot;", "&apos", "&sbquo;", "&dbquo;", "&lsaquo;", "&lsquo;", "&rsquo;", "&ldquo;", "&rdquo;", "&rsaquo;", "&laquo;", "&raquo;", "&none;", "&none;", "&none;", "&none;", "&none;", "&none;"};
    private String[] htmlNumbersQ = {"&#34;", "&#39;", "&#130;", "&#132;", "&#139;", "&#145;", "&#146;", "&#147;", "&#148;", "&#155;", "&#171;", "&#187;", "&#8216;", "&#8217;", "&#8218;", "&#8220;", "&#8221;", "&#8222;"};
    private int[] decNumbersQ = {34, 39, 130, 132, 139, 145, 146, 147, 148, 155, 171, 187, 8216, 8217, 8218, 8220, 8221, 8222};  
    private int nSymbolsQ = symbolQ.length;
    
    // Single quotation mark arrays   
    private String[] symbolSQ = {"'", "", "", "", "",  "", "", "", ""};
    private String[] htmlNamesSQ = {"&apos", "&sbquo;", "&lsaquo;", "&lsquo;", "&rsquo;", "&rsaquo;", "&none;", "&none;", "&none;"};
    private String[] htmlNumbersSQ = {"&#39;", "&#130;", "&#139;", "&#145;", "&#146;",  "&#155;", "&#8216;", "&#8217;", "&#8218;"};           
    private int[] decNumbersSQ = {39, 130, 139, 145, 146,  155, 8216, 8217, 8218}; 
    private int nSymbolsSQ = symbolSQ.length;
    
    //Double quotation mark arrays   
    private String[] symbolDQ = {"\"", "", "", "", "", "", "", "", ""};  
    private String[] htmlNamesDQ = {"&quot;", "&dbquo;", "&ldquo;", "&rdquo;", "&laquo;", "&raquo;", "&none;", "&none;", "&none;"};
    private String[] htmlNumbersDQ = {"&#34;", "&#132;", "&#148;", "&#155;", "&#171;", "&#187;",  "&#8220;", "&#8221;", "&#8222;"};
    private int[] decNumbersDQ = {34, 132, 147, 155, 171, 187, 8220, 8221, 8222};  
    private int nSymbolsDQ = symbolDQ.length;
    
    // Apostrophes  
    private String[] symbolsAp = {"'", "", "", "", ""};
    private String[] htmlNamesAp = {"&apos",  "&lsquo;", "&rsquo;", "&none;", "&none;"};
    private String[] htmlNumbersAp = {"&#39;",  "&#145;", "&#146;",   "&#8216;", "&#8217;"};           
    private int[] decNumbersAp = {39,  145, 146, 8216, 8217}; 
    private int nSymbolsAp = symbolsAp.length;
 
    // dash arrays
    private String[] symbolsD = {"-", "", "", "", "", "", ""};
    private String[] htmlNamesD = {"&none;", "&ndash;", "&mdash;", "", "&oline;", "&none;", "&none;"};
    private String[] htmlNumbersD = {"&#45;", "&#150;", "&#151;", "&#8213;", "&#8254;", "&#8211;", "&#8212;"};
    private int[] decNumbersD = {45, 150, 151, 8213, 8254, 8211, 8212};
    private int nSymbolsD = symbolsD.length;
    
    // Currency arrays
    private String[] symbolsC = {"?", "?", "?", "?", "?", "?", "?", "?", "?", "?", "?", "?", "", "?", "?", "?", "?", "?", "?", "?", "?", "?", "?", "?", "?", "?", "?", "?", "?", "?", "?", "?"};
    private String[] htmlNumbersC = {"&#8352;", "&#8353;", "&#8354;", "&#8355;", "&#8356;", "&#8357;", "&#8358;", "&#8359;", "&#8360;", "&#8361;", "&#8362;", "&#8363;", "&#8364;", "&#8365;", "&#8366;", "&#8367;", "&#8368;", "&#8369;", "&#8370;", "&#8371;", "&#8372;", "&#8373;", "&#8374;", "&#8375;", "&#8376;", "&#8377;", "&#8378;", "&#8379;", "&#8380;", "&#8381;", "&#8382;", "&#8383;"};
    private String[] htmlNamesC = {"&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#euro;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;"};
    private int[] decNumbersC = {8352, 8353, 8354, 8355, 8356, 8357, 8358, 8359, 8360, 8361, 8362, 8363, 8364, 8365, 8366, 8367, 8368, 8369, 8370, 8371, 8372, 8373, 8374, 8375, 8376, 8377, 8378, 8379, 8380, 8381, 8382, 8383};
    private String[] commentsC = {"EURO-CURRENCY SIGN ", "COLON SIGN ", "CRUZEIRO SIGN ", "FRENCH FRANC SIGN ", "LIRA SIGN ", "MILL SIGN ", "NAIRA SIGN ", "PESETA SIGN ", "RUPEE SIGN ", "WON SIGN ", "NEW SHEQEL SIGN ", "DONG SIGN ", "EURO SIGN ", "KIP SIGN ", "TUGRIK SIGN ", "DRACHMA SIGN ", "GERMAN PENNY SYMBOL ", "PESO SIGN ", "GUARANI SIGN ", "AUSTRAL SIGN ", "HRYVNIA SIGN ", "CEDI SIGN ", "LIVRE TOURNOIS SIGN ", "SPESMILO SIGN ", "TENGE SIGN ", "INDIAN RUPEE SIGN ", "TURKISH LIRA SIGN ", "NORDIC MARK SIGN ", "MANAT SIGN ", "RUBLE SIGN ", "LARI SIGN ", "BITCOIN SIGN "};
    private int nSymbolsC = symbolsC.length;
                                                                                                                                                                                                            
   // Greek alphabet and mathematicalsymbols
    private char[] symbolsGM = {'', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '', '', '?', '?', '?', '?', '?', '?', '?', '', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?', '?'};
    private String[] htmlNumbersGM = {"&#402;", "&#913;", "&#914;", "&#915;", "&#916;", "&#917;", "&#918;", "&#919;", "&#920;", "&#921;", "&#922;", "&#923;", "&#924;", "&#925;", "&#926;", "&#927;", "&#928;", "&#929;", "&#931;", "&#932;", "&#933;", "&#934;", "&#935;", "&#936;", "&#937;", "&#945;", "&#946;", "&#947;", "&#948;", "&#949;", "&#950;", "&#951;", "&#952;", "&#953;", "&#954;", "&#955;", "&#956;", "&#957;", "&#958;", "&#959;", "&#960;", "&#961;", "&#962;", "&#963;", "&#964;", "&#965;", "&#966;", "&#967;", "&#968;", "&#969;", "&#977;", "&#978;", "&#982;", "&#8226;", "&#8230;", "&#8242;", "&#8243;", "&#8254;", "&#8260;", "&#8472;", "&#8465;", "&#8476;", "&#8482;", "&#8501;", "&#8592;", "&#8593;", "&#8594;", "&#8595;", "&#8596;", "&#8629;", "&#8656;", "&#8657;", "&#8658;", "&#8659;", "&#8660;", "&#8704;", "&#8706;", "&#8707;", "&#8709;", "&#8711;", "&#8712;", "&#8713;", "&#8715;", "&#8719;", "&#8721;", "&#8722;", "&#8727;", "&#8730;", "&#8733;", "&#8734;", "&#8736;", "&#8743;", "&#8744;", "&#8745;", "&#8746;", "&#8747;", "&#8756;", "&#8764;", "&#8773;", "&#8776;", "&#8800;", "&#8801;", "&#8804;", "&#8805;", "&#8834;", "&#8835;", "&#8836;", "&#8838;", "&#8839;", "&#8853;", "&#8855;", "&#8869;", "&#8901;", "&#8968;", "&#8969;", "&#8970;", "&#8971;", "&#9001;", "&#9002;", "&#9674;", "&#9824;", "&#9827;", "&#9829;", "&#9830;"};
    private String[] htmlNamesGM = {"&fnof;", "&Alpha;", "&Beta;", "&Gamma;", "&Delta;", "&Epsilon;", "&Zeta;", "&Eta;", "&Theta;", "&Iota;", "&Kappa;", "&Lambda;", "&Mu;", "&Nu;", "&Xi;", "&Omicron;", "&Pi;", "&Rho;", "&Sigma;", "&Tau;", "&Upsilon;", "&Phi;", "&Chi;", "&Psi;", "&Omega;", "&alpha;", "&beta;", "&gamma;", "&delta;", "&epsilon;", "&zeta;", "&eta;", "&theta;", "&iota;", "&kappa;", "&lambda;", "&mu;", "&nu;", "&xi;", "&omicron;", "&pi;", "&rho;", "&sigmaf;", "&sigma;", "&tau;", "&upsilon;", "&phi;", "&chi;", "&psi;", "&omega;", "&thetasym;", "&upsih;", "&piv;", "&bull;", "&hellip;", "&prime;", "&Prime;", "&oline;", "&frasl;", "&weierp;", "&image;", "&real;", "&trade;", "&alefsym;", "&larr;", "&uarr;", "&rarr;", "&darr;", "&harr;", "&crarr;", "&lArr;", "&uArr;", "&rArr;", "&dArr;", "&hArr;", "&forall;", "&part;", "&exist;", "&empty;", "&nabla;", "&isin;", "&notin;", "&ni;", "&prod;", "&sum;", "&minus;", "&lowast;", "&radic;", "&prop;", "&infin;", "&ang;", "&and;", "&or;", "&cap;", "&cup;", "&int;", "&there4;", "&sim;", "&cong;", "&asymp;", "&ne;", "&equiv;", "&le;", "&ge;", "&sub;", "&sup;", "&nsub;", "&sube;", "&supe;", "&oplus;", "&otimes;", "&perp;", "&sdot;", "&lceil;", "&rceil;", "&lfloor;", "&rfloor;", "&lang;", "&rang;", "&loz;", "&spades;", "&clubs;", "&hearts;", "&diams;"};
    private int[] decNumbersGM = {402, 913, 914, 915, 916, 917, 918, 919, 920, 921, 922, 923, 924, 925, 926, 927, 928, 929, 931, 932, 933, 934, 935, 936, 937, 945, 946, 947, 948, 949, 950, 951, 952, 953, 954, 955, 956, 957, 958, 959, 960, 961, 962, 963, 964, 965, 966, 967, 968, 969, 977, 978, 982, 8226, 8230, 8242, 8243, 8254, 8260, 8472, 8465, 8476, 8482, 8501, 8592, 8593, 8594, 8595, 8596, 8629, 8656, 8657, 8658, 8659, 8660, 8704, 8706, 8707, 8709, 8711, 8712, 8713, 8715, 8719, 8721, 8722, 8727, 8730, 8733, 8734, 8736, 8743, 8744, 8745, 8746, 8747, 8756, 8764, 8773, 8776, 8800, 8801, 8804, 8805, 8834, 8835, 8836, 8838, 8839, 8853, 8855, 8869, 8901, 8968, 8969, 8970, 8971, 9001, 9002, 9674, 9824, 9827, 9829, 9830};
    private String[] commentsGM = {"Latin small f with hook = function = florin ", "Greek capital letter alpha ", "Greek capital letter beta ", "Greek capital letter gamma ", "Greek capital letter delta ", "Greek capital letter epsilon ", "Greek capital letter zeta ", "Greek capital letter eta ", "Greek capital letter theta ", "Greek capital letter iota ", "Greek capital letter kappa ", "Greek capital letter lambda ", "Greek capital letter mu ", "Greek capital letter nu ", "Greek capital letter xi ", "Greek capital letter omicron ", "Greek capital letter pi ", "Greek capital letter rho ", "Greek capital letter sigma ", "Greek capital letter tau ", "Greek capital letter upsilon ", "Greek capital letter phi ", "Greek capital letter chi ", "Greek capital letter psi ", "Greek capital letter omega ", "Greek small letter alpha ", "Greek small letter beta ", "Greek small letter gamma ", "Greek small letter delta ", "Greek small letter epsilon ", "Greek small letter zeta ", "Greek small letter eta ", "Greek small letter theta ", "Greek small letter iota ", "Greek small letter kappa ", "Greek small letter lambda ", "Greek small letter mu ", "Greek small letter nu ", "Greek small letter xi ", "Greek small letter omicron ", "Greek small letter pi ", "Greek small letter rho ", "Greek small letter final sigma ", "Greek small letter sigma ", "Greek small letter tau ", "Greek small letter upsilon ", "Greek small letter phi ", "Greek small letter chi ", "Greek small letter psi ", "Greek small letter omega ", "Greek small letter theta symbol ", "Greek upsilon with hook symbol ", "Greek pi symbol ", "bullet = black small circle ", "horizontal ellipsis = three dot leader ", "prime = minutes = feet ", "double prime = seconds = inches ", "overline = spacing overscore ", "fraction slash ", "script capital P = power set = Weierstrass p ", "blackletter capital I = imaginary part ", "blackletter capital R = real part symbol ", "trade mark sign ", "alef symbol = first transfinite cardinal ", "leftwards arrow ", "upwards arrow ", "rightwards arrow ", "downwards arrow ", "left right arrow ", "downwards arrow with corner leftwards = carriage return ", "leftwards double arrow ", "upwards double arrow ", "rightwards double arrow ", "downwards double arrow ", "left right double arrow ", "for all ", "partial differential ", "there exists ", "empty set = null set = diameter ", "nabla = backward difference ", "element of ", "not an element of ", "contains as member ", "n-ary product = product sign ", "n-ary sumation ", "minus sign ", "asterisk operator ", "square root = radical sign ", "proportional to ", "infinity ", "angle ", "logical and = wedge ", "logical or = vee ", "intersection = cap ", "union = cup ", "integral ", "therefore ", "tilde operator = varies with = similar to ", "approximately equal to ", "almost equal to = asymptotic to ", "not equal to ", "identical to ", "less-than or equal to ", "greater-than or equal to ", "subset of ", "superset of ", "not a subset of ", "subset of or equal to ", "superset of or equal to ", "circled plus = direct sum ", "circled times = vector product ", "up tack = orthogonal to = perpendicular ", "dot operator ", "left ceiling = APL upstile ", "right ceiling ", "left floor = APL downstile ", "right floor ", "left-pointing angle bracket = bra ", "right-pointing angle bracket = ket ", "lozenge ", "black spade suit ", "black club suit = shamrock ", "black heart suit = valentine ", "black diamond suit "};
    private int nSymbolsGM = symbolsGM.length;
      
    // All codes listed
    private String[] allNumberCodes = {"&#34;", "&#38;", "&#60;", "&#62;", "&#160;", "&#161;", "&#162;", "&#163;", "&#164;", "&#165;", "&#166;", "&#167;", "&#168;", "&#169;", "&#170;", "&#171;", "&#172;", "&#173;", "&#174;", "&#175;", "&#176;", "&#177;", "&#178;", "&#179;", "&#180;", "&#181;", "&#182;", "&#183;", "&#184;", "&#185;", "&#186;", "&#187;", "&#188;", "&#189;", "&#190;", "&#191;", "&#192;", "&#193;", "&#194;", "&#195;", "&#196;", "&#197;", "&#198;", "&#199;", "&#200;", "&#201;", "&#202;", "&#203;", "&#204;", "&#205;", "&#206;", "&#207;", "&#208;", "&#209;", "&#210;", "&#211;", "&#212;", "&#213;", "&#214;", "&#215;", "&#216;", "&#217;", "&#218;", "&#219;", "&#220;", "&#221;", "&#222;", "&#223;", "&#224;", "&#225;", "&#226;", "&#227;", "&#228;", "&#229;", "&#230;", "&#231;", "&#232;", "&#233;", "&#234;", "&#235;", "&#236;", "&#237;", "&#238;", "&#239;", "&#8364;", "&#240;", "&#241;", "&#242;", "&#243;", "&#244;", "&#245;", "&#246;", "&#248;", "&#249;", "&#250;", "&#251;", "&#252;", "&#253;", "&#254;", "&#255;", "&#338;", "&#339;", "&#352;", "&#353;", "&#376;", "&#402;", "&#39;", "&#130;", "&#132;", "&#139;", "&#145;", "&#146;", "&#147;", "&#148;", "&#155;", "&#8216;", "&#8217;", "&#8218;", "&#8220;", "&#8221;", "&#8222;", "&#45;", "&#150;", "&#151;", "&#8213;", "&#8254;", "&#8211;", "&#8212;", "&#8352;", "&#8353;", "&#8354;", "&#8355;", "&#8356;", "&#8357;", "&#8358;", "&#8359;", "&#8360;", "&#8361;", "&#8362;", "&#8363;", "&#8365;", "&#8366;", "&#8367;", "&#8368;", "&#8369;", "&#8370;", "&#8371;", "&#8372;", "&#8373;", "&#8374;", "&#8375;", "&#8376;", "&#8377;", "&#8378;", "&#8379;", "&#8380;", "&#8381;", "&#8382;", "&#8383;", "&#402;", "&#913;", "&#914;", "&#915;", "&#916;", "&#917;", "&#918;", "&#919;", "&#920;", "&#921;", "&#922;", "&#923;", "&#924;", "&#925;", "&#926;", "&#927;", "&#928;", "&#929;", "&#931;", "&#932;", "&#933;", "&#934;", "&#935;", "&#936;", "&#937;", "&#945;", "&#946;", "&#947;", "&#948;", "&#949;", "&#950;", "&#951;", "&#952;", "&#953;", "&#954;", "&#955;", "&#956;", "&#957;", "&#958;", "&#959;", "&#960;", "&#961;", "&#962;", "&#963;", "&#964;", "&#965;", "&#966;", "&#967;", "&#968;", "&#969;", "&#977;", "&#978;", "&#982;", "&#8226;", "&#8230;", "&#8242;", "&#8243;", "&#8260;", "&#8472;", "&#8465;", "&#8476;", "&#8482;", "&#8501;", "&#8592;", "&#8593;", "&#8594;", "&#8595;", "&#8596;", "&#8629;", "&#8656;", "&#8657;", "&#8658;", "&#8659;", "&#8660;", "&#8704;", "&#8706;", "&#8707;", "&#8709;", "&#8711;", "&#8712;", "&#8713;", "&#8715;", "&#8719;", "&#8721;", "&#8722;", "&#8727;", "&#8730;", "&#8733;", "&#8734;", "&#8736;", "&#8743;", "&#8744;", "&#8745;", "&#8746;", "&#8747;", "&#8756;", "&#8764;", "&#8773;", "&#8776;", "&#8800;", "&#8801;", "&#8804;", "&#8805;", "&#8834;", "&#8835;", "&#8836;", "&#8838;", "&#8839;", "&#8853;", "&#8855;", "&#8869;", "&#8901;", "&#8968;", "&#8969;", "&#8970;", "&#8971;", "&#9001;", "&#9002;", "&#9674;", "&#9824;", "&#9827;", "&#9829;", "&#9830;"};
    private String[] allNameCodes = {"&quot;", "&amp;", "&lt;", "&gt;", "&nbsp;", "&iexcl;", "&cent;", "&pound;", "&curren;", "&yen;", "&brvbar;", "&sect;", "&uml;", "&copy;", "&ordf;", "&laquo;", "&not;", "&shy;", "&reg;", "&macr;", "&deg;", "&plusmn;", "&sup2;", "&sup3;", "&acute;", "&micro;", "&para;", "&middot;", "&cedil;", "&sup1;", "&ordm;", "&raquo;", "&frac14;", "&frac12;", "&frac34;", "&iquest;", "&Agrave;", "&Aacute;", "&Acirc;", "&Atilde;", "&Auml;", "&Aring;", "&AElig;", "&Ccedil;", "&Egrave;", "&Eacute;", "&Ecirc;", "&Euml;", "&Igrave;", "&Iacute;", "&Icirc;", "&Iuml;", "&ETH;", "&Ntilde;", "&Ograve;", "&Oacute;", "&Ocirc;", "&Otilde;", "&Ouml;", "&times;", "&Oslash;", "&Ugrave;", "&Uacute;", "&Ucirc;", "&Uuml;", "&Yacute;", "&THORN;", "&szlig;", "&agrave;", "&aacute;", "&acirc;", "&atilde;", "&auml;", "&aring;", "&aelig;", "&ccedil;", "&egrave;", "&eacute;", "&ecirc;", "&euml;", "&igrave;", "&iacute;", "&icirc;", "&iuml;", "&#euro;", "&eth;", "&ntilde;", "&ograve;", "&oacute;", "&ocirc;", "&otilde;", "&ouml;", "&oslash;", "&ugrave;", "&uacute;", "&ucirc;", "&uuml;", "&yacute;", "&thorn;", "&yuml;", "&OE;", "&oe;", "&Scaron;", "&scaron;", "&Ydia;", "&fhook;", "&apos;", "&sbquo;", "&dbquo;", "&lsaquo;", "&lsquo;", "&rsquo;", "&ldquo;", "&rdquo;", "&rsaquo;", "&none;", "&none;", "&none;", "&none;", "&none;", "&none;", "&none;", "&ndash;", "&mdash;", "?", "&oline;", "&none;", "&none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&#none;", "&fnof;", "&Alpha;", "&Beta;", "&Gamma;", "&Delta;", "&Epsilon;", "&Zeta;", "&Eta;", "&Theta;", "&Iota;", "&Kappa;", "&Lambda;", "&Mu;", "&Nu;", "&Xi;", "&Omicron;", "&Pi;", "&Rho;", "&Sigma;", "&Tau;", "&Upsilon;", "&Phi;", "&Chi;", "&Psi;", "&Omega;", "&alpha;", "&beta;", "&gamma;", "&delta;", "&epsilon;", "&zeta;", "&eta;", "&theta;", "&iota;", "&kappa;", "&lambda;", "&mu;", "&nu;", "&xi;", "&omicron;", "&pi;", "&rho;", "&sigmaf;", "&sigma;", "&tau;", "&upsilon;", "&phi;", "&chi;", "&psi;", "&omega;", "&thetasym;", "&upsih;", "&piv;", "&bull;", "&hellip;", "&prime;", "&Prime;", "&frasl;", "&weierp;", "&image;", "&real;", "&trade;", "&alefsym;", "&larr;", "&uarr;", "&rarr;", "&darr;", "&harr;", "&crarr;", "&lArr;", "&uArr;", "&rArr;", "&dArr;", "&hArr;", "&forall;", "&part;", "&exist;", "&empty;", "&nabla;", "&isin;", "&notin;", "&ni;", "&prod;", "&sum;", "&minus;", "&lowast;", "&radic;", "&prop;", "&infin;", "&ang;", "&and;", "&or;", "&cap;", "&cup;", "&int;", "&there4;", "&sim;", "&cong;", "&asymp;", "&ne;", "&equiv;", "&le;", "&ge;", "&sub;", "&sup;", "&nsub;", "&sube;", "&supe;", "&oplus;", "&otimes;", "&perp;", "&sdot;", "&lceil;", "&rceil;", "&lfloor;", "&rfloor;", "&lang;", "&rang;", "&loz;", "&spades;", "&clubs;", "&hearts;", "&diams;"};   
    private int[] allAsciiDecs = {34, 38, 60, 62, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 8364, 240, 241, 242, 243, 244, 245, 246, 248, 249, 250, 251, 252, 253, 254, 255, 338, 339, 352, 353, 376, 402, 39, 130, 132, 139, 145, 146, 147, 148, 155, 8216, 8217, 8218, 8220, 8221, 8222, 45, 150, 151, 8213, 8254, 8211, 8212, 8352, 8353, 8354, 8355, 8356, 8357, 8358, 8359, 8360, 8361, 8362, 8363, 8365, 8366, 8367, 8368, 8369, 8370, 8371, 8372, 8373, 8374, 8375, 8376, 8377, 8378, 8379, 8380, 8381, 8382, 8383, 402, 913, 914, 915, 916, 917, 918, 919, 920, 921, 922, 923, 924, 925, 926, 927, 928, 929, 931, 932, 933, 934, 935, 936, 937, 945, 946, 947, 948, 949, 950, 951, 952, 953, 954, 955, 956, 957, 958, 959, 960, 961, 962, 963, 964, 965, 966, 967, 968, 969, 977, 978, 982, 8226, 8230, 8242, 8243, 8260, 8472, 8465, 8476, 8482, 8501, 8592, 8593, 8594, 8595, 8596, 8629, 8656, 8657, 8658, 8659, 8660, 8704, 8706, 8707, 8709, 8711, 8712, 8713, 8715, 8719, 8721, 8722, 8727, 8730, 8733, 8734, 8736, 8743, 8744, 8745, 8746, 8747, 8756, 8764, 8773, 8776, 8800, 8801, 8804, 8805, 8834, 8835, 8836, 8838, 8839, 8853, 8855, 8869, 8901, 8968, 8969, 8970, 8971, 9001, 9002, 9674, 9824, 9827, 9829, 9830};
    private int nAllCodes = allNumberCodes.length;
    
    // all name codes listed
    private String[] allNameCodesNumber = {"&#39;", "&#34;", "&#38;", "&#60;", "&#62;", "&#160;", "&#161;", "&#162;", "&#163;", "&#164;", "&#165;", "&#166;", "&#167;", "&#168;", "&#169;", "&#170;", "&#171;", "&#172;", "&#173;", "&#174;", "&#175;", "&#176;", "&#177;", "&#178;", "&#179;", "&#180;", "&#181;", "&#182;", "&#183;", "&#184;", "&#185;", "&#186;", "&#187;", "&#188;", "&#189;", "&#190;", "&#191;", "&#192;", "&#193;", "&#194;", "&#195;", "&#196;", "&#197;", "&#198;", "&#199;", "&#200;", "&#201;", "&#202;", "&#203;", "&#204;", "&#205;", "&#206;", "&#207;", "&#208;", "&#209;", "&#210;", "&#211;", "&#212;", "&#213;", "&#214;", "&#215;", "&#216;", "&#217;", "&#218;", "&#219;", "&#220;", "&#221;", "&#222;", "&#223;", "&#224;", "&#225;", "&#226;", "&#227;", "&#228;", "&#229;", "&#230;", "&#231;", "&#232;", "&#233;", "&#234;", "&#235;", "&#236;", "&#237;", "&#238;", "&#239;", "&#8364;", "&#240;", "&#241;", "&#242;", "&#243;", "&#244;", "&#245;", "&#246;", "&#248;", "&#249;", "&#250;", "&#251;", "&#252;", "&#253;", "&#254;", "&#255;", "&#338;", "&#339;", "&#352;", "&#353;", "&#376;", "&#402;", "&#130;", "&#132;", "&#139;", "&#145;", "&#146;", "&#147;", "&#148;", "&#155;", "&#150;", "&#151;", "&#8213;", "&#8254;", "&#402;", "&#913;", "&#914;", "&#915;", "&#916;", "&#917;", "&#918;", "&#919;", "&#920;", "&#921;", "&#922;", "&#923;", "&#924;", "&#925;", "&#926;", "&#927;", "&#928;", "&#929;", "&#931;", "&#932;", "&#933;", "&#934;", "&#935;", "&#936;", "&#937;", "&#945;", "&#946;", "&#947;", "&#948;", "&#949;", "&#950;", "&#951;", "&#952;", "&#953;", "&#954;", "&#955;", "&#956;", "&#957;", "&#958;", "&#959;", "&#960;", "&#961;", "&#962;", "&#963;", "&#964;", "&#965;", "&#966;", "&#967;", "&#968;", "&#969;", "&#977;", "&#978;", "&#982;", "&#8226;", "&#8230;", "&#8242;", "&#8243;", "&#8260;", "&#8472;", "&#8465;", "&#8476;", "&#8482;", "&#8501;", "&#8592;", "&#8593;", "&#8594;", "&#8595;", "&#8596;", "&#8629;", "&#8656;", "&#8657;", "&#8658;", "&#8659;", "&#8660;", "&#8704;", "&#8706;", "&#8707;", "&#8709;", "&#8711;", "&#8712;", "&#8713;", "&#8715;", "&#8719;", "&#8721;", "&#8722;", "&#8727;", "&#8730;", "&#8733;", "&#8734;", "&#8736;", "&#8743;", "&#8744;", "&#8745;", "&#8746;", "&#8747;", "&#8756;", "&#8764;", "&#8773;", "&#8776;", "&#8800;", "&#8801;", "&#8804;", "&#8805;", "&#8834;", "&#8835;", "&#8836;", "&#8838;", "&#8839;", "&#8853;", "&#8855;", "&#8869;", "&#8901;", "&#8968;", "&#8969;", "&#8970;", "&#8971;", "&#9001;", "&#9002;", "&#9674;", "&#9824;", "&#9827;", "&#9829;", "&#9830;"};
    private String[] allNameCodesName = {"&apos", "&quot;", "&amp;", "&lt;", "&gt;", "&nbsp;", "&iexcl;", "&cent;", "&pound;", "&curren;", "&yen;", "&brvbar;", "&sect;", "&uml;", "&copy;", "&ordf;", "&laquo;", "&not;", "&shy;", "&reg;", "&macr;", "&deg;", "&plusmn;", "&sup2;", "&sup3;", "&acute;", "&micro;", "&para;", "&middot;", "&cedil;", "&sup1;", "&ordm;", "&raquo;", "&frac14;", "&frac12;", "&frac34;", "&iquest;", "&Agrave;", "&Aacute;", "&Acirc;", "&Atilde;", "&Auml;", "&Aring;", "&AElig;", "&Ccedil;", "&Egrave;", "&Eacute;", "&Ecirc;", "&Euml;", "&Igrave;", "&Iacute;", "&Icirc;", "&Iuml;", "&ETH;", "&Ntilde;", "&Ograve;", "&Oacute;", "&Ocirc;", "&Otilde;", "&Ouml;", "&times;", "&Oslash;", "&Ugrave;", "&Uacute;", "&Ucirc;", "&Uuml;", "&Yacute;", "&THORN;", "&szlig;", "&agrave;", "&aacute;", "&acirc;", "&atilde;", "&auml;", "&aring;", "&aelig;", "&ccedil;", "&egrave;", "&eacute;", "&ecirc;", "&euml;", "&igrave;", "&iacute;", "&icirc;", "&iuml;", "&#euro;", "&eth;", "&ntilde;", "&ograve;", "&oacute;", "&ocirc;", "&otilde;", "&ouml;", "&oslash;", "&ugrave;", "&uacute;", "&ucirc;", "&uuml;", "&yacute;", "&thorn;", "&yuml;", "&OE;", "&oe;", "&Scaron;", "&scaron;", "&Ydia;", "&fhook;", "&sbquo;", "&dbquo;", "&lsaquo;", "&lsquo;", "&rsquo;", "&ldquo;", "&rdquo;", "&rsaquo;", "&ndash;", "&mdash;", "", "&oline;", "&fnof;", "&Alpha;", "&Beta;", "&Gamma;", "&Delta;", "&Epsilon;", "&Zeta;", "&Eta;", "&Theta;", "&Iota;", "&Kappa;", "&Lambda;", "&Mu;", "&Nu;", "&Xi;", "&Omicron;", "&Pi;", "&Rho;", "&Sigma;", "&Tau;", "&Upsilon;", "&Phi;", "&Chi;", "&Psi;", "&Omega;", "&alpha;", "&beta;", "&gamma;", "&delta;", "&epsilon;", "&zeta;", "&eta;", "&theta;", "&iota;", "&kappa;", "&lambda;", "&mu;", "&nu;", "&xi;", "&omicron;", "&pi;", "&rho;", "&sigmaf;", "&sigma;", "&tau;", "&upsilon;", "&phi;", "&chi;", "&psi;", "&omega;", "&thetasym;", "&upsih;", "&piv;", "&bull;", "&hellip;", "&prime;", "&Prime;", "&frasl;", "&weierp;", "&image;", "&real;", "&trade;", "&alefsym;", "&larr;", "&uarr;", "&rarr;", "&darr;", "&harr;", "&crarr;", "&lArr;", "&uArr;", "&rArr;", "&dArr;", "&hArr;", "&forall;", "&part;", "&exist;", "&empty;", "&nabla;", "&isin;", "&notin;", "&ni;", "&prod;", "&sum;", "&minus;", "&lowast;", "&radic;", "&prop;", "&infin;", "&ang;", "&and;", "&or;", "&cap;", "&cup;", "&int;", "&there4;", "&sim;", "&cong;", "&asymp;", "&ne;", "&equiv;", "&le;", "&ge;", "&sub;", "&sup;", "&nsub;", "&sube;", "&supe;", "&oplus;", "&otimes;", "&perp;", "&sdot;", "&lceil;", "&rceil;", "&lfloor;", "&rfloor;", "&lang;", "&rang;", "&loz;", "&spades;", "&clubs;", "&hearts;", "&diams;"};
    private int[] allNameCodesDec = {39, 34, 38, 60, 62, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 8364, 240, 241, 242, 243, 244, 245, 246, 248, 249, 250, 251, 252, 253, 254, 255, 338, 339, 352, 353, 376, 402, 130, 132, 139, 145, 146, 147, 148, 155, 150, 151, 8213, 8254, 402, 913, 914, 915, 916, 917, 918, 919, 920, 921, 922, 923, 924, 925, 926, 927, 928, 929, 931, 932, 933, 934, 935, 936, 937, 945, 946, 947, 948, 949, 950, 951, 952, 953, 954, 955, 956, 957, 958, 959, 960, 961, 962, 963, 964, 965, 966, 967, 968, 969, 977, 978, 982, 8226, 8230, 8242, 8243, 8260, 8472, 8465, 8476, 8482, 8501, 8592, 8593, 8594, 8595, 8596, 8629, 8656, 8657, 8658, 8659, 8660, 8704, 8706, 8707, 8709, 8711, 8712, 8713, 8715, 8719, 8721, 8722, 8727, 8730, 8733, 8734, 8736, 8743, 8744, 8745, 8746, 8747, 8756, 8764, 8773, 8776, 8800, 8801, 8804, 8805, 8834, 8835, 8836, 8838, 8839, 8853, 8855, 8869, 8901, 8968, 8969, 8970, 8971, 9001, 9002, 9674, 9824, 9827, 9829, 9830};
    private int nAllNameCodes = allNameCodesNumber.length;
      
    // Metacharacters
    private char[] metacharactersC = {'\\','^','$','.','|','?','*','+','(',')','[','{'};
    private String[] metacharactersS = {"\\","^","$",".","|","?","*","+","(",")","[","{"};
    private int nMeta = metacharactersS.length;   
    
    // Articles
    private String[] articles = {"a", "an", "the"};
    private int nArticles = articles.length;
    // Prepositions
    private String[] prepositions = {"about", "above", "across", "after", "against", "along", "among", "around", "at", "before", "behind", "below", "beneath", "beside", "between", "beyond", "but", "by", "despite", "down", "during", "except", "for", "from", "in", "inside", "into", "like", "near", "of", "off", "on", "onto", "out", "outside", "over", "past", "since", "through", "throughout", "till", "to", "toward", "under", "underneath", "until", "up", "upon", "with", "within", "without"};
    private int nPrepositions = prepositions.length;
   // Conjunctions
    private String[] conjunctions = {"and", "&amp", "&", "but", "or", "nor", "for", "so", "yet", "not", "only", "also", "either", "neither", "although", "because", "since", "unless", "until", "while"};
    private int nConjunctions = conjunctions.length;
    
    // Constructors 
    public Strings(String string){
        this.setData(string);
    }
    
    public Strings(){
        // for use with Updater
    }
    
    // Set entered data
    private void setData(String string){
        this.originalString = string;
        this.nOriginal = string.length();
        this.enteredString = string;
        this.enteredString = this.allHtmlNumberCodesToAsciiChars();
        this.enteredString = this.allHtmlNameCodesToAsciiChars();
        this.nEntered = this.enteredString.length();
    }
   
    // Suppress dialogue method
    public void suppressDialogue(){
        this.suppressDialogue = true;
    }
    
    // Enter a String
    public void setEntered(String string){
        this.setData(string);
    }
    
    // Return enteredString 
    public String getEnteredString(){
        return this.enteredString;
    }
       
    // Return editedString 
    public String getEditedString(){
        return this.editedString;
    }
    
    // Return number of characters in the entered String 
    public int getEnteredStringLength(){
        return this.nEntered;
    }
    
    // CONVERSIONS
    // Reverse String
    // Instance method
    public String reverse(){
        String retn = " ";
        for(int i=this.nEntered-1; i>=0; i--){
            retn += this.enteredString.charAt(i);
        }     
        return retn;
    }
    
    // Static method
    public static String reverse(String ss){
        String retn = " ";
        for(int i=ss.length()-1; i>=0; i--){
            retn += ss.charAt(i);
        }     
        return retn.trim();
    }
    
    // converts all the string to upper case
    public String toUpperCase(){
        this.upperCaseString = this.enteredString.toUpperCase();
        return this.upperCaseString;
    }
    
    public static String toUpperCase(String ss){
        Strings sss = new Strings(ss);
        return sss.toUpperCase();
    }
    
    // converts to the character at i to upper case
    public String toUpperCase(int i){
        String upperI = this.enteredString.substring(0,i) + this.enteredString.substring(i,i+1).toUpperCase() + this.enteredString.substring(i+1);
        return upperI;
    }
    
    public static String toUpperCase(String ss, int i){
        Strings sss = new Strings(ss);
        return sss.toUpperCase(i);
    }
    
    // converts to lower case
    public String toLowerCase(){
        this.lowerCaseString = this.enteredString.toLowerCase();
        return this.lowerCaseString;
    }
    
    // converts to the character at i to lower case
    public String toLowerCase(int i){
        String lowerI = this.enteredString.substring(0,i) + this.enteredString.substring(i,i+1).toLowerCase() + this.enteredString.substring(i+1);
        return lowerI;
    }

    public static String toLowerCase(String ss, int i){
        Strings sss = new Strings(ss);
        return sss.toLowerCase(i);
    }
    
    // Return string in full title case
    // instance method
    public String toAllTitleCase(){
        
        this.allTitleCaseString = this.enteredString.toLowerCase();
        this.tokens();
        
        int jj = 0;
        for(int i=0; i<this.nTokens; i++){
            jj = this.tokenInitialIndices[i];
            if(Chars.isQuotationMark(this.enteredString.charAt(jj)) && jj<this.nEntered)jj++;
            if(!this.whiteSpaces[jj]){
                this.allTitleCaseString = this.toUpperCase(jj);
            }   
        }     
        return this.allTitleCaseString;
    }
    
    // Return string in full title case
    // static method
    public static String toAllTitleCase(String ss){
       Strings strs = new Strings(ss);
       return strs.toAllTitleCase();
    }
    
    // Return string in title case, i.e. only prncipal words with initial upper case
    // instance method
    public String toTitleCase(){
        this.titleCaseString = this.enteredString.toLowerCase();
        this.principalTokens();
        
        int jj = 0;
        for(int i=0; i<this.nPrincipalTokens; i++){
            jj = this.principalTokenInitialIndices[i];
            if(Chars.isQuotationMark(this.titleCaseString.charAt(jj)) && jj<this.nEntered)jj++;
            if(!this.whiteSpaces[jj]){
                this.toUpperCase(jj);
            }   
        }     
        
        // check for colon
        String[] tokens = this.tokens(this.titleCaseString);
        int nT = tokens.length;
        String newS = tokens[0];
        boolean colonFound = false;
        for(int i=0;i<nT-1;i++){
            int nTi = tokens[i].length();
            char cti = tokens[i].charAt(nTi-1);
            if(cti==':'){
                colonFound = true;
                tokens[i+1] = tokens[i+1].substring(0,1).toUpperCase() + tokens[i+1].substring(1);
            }
            newS = newS + " " + tokens[i+1];
        }
        if(colonFound)this.titleCaseString = newS.trim();
        
        return this.titleCaseString;
    }
    
    // Return string in title case, i.e. only prncipal words with initial upper case
    // static method
    public static String toTitleCase(String ss){
       Strings strs = new Strings(ss);
       return strs.toTitleCase();
    }
    
    
    // Converts to sentence case
    // instance method
    public String toSentenceCase(){
        this.sentenceCaseString = this.enteredString.toLowerCase();
        this.tokens();
        
        int jj = this.tokenInitialIndices[0];
        int kk = -1;
        if(Chars.isQuotationMark(this.sentenceCaseString.charAt(jj)) && jj<this.nEntered)jj++;
        if(!this.whiteSpaces[jj]){
            this.toUpperCase(jj);
        }
        for(int i=0;i<this.nTokens-1; i++){
            jj = this.tokenFinalIndices[i];
            if(Chars.isQuotationMark(this.sentenceCaseString.charAt(jj)))jj--;
            if(this.enteredString.charAt(jj)=='.'){
                kk = this.tokenInitialIndices[i+1];
                if(Chars.isQuotationMark(this.sentenceCaseString.charAt(kk)))kk++;
                if(!this.whiteSpaces[kk]){
                    this.toUpperCase(kk);
                }
            }
        }
        return this.sentenceCaseString;
    }
    
    // Converts to sentence case
    // static method
      public static String toSentenceCase(String ss){
       Strings strs = new Strings(ss);
       return strs.toSentenceCase();
    }
      
          // Removes all accents and replaces the accented letter with its non-accented equivalent 
    // instance method
    public String removeAccents(){
        
        String ss = this.enteredString;
        int nSS = ss.length();
        
        // Remove html names and replace with unaccented equivalent
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsA; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNamesA[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNamesA[i].length();
                    String hold0 = ss.substring(0,pos0) + this.replSymbolsA[i];
                    if(pos1<this.nEntered)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Remove html numbers and replace with unaccented equivalent
        for(int i=0; i<this.nSymbolsA; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNumbersA[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNumbersA[i].length();
                    String hold0 = ss.substring(0,pos0) + this.replSymbolsA[i];
                    if(pos1<this.nEntered)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Replace int equivalent with unaccented equivalent
        for(int i=0; i<this.nSymbolsA; i++){
            for(int j=0; j<nSS; j++){
               if((int)ss.charAt(j)==this.decNumbersA[i]){
                   ss = ss.substring(0,j) + this.replSymbolsA[i] + ss.substring(j+1);
               }
            }
            nSS = ss.length();
        }
        
        this.unAccentedString = ss;
        return ss;
    }
    
    // Removes all accents and replaces the accented letter with its non-accented equivalent 
    // static method
    public static String removeAccents(String ss){
        Strings strs = new Strings(ss);
        return strs.removeAccents();
    } 
    
    // TOKENISATION
    
    // Treat tab as tab, otherwise replace by white spave
    public void tabAsTab(){
        this.tabastab = true;
    }
    
    // Check presence of a separator
    public String checkForSeparator(char sepc){
        boolean sepPresent = false;
        String seps = "" + sepc;
        if(sepc=='"')seps = "\"";
        seps = this.separatorCheck(seps);
        return seps;
    }
    
       
    // Required by tokenisation methods
    private String checkForSeparator(String seps){
        seps = this.separatorCheck(seps);
        return seps;
    }
    
    // Checks tat separator is present and corrects its form if the separator is a metacharacter
    private String separatorCheck(String seps){
        if(seps.equals("\\"))throw new IllegalArgumentException("Method does not handle backslash \\ as separator");
        
        // If separator is white space checck for tabs
        if(seps.equals(" ") && !this.tabastab){
            if(this.enteredString.indexOf('\t')>-1){
                this.editedString = this.enteredString;
                this.enteredString = this.enteredString.replace('\t',' ');
                this.tabsReplaced = true;
            }           
        }
        
        
        // chech if separator present
        if (this.enteredString.contains(seps)) {
            // Check if separator is a metacharacter
            for(int i=0; i<this.nMeta; i++){
                if(seps.equals(this.metacharactersS[i])){
                    seps = "[" + seps + "}";
                    break;
                }
            }
        } 
        else{
            throw new IllegalArgumentException("String " + this.enteredString + " does not contain " + seps);
        }
        return seps;
    }
    
    // Split the string into tokens with supplied separator 
    // instance methods
    // char separator
    public String[] tokens(char sepr){
        String seps = this.checkForSeparator(sepr);
        return this.calculateTokens(seps);
    }
    
    // String separator
    public String[] tokens(String sepr){
        String seps = this.checkForSeparator(sepr);
        return this.calculateTokens(seps);
    }
    
    // white space separator
    public String[] tokens(){
        String seps = " ";
        return this.calculateTokens(seps);
    }
    
    public String[] tokensWhiteSpace(){
        String seps = " ";
        return this.calculateTokens(seps);
    }
    
    // static methods
    // char separator
    public static String[] tokens(String string, char sepr){
        Strings sss = new Strings(string);
        return sss.tokens(sepr);
    }
    
    // String separator
    public static String[] tokens(String string, String sepr){
        Strings sss = new Strings(string);
        return sss.tokens(sepr);
    }
    
    // white space separator
    public static String[] tokensWhiteSpace(String string){
        Strings sss = new Strings(string);
        return sss.tokens();
    }
    
       
    // Checks for white spaces
    // instance method
    public boolean[] whiteSpaces(){
        this.whiteSpaces = new boolean[this.nEntered];
        boolean test = true;
        int iFirst = -1;
        for(int i=0; i<this.nEntered; i++){
            whiteSpaces[i] = false;
            if(Character.isWhitespace(this.enteredString.charAt(i))){
                whiteSpaces[i] = true;
            }
        }
        return this.whiteSpaces;
    }
   
    // Checks for white spaces
    // static method
    public static boolean[] whiteSpaces(String ss){
       Strings strs = new Strings(ss);
       return strs.whiteSpaces();
    } 
    
    // Removes white spaces
    // instance method
    public String removeWhiteSpaces(){
        String ret = "";
        for(int i=0; i<this.nEntered; i++){
            char cc = this.enteredString.charAt(i);
            if(!Character.isWhitespace(cc)){
                ret += cc;
            }
        } 
        return ret.trim();
    }
    
    // Removes white spaves 
    // static method
    public static String removeWhiteSpaces(String line){
        Strings ss = new Strings(line);
        return ss.removeWhiteSpaces();
    }
    
    // Convert tabs to white spaves
    // instance method
    public String convertTabsToWhiteSpaces(){
        return this.enteredString.replace('\t', ' ');
    }
    
    // static method
    public static String convertTabsToWhiteSpaces(String string){
        return string.replace('\t', ' ');
    }
   
    
    private String[] calculateTokens(String sepr){    
        this.tokensDone = false;
        // check separator is present and in an appropriate form
         this.separator = this.checkForSeparator(sepr);
         
         // Split string into tokens
         // split ino parts
         String[] parts = this.enteredString.split(this.separator);
         int nParts = parts.length;
         // remove empty parts, eg as caused by adjacent reeated separators
         ArrayList<String> altok = new ArrayList<String>();
        for(int i=0;i<nParts; i++){
            if(parts[i].equals("")){
                // ignore
            }
            else{
                altok.add(parts[i]);
            }
        }
        this.nTokens = altok.size();
        this.tokens = new String[this.nTokens];
        this.tokenInitialIndices = new int[this.nTokens];    
        this.tokenFinalIndices = new int[this.nTokens];  
        for(int i=0;i<this.nTokens;i++){
            this.tokens[i] = altok.get(i).trim();
        }
       
 
        // Find initial and final index of each token
        int toklen = -1;
        int istart = 0;
        int ii = 0;
        String shold = this.enteredString;
        for(int i=0;i<this.nTokens;i++){
            shold = shold.substring(ii);
            toklen = this.tokens[i].length();
            ii = shold.indexOf(tokens[i]);
            this.tokenInitialIndices[i] = ii + istart;
            this.tokenFinalIndices[i] = this.tokenInitialIndices[i] + toklen - 1;
            istart += ii;
        }
        
        if(this.tabsReplaced)this.enteredString = this.editedString;
        this.tokensDone = true;
        return this.tokens;
    }
    
    // Return token initial indices
    // instance method
    public int[] tokenInitialIndices(){
        if(!this.tokensDone)this.tokens();
        return this.tokenInitialIndices;
    }
      
    // Return token initial indices
    // static method
    public static int[] tokenInitialIndices(String ss){
        Strings strs = new Strings(ss);
        return strs.tokenInitialIndices();
    }
    
    // Return token final indices
    // instance method
    public int[] tokenFinalIndices(){
        if(!this.tokensDone)this.tokens();
        return this.tokenFinalIndices;
    }
      
    // Return token final indices
    // static method
    public static int[] tokenFinalIndices(String ss){
        Strings strs = new Strings(ss);
        return strs.tokenFinalIndices();
    }
    
    // Replace separator
    // Instance methods
    // line replacement
    public String replaceSeparator(char oldSep, char newSep){
        boolean test = true;
        return this.replaceSeparator(test, String.valueOf(oldSep), String.valueOf(newSep));
    }
        
    public String replaceSeparator(String oldSep, String newSep){
        boolean test = true;
        String newS = this.enteredString.replaceAll(oldSep, newSep);
        return this.replaceSeparator(test, oldSep, newSep);
    }
    
    private String replaceSeparator(boolean test, String oldSep, String newSep){
        String newS = this.enteredString.replaceAll(oldSep, newSep);
        if(test){
            newS = this.replacement(newS);
        }
        return newS;
      }
    
    // file replacement
    public void replaceFileSeparatorI(String fileName, char oldSep, char newSep){ 
        this.replaceFileSeparatorI(fileName, String.valueOf(oldSep), String.valueOf(newSep));
    }   
    
    public void replaceFileSeparatorI(String fileName, String oldSep, String newSep){
        this.replaceFileSeparatorI(fileName, "-1", oldSep, newSep);
    }
    
    public void replaceFileSeparatorI(String fileNameIn, String fileNameOut, char oldSep, char newSep){ 
        this.replaceFileSeparatorI(fileNameIn, fileNameOut, String.valueOf(oldSep), String.valueOf(newSep));
    }
    
    public void replaceFileSeparatorI(String fileNameIn, String fileNameOut, String oldSep, String newSep){ 
        
        FileInput fin = new FileInput(fileNameIn);
        boolean testf = fin.fileFound();
        if(!testf){
            throw new IllegalArgumentException("File " + fileNameIn + " cannot be found\nDid you add the correct extension?");
        }
        else{
            String stem = fin.getStemName();
            int nStem = stem.length();
            String ext = fileNameIn.substring(nStem);
            if(!ext.trim().equals(".txt")) throw new IllegalArgumentException("File " + fileNameIn + " does not appear to be a txt file");
            String fileOutName = stem +"Edited" + fileNameIn.substring(nStem);
        
            int n = fin.numberOfLines();
            String[] lines = new String[n];
            String hold = null;
            for(int i=0; i<n;i++){
                hold = fin.readLine();
                lines[i] = Strings.replaceSeparator(hold, oldSep, newSep);
            }
            if(fileNameOut.equals("-1")){
                int opt = this.fileReplacement();
                if(opt==0){
                    // New file
                    fileOutName = Db.readLine("Enter new file name (remember any extension)", fileNameOut);
                }
                else{
                    // overwrite
                    fileNameOut = fileNameIn;   
            
                }
            }
            FileOutput fout = new FileOutput(fileNameOut);
            for(int i=0; i<n;i++){
                fout.println(lines[i]);
            }
        }
    }
    
    // Static methods
    // line replacement
    public static String replaceSeparator(String ss, char oldSep, char newSep){
        Strings sss = new Strings(ss);
        boolean test = false;
        return sss.replaceSeparator(test, String.valueOf(oldSep), String.valueOf(newSep));
    }
    
    public static String replaceSeparator(String ss, String oldSep, String newSep){
        Strings sss = new Strings(ss);
        boolean test = false;
        return sss.replaceSeparator(test, oldSep, newSep);
    }
    
    // file replacement
    public static void replaceFileSeparatorS(String fileNameIn, String fileNameOut, char oldSep, char newSep){ 
        Strings.replaceFileSeparatorS(fileNameIn, fileNameOut, String.valueOf(oldSep), String.valueOf(newSep));        
    }
    
    public static void replaceFileSeparatorS(String fileNameIn, String fileNameOut, String oldSep, String newSep){ 
        Strings sss = new Strings();
        sss.replaceFileSeparatorI(fileNameIn, fileNameOut, oldSep, newSep);;        
    }

    
    public static void replaceFileSeparatorS(String fileName, char oldSep, char newSep){ 
        Strings.replaceFileSeparatorS(fileName, String.valueOf(oldSep), String.valueOf(newSep));        
    }
        
    public static void replaceFileSeparatorS(String fileName, String oldSep, String newSep){
        Strings sss = new Strings();
        sss.replaceFileSeparatorI(fileName, oldSep, newSep);
    }

    // Required be replaceSeparator methods
    private String replacement(String newS){
        if(this.suppressDialogue){
            this.editedString = newS;
        }
        else{
            String headerComment = "Do you wish to:";
            String[] comments = {"Store as the edited String?", "Replace the current entered String?"};
            String[] boxTitles = {"Edited String", "Replace entered String"};
            int defaultBox = 1;
            int opt =  Db.optionBox(headerComment, comments, boxTitles, defaultBox);
            if(opt==0){
                this.editedString = newS;
            }
            else{
                this.enteredString = newS;
            }
        }
        return newS;
    } 
    
   // Required be replaceFileSeparator methods
    private int fileReplacement(){
        String headerComment = "Do you wish to:";
        String[] comments = {"Create a new file?", "Replace the current file?"};
        String[] boxTitles = {"New file", "Replace old file"};
        int defaultBox = 1;
        int opt =  Db.optionBox(headerComment, comments, boxTitles, defaultBox);
        return opt;
    } 
    

       
    // Split the string into principal tokens
    // instance method
    public String[] principalTokens(){
        if(!this.tokensDone)this.tokens();
        ArrayList<Integer> al = new ArrayList<Integer>();
        al.add(0);
        
        for(int i=1; i<this.nTokens; i++){
            String tt = Strings.removeQuotationMarks(this.tokens[i]);
            boolean test = true;
            for(int j=0; j<this.nArticles; j++){
                if(tt.equalsIgnoreCase(this.articles[j])){
                    test = false;
                    break;
                }
            }
            if(test){
                for(int j=0; j<this.nPrepositions; j++){
                    if(tt.equalsIgnoreCase(this.prepositions[j])){
                        test = false;
                        break;
                    }
                }
            }
            if(test){
                for(int j=0; j<this.nConjunctions; j++){
                    if(tt.equalsIgnoreCase(this.conjunctions[j])){
                        test = false;
                        break;
                    }
                }
            }
            if(test)al.add(i);
        }
        
        this.nPrincipalTokens = al.size();
        this.principalTokens = new String[this.nPrincipalTokens];
        this.principalTokenInitialIndices = new int[this.nPrincipalTokens];
        this.principalTokenFinalIndices = new int[this.nPrincipalTokens];
        for(int i=0; i<this.nPrincipalTokens; i++){
            int j = al.get(i);
            this.principalTokens[i] = this.tokens[j];
            this.principalTokenInitialIndices[i] = this.tokenInitialIndices[j];
            this.principalTokenFinalIndices[i] = this.tokenFinalIndices[j];
        }
        
        return this.principalTokens;
    }
    
    // Split the string into principal tokens
    // static method
    public static String[] principalTokens(String ss){
        Strings strs = new Strings(ss);
        return strs.principalTokens();
    }
    
    // Split the string into principal tokens ignoring obligatory  initial principal token
    // instance method
    public String[] principalTokensZero(){
        if(!this.tokensDone)this.tokens();
        ArrayList<Integer> al = new ArrayList<Integer>();
        
        for(int i=0; i<this.nTokens; i++){
            String tt = Strings.removeQuotationMarks(this.tokens[i]);
            boolean test = true;
            for(int j=0; j<this.nArticles; j++){
                if(tt.equalsIgnoreCase(this.articles[j])){
                    test = false;
                    break;
                }
            }
            if(test){
                for(int j=0; j<this.nPrepositions; j++){
                    if(tt.equalsIgnoreCase(this.prepositions[j])){
                        test = false;
                        break;
                    }
                }
            }
            if(test){
                for(int j=0; j<this.nConjunctions; j++){
                    if(tt.equalsIgnoreCase(this.conjunctions[j])){
                        test = false;
                        break;
                    }
                }
            }
            if(test)al.add(i);
        }
        
        this.nPrincipalTokens = al.size();
        this.principalTokens = new String[this.nPrincipalTokens];
        this.principalTokenInitialIndices = new int[this.nPrincipalTokens];
        this.principalTokenFinalIndices = new int[this.nPrincipalTokens];
        for(int i=0; i<this.nPrincipalTokens; i++){
            int j = al.get(i);
            this.principalTokens[i] = this.tokens[j];
            this.principalTokenInitialIndices[i] = this.tokenInitialIndices[j];
            this.principalTokenFinalIndices[i] = this.tokenFinalIndices[j];
        }
        
        return this.principalTokens;
    }
    
    // Split the string into principal tokens ignoring obligatory  initial principal token
    // static method
    public static String[] principalTokensZero(String ss){
        Strings strs = new Strings(ss);
        return strs.principalTokensZero();
    }
    
    // CONCATENATION
    // instance methods
    public String concat(String[] strings){
        int n = strings.length;
        String con =  this.editedString;
        for(int i=0; i<n; i++){
            con = con.concat(strings[i]);
        }
        return con;
    }
    
     public String concat(String string){
        String con =  this.editedString;
        con = con.concat(string);
        return con;
    }
     
     // static methods
     public static String concat(String ss, String[] strings){
        int n = strings.length;
        String con =  ss;
        for(int i=0; i<n; i++){
            con = con.concat(strings[i]);
        }
        return con;
    }
    
     public static String concat(String string1, String string2){
        return string1.concat(string2);
    }
     
    public static String concatS(String[] strings){
        int n = strings.length;
        String con =  strings[0];
        for(int i=1; i<n; i++){
            con = con.concat(strings[i]);
        }
        return con;
    }
    

    
    // ANALYSIS
    
    // RReturn all indices of a string or chacter
    // instace methods
    public int[] indicesOf(char cc){
        return this.indicesOf(String.valueOf(cc));
       
    }
    
    public int[] indicesOf(String ss){
        String sss = this.enteredString;
        ArrayList<Integer> alif = new ArrayList<Integer>();
        boolean test = true;
        int point = 0;
        int[] indss = null;
        int nio =0;
        while(test){
            int pos = sss.indexOf(ss);
            if(pos!=-1){
                    nio++;
                    point = point + pos;
                    alif.add(point);
                    sss = sss.substring(pos+1);
                    point += 1;
                    if(point>=this.nEntered)test = false;
            }
            else{
                    test = false;
           }
        }
        if(nio>0){
            int nI = alif.size();
            indss = new int[nI];
            for(int i=0; i<nI; i++){
                indss[i] = alif.get(i);
            }
        }
        return indss;
    }
        
    // static methods
    public static int[] indicesOf(char cc, String ss){
       return Strings.indicesOf(String.valueOf(cc), ss);
    }
    
    public static int[] indicesOf(String cc, String ss){
        Strings sss = new Strings(ss);
        return sss.indicesOf(cc);
    }
    
    // Return ASCII decimal equivalents
    // Instance method
    public int[] getASCIIdecimal(){
        int[] asciiDec = new int[this.nEntered];
        for(int i=0; i<nEntered; i++){
            asciiDec[i] = Chars.getASCIIdec(this.enteredString.charAt(i));
        }
        return asciiDec;
    }
    
    // Static method
    public static int[] getASCIIdecimal(String ss){
        Strings sss = new Strings(ss);
        return sss.getASCIIdecimal();
    }    
    
        // Return ASCII hexadecimal equivalents
    // Instance method
    public String[] getASCIIhexadecimal(){
        String[] asciiHex = new String[this.nEntered];
        for(int i=0; i<nEntered; i++){
            asciiHex[i] = Chars.getASCIIhex(this.enteredString.charAt(i));
        }
        return asciiHex;
    }
    
    // Static method
    public static String[] getASCIIhexadecimal(String ss){
        Strings sss = new Strings(ss);
        return sss.getASCIIhexadecimal();
    } 
    
    
    // TESTS
    // Check if all characters are alphabetic
    // instance method
    public boolean areAlphabetic(){
        boolean test0 = true;
        char cc = '\u0000';
        for(int i=0;i<this.nEntered; i++){
            cc = this.enteredString.charAt(i);
            test0 = Character.isAlphabetic(cc);
            if(!test0)break;
        }
        return test0;
    }
    
    // static method
    public static boolean areAlphabetic(String ss){
        Strings sss = new Strings(ss);
        return sss.areAlphabetic();
    }
    
    // Check if all characters are digits
    // instance method
    public boolean areDigits(){
        boolean test0 = true;
        char cc = '\u0000';
        for(int i=0;i<this.nEntered; i++){
            cc = this.enteredString.charAt(i);
            test0 = Character.isDigit(cc);
            if(!test0)break;
        }
        return test0;
    }
    
    // static method
    public static boolean areDigits(String ss){
        Strings sss = new Strings(ss);
        return sss.areDigits();
    }
    
    
    // PREPARATION FOR HTML DISPLAY
    
    // Conver tHtml Names To Html Numbers
    public String convertHtmlNamesToHtmlNumbers(){
    
        String ss = this.enteredString;
        for(int i=0; i<this.nAllCodes; i++){
            
        }
        return null;
}
    
    
    // check if a code is an html number code
    // Returns true if code is, false if not
    private boolean checkIfHtmlNumberCode(String code){
	boolean test0 = true;
	code = code.trim();
	int nc = code.length();
	if((int)code.charAt(0)!=38)test0 = false;
        if((int)code.charAt(1)!=35)test0 = false;
        if((int)code.charAt(nc-1)!=59)test0 = false;
        String num = code.substring(2,nc-1);
        int nss = num.length();
        for(int i=0; i<nss; i++){
            if(!Chars.isDigit(num.charAt(i)))test0 = false;
	}
        return test0;
    }

      //Convert all dashes to html number code representation 
    // instance method
    public String dashesToHtmlNumberCodes(){
        
        String ss = this.enteredString;
        int nSS = ss.length();
        
        // Convert html names to html numbers
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsD; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(this.htmlNamesD[i]);
                if(pos0!=-1){
                    pos1 = pos0 + this.htmlNamesD[i].length();
                    String hold0 = ss.substring(0,pos0) + this.htmlNumbersD[i];
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                    break;
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Convert int equivalents to html numbers 
        for(int i=0; i<nSS; i++){
            int cc = (int)ss.charAt(i);
            for(int j=0; j<this.nSymbolsD; j++){
                if(cc==this.decNumbersD[j]){
                    String hold1 = ss.substring(0,i) + this.htmlNumbersD[j];
                    if(i+1<nSS)hold1 += ss.substring(i+1);
                    ss = hold1;
                    nSS = ss.length();
                    break;
                }
            }
        }
        return ss;
    }
    
    // Convert all dashes to html number representation 
    // static method
    public static String dashesToHtmlNumberCodes(String ss){
        Strings strs = new Strings(ss);
        return strs.dashesToHtmlNumberCodes();
    }
    
    //Convert all dashes to a specific html number code representation 
    // instance method
    public String dashesToSpecificHtmlNumberCode(String code){

	// check code is html code
	boolean test1 = this.checkIfHtmlNumberCode(code);
	if(!test1)throw new IllegalArgumentException("Argument," + code + ", does not appear to be a html number code");
        
        String ss = this.enteredString;
        int nSS = ss.length();
        
        // Convert html names to specific html number code
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsD; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(this.htmlNamesD[i]);
                if(pos0!=-1){
                    pos1 = pos0 + this.htmlNamesD[i].length();
                    String hold0 = ss.substring(0,pos0) + code;
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                    break;
                }
                else{
                    test0 = false;
                }
            }
        }
     
        // Convert html number codes to specific html number code
        test0 = true;
        pos0 = -1;
        pos1 = -1;
        for(int i=0; i<this.nSymbolsD; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(this.htmlNumbersD[i]);
                if(pos0!=-1){
                    pos1 = pos0 + this.htmlNumbersD[i].length();
                    String hold0 = ss.substring(0,pos0) + code;
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                    break;
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Convert int equivalents to html numbers 
        for(int i=0; i<nSS; i++){
            int cc = (int)ss.charAt(i);
            for(int j=0; j<this.nSymbolsD; j++){
                if(cc==this.decNumbersD[j]){
                    String hold1 = ss.substring(0,i) + code;
                    if(i+1<nSS)hold1 += ss.substring(i+1);
                    ss = hold1;
                    nSS = ss.length();
                    break;
                }
            }
        }
        return ss;
    }
    
    // Convert all dashes to a specific html number code representation 
    // static method
    public static String dashesToSpecificHtmlNumberCode(String code, String ss){
        Strings strs = new Strings(ss);
        return strs.dashesToSpecificHtmlNumberCode(code);
    }

    
    // Convert all accented characters to html number representation 
    // instance method
    public String accentedToHtmlCodes(){
        return this.accentsToHtmlNumbers();
    }
    
    // originally titled method (used in TC updatee
    public String accentsToHtmlNumbers(){
        
        String ss = this.enteredString;
        int nSS = ss.length();
        
        // Convert html names to html numbers
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsA; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(this.htmlNamesA[i]);
                if(pos0!=-1){
                    pos1 = pos0 + this.htmlNamesA[i].length();
                    String hold0 = ss.substring(0,pos0) + this.htmlNumbersA[i];
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                    break;
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Convert int equivalents to html numbers 
        for(int i=0; i<nSS; i++){
            int cc = (int)ss.charAt(i);
            for(int j=0; j<this.nSymbolsA; j++){
                if(cc==this.decNumbersA[j]){
                    String hold1 = ss.substring(0,i) + this.htmlNumbersA[j];
                    if(i+1<nSS)hold1 += ss.substring(i+1);
                    ss = hold1;
                    nSS = ss.length();
                    break;
                }
            }
        }
        
        return ss;
    }
    
    // Convert all accented characters to html number representation 
    // static method
    public static String accentsToHtmlNumbers(String ss){
        Strings strs = new Strings(ss);
        return strs.accentsToHtmlNumbers();
    }
    
    
    
    //Convert all single quoatation marks to html number code representation 
    // instance method
    public String singleQuotationMarksToHtmlNumberCodes(){
        
        String ss = this.enteredString;
        int nSS = ss.length();
        
        // Convert html names to html numbers
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsSQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(this.htmlNamesSQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + this.htmlNamesSQ[i].length();
                    String hold0 = ss.substring(0,pos0) + this.htmlNumbersSQ[i];
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                    break;
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Convert int equivalents to html numbers 
        for(int i=0; i<nSS; i++){
            int cc = (int)ss.charAt(i);
            for(int j=0; j<this.nSymbolsSQ; j++){
                if(cc==this.decNumbersSQ[j]){
                    String hold1 = ss.substring(0,i) + this.htmlNumbersSQ[j];
                    if(i+1<nSS)hold1 += ss.substring(i+1);
                    ss = hold1;
                    nSS = ss.length();
                    break;
                }
            }
        }
        return ss;
    }
    
    // Convert all single quotation marks to html number representation 
    // static method
    public static String singleQuotationMarksToHtmlNumberCodes(String ss){
        Strings strs = new Strings(ss);
        return strs.singleQuotationMarksToHtmlNumberCodes();
    }
    
    //Convert all single quotation marks to a specific html number code representation 
    // instance method
    public String singleQuotationMarksToSpecificHtmlNumberCode(String code){

	// check code is html code
	boolean test1 = this.checkIfHtmlNumberCode(code);
	if(!test1)throw new IllegalArgumentException("Argument," + code + ", does not appear to be a html number code");
        
        String ss = this.enteredString;
        int nSS = ss.length();
        
        // Convert html names to specific html number code
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsSQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(this.htmlNamesSQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + this.htmlNamesSQ[i].length();
                    String hold0 = ss.substring(0,pos0) + code;
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                    break;
                }
                else{
                    test0 = false;
                }
            }
        }
     
        // Convert html number codes to specific html number code
        test0 = true;
        pos0 = -1;
        pos1 = -1;
        for(int i=0; i<this.nSymbolsSQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(this.htmlNumbersSQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + this.htmlNumbersSQ[i].length();
                    String hold0 = ss.substring(0,pos0) + code;
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                    break;
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Convert int equivalents to html numbers 
        for(int i=0; i<nSS; i++){
            int cc = (int)ss.charAt(i);
            for(int j=0; j<this.nSymbolsSQ; j++){
                if(cc==this.decNumbersSQ[j]){
                    String hold1 = ss.substring(0,i) + code;
                    if(i+1<nSS)hold1 += ss.substring(i+1);
                    ss = hold1;
                    nSS = ss.length();
                    break;
                }
            }
        }
        return ss;
    }
    
    // Convert all dashes to a specific html number code representation 
    // static method
    public static String singleQuotationMarksToSpecificHtmlNumberCode(String code, String ss){
        Strings strs = new Strings(ss);
        return strs.singleQuotationMarksToSpecificHtmlNumberCode(code);
    }

    
    
     // Return array of single quotation mark numbers
    // instance method
    public String[] getSingleQuotationMarkHtmlNumbers(){
        return this.htmlNumbersSQ;
    }
    
      //Convert all double quoatation marks to html number code representation 
    // instance method
    public String doubleQuotationMarksToHtmlNumberCodes(){
        
        String ss = this.enteredString;
        int nSS = ss.length();
        
        // Convert html names to html numbers
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsDQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(this.htmlNamesDQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + this.htmlNamesDQ[i].length();
                    String hold0 = ss.substring(0,pos0) + this.htmlNumbersDQ[i];
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                    break;
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Convert int equivalents to html numbers 
        for(int i=0; i<nSS; i++){
            int cc = (int)ss.charAt(i);
            for(int j=0; j<this.nSymbolsDQ; j++){
                if(cc==this.decNumbersDQ[j]){
                    String hold1 = ss.substring(0,i) + this.htmlNumbersDQ[j];
                    if(i+1<nSS)hold1 += ss.substring(i+1);
                    ss = hold1;
                    nSS = ss.length();
                    break;
                }
            }
        }
        return ss;
    }
    
    // Convert all double quotation marks to html number representation 
    // static method
    public static String doubleQuotationMarksToHtmlNumberCodes(String ss){
        Strings strs = new Strings(ss);
        return strs.doubleQuotationMarksToHtmlNumberCodes();
    }
    
    //Convert all double quotation marks to a specific html number code representation 
    // instance method
    public String doubleQuotationMarksToSpecificHtmlNumberCode(String code){

	// check code is html code
	boolean test1 = this.checkIfHtmlNumberCode(code);
	if(!test1)throw new IllegalArgumentException("Argument," + code + ", does not appear to be a html number code");
        
        String ss = this.enteredString;
        int nSS = ss.length();
        
        // Convert html names to specific html number code
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsDQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(this.htmlNamesDQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + this.htmlNamesDQ[i].length();
                    String hold0 = ss.substring(0,pos0) + code;
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                    break;
                }
                else{
                    test0 = false;
                }
            }
        }
     
        // Convert html number codes to specific html number code
        test0 = true;
        pos0 = -1;
        pos1 = -1;
        for(int i=0; i<this.nSymbolsDQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(this.htmlNumbersDQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + this.htmlNumbersDQ[i].length();
                    String hold0 = ss.substring(0,pos0) + code;
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                    break;
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Convert int equivalents to html numbers 
        for(int i=0; i<nSS; i++){
            int cc = (int)ss.charAt(i);
            for(int j=0; j<this.nSymbolsDQ; j++){
                if(cc==this.decNumbersDQ[j]){
                    String hold1 = ss.substring(0,i) + code;
                    if(i+1<nSS)hold1 += ss.substring(i+1);
                    ss = hold1;
                    nSS = ss.length();
                    break;
                }
            }
        }
        return ss;
    }
    
    // Convert all double quotation marks to a specific html number code representation 
    // static method
    public static String doubleQuotationMarksToSpecificHtmlNumberCode(String code, String ss){
        Strings strs = new Strings(ss);
        return strs.doubleQuotationMarksToSpecificHtmlNumberCode(code);
    }
   
    // Convert all quotation mark to html number representation 
    // instance method
    public String allQuotationMarksToHtmlNumbers(){
        return this.quotationMarksToHtmlNumbers();
    }

    public String quotationMarksToHtmlNumbers(){
        
        String ss = this.enteredString;
        int nSS = ss.length();
        
        // Convert html names to html numbers
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(this.htmlNamesQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + this.htmlNamesQ[i].length();
                    String hold0 = ss.substring(0,pos0) + this.htmlNumbersQ[i];
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                    break;
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Convert int equivalents to html numbers 
        for(int i=0; i<nSS; i++){
            int cc = (int)ss.charAt(i);
            for(int j=0; j<this.nSymbolsQ; j++){
                if(cc==this.decNumbersQ[j]){
                    String hold1 = ss.substring(0,i) + this.htmlNumbersQ[j];
                    if(i+1<this.nEntered)hold1 += ss.substring(i+1);
                    ss = hold1;
                    nSS = ss.length();
                    break;
                }
            }
        }
        
        return ss;
    }
    
    // Convert all quotation marks to html number representation except "
    // instance method
    // REplaces " html code by "\22
    // For update program
    public String quotationMarksToHtmlNumbersD(){
        String ss = quotationMarksToHtmlNumbers();
        
        boolean test = true;
        String testString = "&#34;";
        while(test){
            int pos = ss.indexOf(testString);
            if(pos==-1){
                test = false;
            }
            else{
                ss = ss.substring(0,pos) + "\"" + ss.substring(pos+5);
            }
        }
        return ss;
    }
    
    // Convert all quotation mark to html number representation 
    // static method
    public static String allQuotationMarksToHtmlNumbers(String ss){
        return Strings.quotationMarksToHtmlNumbers(ss);
    }
   
            
    public static String quotationMarksToHtmlNumbers(String ss){
        Strings strs = new Strings(ss);
        return strs.quotationMarksToHtmlNumbers();
    }
    
    // Convert all quotation mark to html number representation except "
    // static method
    public static String quotationMarksToHtmlNumbersD(String ss){
        Strings strs = new Strings(ss);
        return strs.quotationMarksToHtmlNumbersD();
    }
    
    //Convert all double q marks to a specific html number code representation 
    // instance method
    public String allQuotationMarksToSpecificHtmlNumberCode(String code){

	// check code is html code
	boolean test1 = this.checkIfHtmlNumberCode(code);
	if(!test1)throw new IllegalArgumentException("Argument," + code + ", does not appear to be a html number code");
        
        String ss = this.enteredString;
        int nSS = ss.length();
        
        // Convert html names to specific html number code
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(this.htmlNamesQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + this.htmlNamesQ[i].length();
                    String hold0 = ss.substring(0,pos0) + code;
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                    break;
                }
                else{
                    test0 = false;
                }
            }
        }
     
        // Convert html number codes to specific html number code
        test0 = true;
        pos0 = -1;
        pos1 = -1;
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(this.htmlNumbersQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + this.htmlNumbersQ[i].length();
                    String hold0 = ss.substring(0,pos0) + code;
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                    break;
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Convert int equivalents to html numbers 
        for(int i=0; i<nSS; i++){
            int cc = (int)ss.charAt(i);
            for(int j=0; j<this.nSymbolsQ; j++){
                if(cc==this.decNumbersQ[j]){
                    String hold1 = ss.substring(0,i) + code;
                    if(i+1<nSS)hold1 += ss.substring(i+1);
                    ss = hold1;
                    nSS = ss.length();
                    break;
                }
            }
        }
        return ss;
    }
    
    // Convert all quotation marks to a specific html number code representation 
    // static method
    public static String allQuotationMarksToSpecificHtmlNumberCode(String code, String ss){
        Strings strs = new Strings(ss);
        return strs.allQuotationMarksToSpecificHtmlNumberCode(code);
    }
   
    
    
      // Convert all accent html name and html number representations to decimal integer representation 
    // instance method
    public String accentsToDec(){
        
        String ss = this.enteredString;
        int nSS = ss.length();
        
        // Convert html names 
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsA; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNamesA[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNamesA[i].length();
                    String hold0 = ss.substring(0,pos0) + (char)this.decNumbersA[i];
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Convert html numbers 
        for(int i=0; i<this.nSymbolsA; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNumbersA[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNumbersA[i].length();
                    String hold0 = ss.substring(0,pos0) + (char)this.decNumbersA[i];
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        return ss;
    }
    
    // Convert all quotation mark html name and html number representations to decimal integer representation 
    // static method
    public static String accentsToDec(String ss){
        Strings strs = new Strings(ss);
        return strs.accentsToDec();
    }
    
    // Convert all dash html name and html number representations to decimal integer representation 
    // instance method
    public String dashesToDec(){
        
        String ss = this.enteredString;
        int nSS = ss.length();
        
        // Convert html names 
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsD; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNamesD[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNamesD[i].length();
                    String hold0 = ss.substring(0,pos0) + (char)this.decNumbersD[i];
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Convert html numbers 
        for(int i=0; i<this.nSymbolsD; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNumbersD[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNumbersD[i].length();
                    String hold0 = ss.substring(0,pos0) + (char)this.decNumbersD[i];
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        return ss;
    }
    
    // Convert all dash html name and html number representations to decimal integer representation 
    // static method
    public static String dashesToDec(String ss){
        Strings strs = new Strings(ss);
        return strs.dashesToDec();
    }
    
    // Convert all quotation mark html name and html number representations to decimal integer representation 
    // instance method
    public String quotationMarksToDec(){
        
        String ss = this.enteredString;
        int nSS = ss.length();
        
        // Convert html names 
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNamesQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNamesQ[i].length();
                    String hold0 = ss.substring(0,pos0) + (char)this.decNumbersQ[i];
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Convert html numbers 
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNumbersQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNumbersQ[i].length();
                    String hold0 = ss.substring(0,pos0) + (char)this.decNumbersQ[i];
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        return ss;
    }
    
    // Convert all quotation mark html name and html number representations to decimal integer representation 
    // static method
    public static String quotationMarksToDec(String ss){
        Strings strs = new Strings(ss);
        return strs.quotationMarksToDec();
    }
    
    // Convert all quotation mark integer dec representations to html name representation 
    // instance method
    public String quotationMarksDecToName(){
              
        // Convert to html names 
        String ss = this.enteredString;
        int nSS = ss.length();
        
        for(int i=0; i<nSS;i++){
            int cc = (int)ss.charAt(i);
            for(int j=0; j<this.nSymbolsQ; j++){
                if(cc==this.decNumbersQ[j]){
                    String hold0 = ss.substring(0,i) + this.htmlNamesQ[j];
                    if(i<nSS-1)hold0 += ss.substring(i+1);
                    ss= hold0;
                    nSS = ss.length();
                    break;
                }
            }   
        }
        return ss;
    }
    
    // Convert all quotation mark integer dec representations to html name representation 
    // static method
    public static String quotationMarksDecToName(String ss){
        Strings strs = new Strings(ss);
        return strs.quotationMarksDecToName();
    }
    
    // Convert all quotation mark integer dec representations to html number representation 
    // instance method
    public String quotationMarksDecToHtmlNumber(){
              
        // Convert to html names 
               String ss = this.enteredString;
        int nSS = ss.length();
        
        for(int i=0; i<nSS;i++){
            int cc = (int)ss.charAt(i);
            for(int j=0; j<this.nSymbolsQ; j++){
                if(cc==this.decNumbersQ[j]){
                    String hold0 = ss.substring(0,i) + this.htmlNumbersQ[j];
                    if(i<nSS-1)hold0 += ss.substring(i+1);
                    ss = hold0;
                    nSS = ss.length();
                    break;
                }
            }   
        }
        return ss;
    }
    
    // Convert all quotation mark integer dec representations to html name representation 
    // static method
    public static String quotationMarksDecToHtmlNumber(String ss){
        Strings strs = new Strings(ss);
        return strs.quotationMarksDecToHtmlNumber();
    }
    
    // Replace dashes with spaces
    // instance method
    public String replaceDashesWithSpaces(){
        
        String ss = this.enteredString;
        int nSS = ss.length();
        
        // Replace html names 
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsD; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNamesD[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNamesD[i].length();
                    String hold0 = ss.substring(0,pos0) + " ";
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Replace html numbers 
        for(int i=0; i<this.nSymbolsD; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNumbersD[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNumbersD[i].length();
                    String hold0 = ss.substring(0,pos0) + " ";
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Replace int equivalents
        for(int i=0; i<this.nSymbolsD; i++){
            for(int j=0; j<nSS; j++){
               if((int)ss.charAt(j)==this.decNumbersD[i]){
                   ss = ss.substring(0,j) + " " + ss.substring(j+1);
                   nSS = ss.length();
               }
            }
        }

        return ss;
    }
    
    // Replace dashes with spaces 
    // static method
    public static String replaceDashesWithSpaces(String ss){
        Strings strs = new Strings(ss);
        return strs.replaceDashesWithSpaces();
    }

    
    // Removes all quotation marks
    // instance method
    public String removeQuotationMarks(){
        
        String ss = this.enteredString;
        int nSS = ss.length();
        
        // Remove html names 
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNamesQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNamesQ[i].length();
                    String hold0 = ss.substring(0,pos0);
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Remove html numbers 
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNumbersQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNumbersQ[i].length();
                    String hold0 = ss.substring(0,pos0);
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Replace int equivalents
        for(int i=0; i<this.nSymbolsQ; i++){
            for(int j=0; j<nSS; j++){
               if((int)ss.charAt(j)==this.decNumbersQ[i]){
                   ss = ss.substring(0,j) + ss.substring(j+1);
                   nSS = ss.length();
               }
            }
        }
        
        return ss;
    }
    
    // Removes all quotation marks 
    // static method
    public static String removeQuotationMarks(String ss){
        Strings strs = new Strings(ss);
        return strs.removeQuotationMarks();
    }
    
    // Replace all quotation marks by spaces
    // instance method
    public String replaceQuotationMarksBySpaces(){
        
        String ss = this.enteredString;
        int nSS = ss.length();
        
        // Replace html names 
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNamesQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNamesQ[i].length();
                    String hold0 = ss.substring(0,pos0);
                    if(pos1<nSS)hold0 += (" " + ss.substring(pos1));
                    ss = hold0;
                    nSS = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Remove html numbers 
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNumbersQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNumbersQ[i].length();
                    String hold0 = ss.substring(0,pos0);
                    if(pos1<nSS)hold0 += (" " + ss.substring(pos1));
                    ss = hold0;
                    nSS = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Replace int equivalents
        for(int i=0; i<this.nSymbolsQ; i++){
            for(int j=0; j<nSS; j++){
               if((int)ss.charAt(j)==this.decNumbersQ[i]){
                   ss = ss.substring(0,j) + " " + ss.substring(j+1);
                   nSS = ss.length();
               }
            }
        }
        
        return ss;
    }
    
    // Replace all quotation marks by spaces
    // instance method
    public String replaceQuotationMarksWithSpaces(){
        
        String ss = this.enteredString;
        int nSS = ss.length();
        
        // Remove html names 
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNamesQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNamesQ[i].length();
                    String hold0 = ss.substring(0,pos0) + " ";
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Remove html numbers 
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNumbersQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNumbersQ[i].length();
                    String hold0 = ss.substring(0,pos0) + " ";
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Replace int equivalents
        for(int i=0; i<this.nSymbolsQ; i++){
            for(int j=0; j<nSS; j++){
               if((int)ss.charAt(j)==this.decNumbersQ[i]){
                   ss = ss.substring(0,j) + " " + ss.substring(j+1);
                   nSS = ss.length();
               }
            }
        }
        
        return ss;
    }
    
    // Replace all quotation marks by spaces
    // static method
    public static String replaceQuotationMarksWithSpaces(String ss){
        Strings strs = new Strings(ss);
        return strs.replaceQuotationMarksWithSpaces();
    }
    
    // Removes all quotation marks and 's
    // instance method
    public String removeQuotationMarksPlus(){
        
        String ss = this.enteredString;
        int nSS = ss.length();
        
        // Remove html names 
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNamesQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNamesQ[i].length();
                    if(pos1<nSS && pos0>0){
                        char ccc = ss.charAt(pos0-1); 
                        if(!Character.isWhitespace(ccc)){
                            if(ss.charAt(pos1)=='s' || ss.charAt(pos1)=='S')pos1++;
                        }
                    }
                    String hold0 = ss.substring(0,pos0);
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Remove html numbers 
        for(int i=0; i<this.nSymbolsQ; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNumbersQ[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNumbersQ[i].length();
                    if(pos1<nSS && pos0>0){
                        char ccc = ss.charAt(pos0-1); 
                        if(!Character.isWhitespace(ccc)){
                            if(ss.charAt(pos1)=='s' || ss.charAt(pos1)=='S')pos1++;
                        }
                    }
                    String hold0 = ss.substring(0,pos0);
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Replace int equivalents
        for(int i=0; i<this.nSymbolsQ; i++){
            for(int j=0; j<nSS; j++){
                pos1 = j+1;
                if(pos1<nSS && j>0){
                    char ccc = ss.charAt(j-1); 
                    if(!Character.isWhitespace(ccc)){
                        if(ss.charAt(pos1)=='s' || ss.charAt(pos1)=='S')pos1++;
                    }
                }
                if((int)ss.charAt(j)==this.decNumbersQ[i]){
                    ss = ss.substring(0,j) + ss.substring(pos1);
                    nSS = ss.length();
                }
            }
        }

        return ss;
    }
    
    // Removes all quotation marks plus's
    // static method
    public static String removeQuotationMarksPlus(String ss){
        Strings strs = new Strings(ss);
        return strs.removeQuotationMarksPlus();
    }
    
        // Removes all quotation marks
    // instance method
    public String removeDashes(){
       
        String ss = this.enteredString;
        int nSS = ss.length();
        
        // Remove html names 
        boolean test0 = true;
        int pos0 = -1;
        int pos1 = -1;
        for(int i=0; i<this.nSymbolsD; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNamesD[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNamesD[i].length();
                    String hold0 = ss.substring(0,pos0);
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Remove html numbers 
        for(int i=0; i<this.nSymbolsD; i++){
            test0 = true;  
            while(test0){
                pos0 = ss.indexOf(htmlNumbersD[i]);
                if(pos0!=-1){
                    pos1 = pos0 + htmlNumbersD[i].length();
                    String hold0 = ss.substring(0,pos0);
                    if(pos1<nSS)hold0 += ss.substring(pos1);
                    ss = hold0;
                    nSS = ss.length();
                }
                else{
                    test0 = false;
                }
            }
        }
        
        // Replace int equivalents
        for(int i=0; i<this.nSymbolsD; i++){
            for(int j=0; j<nSS; j++){
               if((int)ss.charAt(j)==this.decNumbersD[i]){
                   ss = ss.substring(0,j) + ss.substring(j+1);
                   nSS = ss.length();
               }
            }
        }

        return ss;
    }
    
    // Removes all dashes 
    // static method
    public static String removeDashes(String ss){
        Strings strs = new Strings(ss);
        return strs.removeDashes();
    }
    
    // Display integer character equivalents
    // instance method
    public int[] displayIntegers(){
        int[] ints = new int[this.nEntered];
        for(int i=0; i<this.nEntered; i++)ints[i] = (int)this.enteredString.charAt(i);
        PrintToScreen.print(ints);
        
        return ints; 
    }
    
    // Display integer character equivalents
    // static method
    public static int[] displayIntegers(String ss){
        Strings strs = new Strings(ss);
        return strs.displayIntegers();
    }
    
    // Return index of the first occurence of a dash
    // instance method
    public int indexOfDash(){
        int ii = -1;
        int iis = -1;
        
        boolean found = false;
        for(int i=0; i<this.nEntered; i++){
            String cc = this.enteredString.substring(i,i+1);
            for(int j=0; j<this.nSymbolsD; j++){
                if(cc.equals(symbolsD[j])){
                    ii = i;
                    iis = i;
                    found = true;
                    break;
                }
            }
            if(found)break;
        }
        
        String[] namesD = {"&ndash;", "&mdash;", "&oline;"};
        int pos0 = -1;
        int iiss = -1;
        for(int j=0; j<3; j++){
           pos0 = this.enteredString.indexOf(namesD[j]);
           if(pos0!=-1){
               ii = pos0;
               if(iiss==-1){
                   iiss = pos0; 
               }
               else{
                   if(iiss<pos0){
                       ii = iiss;
                   }
                   else{
                       iiss = pos0;
                   }
               }
            }
        }  
        if(iis!=-1){
            if(iiss!=-1){
                ii = Math.min(iis, iiss);
                iis = ii;
            }
            else{
                // no action
            }
        }
        else{
            if(iiss!=-1){
                ii = iiss;
                iis = iiss;
            }
            else{
                // no action 
            }           
        }
        
        pos0 = -1;
        iiss = -1;
        for(int j=0; j<this.nSymbolsD; j++){
           pos0 = this.enteredString.indexOf(htmlNumbersD[j]);
           if(pos0!=-1){
               ii = pos0;
               if(iiss==-1){
                   iiss = pos0; 
               }
               else{
                   if(iiss<pos0){
                       ii = iiss;
                   }
                   else{
                       iiss = pos0;
                   }
               }
            }
        }  
        if(iis!=-1){
            if(iiss!=-1){
                ii = Math.min(iis, iiss);
                iis = ii;
            }
            else{
                // no action
            }
        }
        else{
            if(iiss!=-1){
                ii = iiss;
                iis = iiss;
            }
            else{
                // no action 
            }           
        }

        return ii;
    }
    
    // Return index of the first occurence of a dash
    // static method
    public static int indexOfDash(String ss){
        Strings strs = new Strings(ss);
        return strs.indexOfDash();
    }
    
    // Returns true if a percentage of the shorter string is common to both strings
    // The length of the putative common string is the length of the shorter string times identicalFraction rounded to the nearest integer 
     // instance method
    public boolean nearlyEquals(String ss1, double identicalPerCentage){
        return this.nearlyEqualsF(ss1, identicalPerCentage/100.0);
    }
   
    // Returns true if a percentage of the shorter string is common to both strings
    // The length of the putative common string is the length of the shorter string times identicalFraction rounded to the nearest integer 
    // static  method
    public static boolean nearlyEquals(String ss0, String ss1, double identicalPercentage){
        Strings ss = new Strings(ss0);
        return ss.nearlyEquals(ss1, identicalPercentage);
    }    
    
    // Returns true if 90% of the shorter string is common to both strings
    // The length of the putative common string is the length of the shorter string times identicalFraction rounded to the nearest integer 
     // instance method
    public boolean nearlyEquals(String ss1){
        return this.nearlyEqualsF(ss1, 0.9);
    }
   
    // Returns true if 90% of the shorter string is common to both strings
    // The length of the putative common string is the length of the shorter string times identicalFraction rounded to the nearest integer 
    // static  method
    public static boolean nearlyEquals(String ss0, String ss){
        Strings sss = new Strings(ss0);
        return sss.nearlyEquals(ss);
    }   
    
    
    // Returns true if a fraction (arguent identicalFraction) of the shorter string is common to both strings
    // instance method
    public boolean nearlyEqualsF(String ss1, double identicalFraction){
       
        boolean ret = false;
        String[] ss = new String[2];
        ss[0] = this.enteredString;
        ss[1] = ss1;
        if(ss[0].equalsIgnoreCase(ss[1])){
            ret = true;
        }
        else{
            int n0 = ss[0].length();
            int n1 = ss[1].length();
            int nMin = Math.min(n0,n1);
            int nMax = Math.max(n0,n1);
            int minIndex = 0;
            int maxIndex = 1;
            if(n0==nMax){
                minIndex = 1;
                maxIndex = 0;
            }
            int nLimit = (int)Math.round(identicalFraction*nMin);
            int nExcess = nMax - nMin;
            for(int i=0; i<nExcess+1; i++){
                int nSame = 0;
                for(int j=0; j<nMin; j++){
                    if(ss[minIndex].charAt(j)==ss[maxIndex].charAt(j+i))nSame++;
                }
                if(nSame>=nLimit){
                    ret = true;
                    break;
                }
            }
        }
        return ret;
    }
    
    // Returns true if a continuous fraction of the shorter string is common to both strings
    // The length of the putative common string is the length of the shorter string times identicalFraction rounded to the nearest integer 
    // static  method
    public static boolean nearlyEqualsF(String ss0, String ss1, double identicalFraction){
        Strings ss = new Strings(ss0);
        return ss.nearlyEqualsF(ss1, identicalFraction);
    }
    
    // PREPARATION FOR JAVA DISPLAY
    
    // Converts all html number codes to corresponding ASCII decimal char
    // instance method
    public String allHtmlNumberCodesToAsciiChars(){
        String string = this.enteredString;
        int nLen = string.length();
        int[] codeStarts = this.indicesOf("&#");
        if(codeStarts==null){
            // return with no further actiob
        }
        else{
            int nC = codeStarts.length;
            for(int i=nC-1; i>=0; i--){
                int clast = codeStarts[i] + 6;
                if(clast>nLen-1)clast = nLen-1;
                boolean test0 = false;
                for(int j = codeStarts[i]; j<clast; j++){
                    if(string.charAt(j)==';'){
                        String middle = string.substring(codeStarts[i]+2,j);
                        if(Strings.areDigits(middle)){
                            int decs = Integer.parseInt(middle);
                            char newc = (char)decs;
                            System.out.println("On entering the String &#" + middle + "; was replaced by " + newc);
                            string = string.substring(0,codeStarts[i]) + newc + string.substring(j+1);    
                            nLen = string.length();
                            test0 = true;
                        }
                    }
                    if(test0)break;
                }
            }
        }
        return string;
    }
    
    //Static method
    public static String allHtmlNumberCodesToAsciiChars(String string0){
        Strings sss = new Strings(string0);
        return sss.allHtmlNumberCodesToAsciiChars();      
    }
    
        // Converts all html number codes to corresponding ASCII decimal char
    // instance method
    public String allHtmlNameCodesToAsciiChars(){
        String string = this.enteredString;
        int nLen = string.length();
        int[] codeStarts = this.indicesOf("&");
        if(codeStarts==null){
            // return with no further actiob
        }
        else{
            int nC = codeStarts.length;
            for(int i=nC-1; i>=0; i--){
                int clast = nLen-1;
                boolean test0 = false;
                for(int j = codeStarts[i]; j<clast; j++){
                    if(string.charAt(j)==';'){
                        String middle = string.substring(codeStarts[i]+1,j);
                        if(Strings.areAlphabetic(middle)){
                            String ncode = "&" + middle + ";";
                            for(int k=0; k<this.nAllNameCodes; k++){
                                if(ncode.equals(this.allNameCodesName[k])){
                                    int cdec = this.allNameCodesDec[k];
                                    char cchar = (char)cdec;
                                    System.out.println("On entering the String &" + middle + "; was replaced by " + cchar);
                                    string = string.substring(0,codeStarts[i]) + cchar + string.substring(j+1);    
                                    nLen = string.length();
                                    test0 = true;
                                }
                                if(test0)break;
                            }
                        }
                    }
                    if(test0)break;
                }
            }
        }
        return string;
    }
    
    //Static method
    public static String allHtmlNameCodesToAsciiChars(String string0){
        Strings sss = new Strings(string0);
        return sss.allHtmlNameCodesToAsciiChars();      
    }
    
    // REPLACEMENTS
    
    // Rplace substring1 by substring2
    // instance method
    public String replaceSubstring(String substring1, String substring2){
        String news = this.enteredString;
        int[] ss1indices = this.indicesOf(substring1);
        if(ss1indices==null){
            throw new IllegalArgumentException(substring1 + " not found");
        }
        else{
            int n1 = substring1.length();
            int nI = ss1indices.length;
            for(int i=nI-1; i>=0; i--){
                news = news.substring(ss1indices[i]) + substring2 + news.substring(ss1indices[i]+n1);
            }
        }
        return news;
    }
    
     // static method
    public static String replaceSubstring(String substring1, String substring2, String ss){
        Strings sss = new Strings(ss);
        return sss.replaceSubstring(substring1, substring2);
    }
    
    // Rplace character by character
    // instance method
    public String replaceChar(char char1, char char2){
        String news = this.enteredString;
        // check for equivalent html codes
        boolean test0 = false;
        String codenumb =null;
        String codename =null;
        for(int i= 0; i<this.nAllCodes; i++){
            if((int)char1==this.allAsciiDecs[i]){
                codenumb = this.allNumberCodes[i];
                codename = this.allNameCodes[i];
                test0 = true;
                break;
            }
        }
        if(test0){
            boolean test1 = news.contains(codenumb);
            if(test1)news = this.replaceSubstring(codenumb,String.valueOf(char1));
            boolean test2 = news.contains(codename);
            if(test2)news = this.replaceSubstring(codename,String.valueOf(char1));
        }
        
        int[] cc1indices = this.indicesOf(char1);
         if(cc1indices==null){
            throw new IllegalArgumentException(char1 + " not found");
        }
        else{
            int nI = cc1indices.length;
            for(int i=nI-1; i>=0; i--){
                news = news.substring(cc1indices[i]) + char2 + news.substring(cc1indices[i]+1);
            }
        }
        return news;
    }
    
     // static method
    public static String replaceChar(char char1, char char2, String ss){
        Strings sss = new Strings(ss);
        return sss.replaceChar(char1, char2);
    }
    
    // Replace quotation marks by spaces except for possesive 's
    
   
    // Check for apostrophe
    // return indices of any aostophes or null if none
    public int[] checkForApostrophe(String sssss){
        
        int nSsss = sssss.length();
        String ssss = sssss.toLowerCase();
        int[] apos1 = null;
        for(int i=0; i<nSymbolsAp; i++){
             apos1 = Strings.indicesOf(symbolsAp[i], ssss);  
        }
        
        // Check for duplicate html coding
        int nAp = '\u0000';
        boolean[] boolAp = null;
        boolean testDup = false;
        if(apos1!=null){
            nAp = apos1.length;
            boolAp = new boolean[nAp];;
            for(int i=0; i<nAp; i++)boolAp[i] = false;
            for(int i=0; i<nAp; i++){
                for(int j=i+1; j<nAp; j++){
                    if(apos1[i]==apos1[j]){
                        boolAp[j] = true;
                        testDup = true;
                    }
                }
            }
            if(testDup){
                ArrayList<Integer> aldup0 = new ArrayList<Integer>();
                for(int i=0; i<nAp; i++){
                    aldup0.add(apos1[i]);
                }
                for(int i=nAp-1; i>=0; i--){
                    if(!boolAp[i])aldup0.remove(i);
                }
                nAp = aldup0.size();
                apos1 = new int[nAp];
                for(int i=0; i<nAp; i++){
                    apos1[i] = aldup0.get(i);
                }            
            }
        }     
        
        boolean testAp0 = false;
        ArrayList<Integer> alap0 = new ArrayList<Integer>();
        if(apos1!=null){
            nAp = apos1.length;
            for(int i=0; i<nAp; i++){
                int iAp = apos1[i];
                if(iAp>0 && iAp<nSsss-1){
                    char charf = ssss.charAt(iAp+1);
                    if(charf=='s'){
                        char charb = ssss.charAt(iAp-1);
                        boolean bAp0 = Chars.isAlphanumeric(charb);
                        if(bAp0){
                            alap0.add(iAp);
                            testAp0 = true;
                        }
                    }
                }
            }
        }
        
        int[] apos2 = null;
        if(testAp0){
            int nAlap0 = alap0.size();
            apos2 = new int[nAlap0];
            for(int i=0; i<nAlap0; i++){
                apos2[i] = alap0.get(i);
            }
        }
        return apos2;
    }
    /*
    public int[] checkForApostrophe(){
        String string = this.enteredString.toLowerCase();
        ArrayList<Integer> alaps = new ArrayList<Integer>();
        ArrayList<Integer> alaph = new ArrayList<Integer>();
        int[] apos1 = null;
        for(int i=0; i<nSymbolsAp; i++){
            apos1 = Strings.indicesOf(this.symbolsAp[i], string);
            if(apos1!=null){
                for(int j=0; j<apos1.length; j++){
                    alaph.add(apos1[j]);
                }
            }
        }
   
        int nA = alaph.size();
        for(int i=0; i<nA; i++){
            int ia = alaph.get(i);
            int ja = ia+1;
            int ka = ia-1;
            char choldj ='\u0000';
            char choldk ='\u0000';
            if(ja<nA){
                choldj = string.charAt(ja);
                if(choldj=='s'){
                    if(ka>=0){
                        choldk = string.charAt(ka);
                        if(Chars.isAlphanumeric(choldk)){
                            alaps.add(ia);
                        }
                    }
                }
            }
        }
        int nAl = alaps.size();
        System.out.println("nAl " + nAl);
        int[] apos2 = null;
        if(nAl>0){
            apos2 = new int[nAl];
            for(int i=0; i<nAl; i++){
                apos2[i] = alaps.get(i);
            }
        }
       
        return apos2;
    }
    */
}

        